/**
 * Source file created in 2011 by Southwest Research Institute
 */


package gov.va.med.pharmacy.peps.domain.common.capability.impl;


import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidationType.LOG_FORGING;
import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidator.validateStringInput;
import static gov.va.med.pharmacy.peps.common.utility.IOUtility.safeClose;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

import org.apache.logging.log4j.Logger; 
import org.apache.logging.log4j.LogManager;


import gov.va.med.pharmacy.peps.common.utility.ConfigFileUtility;

import gov.va.med.pharmacy.peps.common.vo.EntityType;

import gov.va.med.pharmacy.peps.domain.common.capability.DatabaseDomainCapability;


/**
 * This class is used to retrieve data from the database originally for the migration
 * effort.
 * 
 */
public class DatabaseDomainCapabilityImpl implements DatabaseDomainCapability {

    private static final Logger LOG = LogManager.getLogger(DatabaseDomainCapabilityImpl.class);
    private static final String MSG_CONN_NULL = "Connection is null.";
    private static final String PPOPINACT = "60";

 
    /**
     * getIdsPropInact gets just the EPL_IDs from the EPL table.
     * 
     * @param type is the entity type for the ids
     * @return The list of Ids
     */
    @Override
    public List<Long> getIdsPropInact(EntityType type) {

        Connection con = null;
        List<Long> list = new ArrayList<Long>();
        PreparedStatement pstmt = null;

        String query = "";

        if (EntityType.NDC.equals(type)) {
            query =
                "select NDC.EPL_ID" 
                    + " from EPL_NDCS NDC, EPL_VADF_OWNERS OWNER, EPL_VADF_NONLIST_VALUES NON_LIST" 
                    + " WHERE OWNER.EPL_ID_NDC_FK = NDC.EPL_ID AND NDC.ITEM_STATUS = 'ACTIVE'" 
                    + " AND OWNER.ID = NON_LIST.VADF_OWNER_ID_FK AND VADF_ID_FK = " + PPOPINACT;
        } else if (EntityType.PRODUCT.equals(type)) {
            query =
                "select PRODUCT.EPL_ID" 
                    + " from EPL_PRODUCTS PRODUCT, EPL_VADF_OWNERS OWNER, EPL_VADF_NONLIST_VALUES NON_LIST" 
                    + " WHERE OWNER.EPL_ID_PRODUCT_FK = PRODUCT.EPL_ID AND PRODUCT.ITEM_STATUS = 'ACTIVE'" 
                    + " AND OWNER.ID = NON_LIST.VADF_OWNER_ID_FK AND VADF_ID_FK = " + PPOPINACT;
        } else if (EntityType.ORDERABLE_ITEM.equals(type)) {
            query =
                "select OI.EPL_ID" 
                    + " from EPL_ORDERABLE_ITEMS OI, EPL_VADF_OWNERS OWNER, EPL_VADF_NONLIST_VALUES NON_LIST" 
                    + " WHERE OWNER.EPL_ID_OI_FK = OI.EPL_ID AND OI.ITEM_STATUS = 'ACTIVE'" 
                    + " AND OWNER.ID = NON_LIST.VADF_OWNER_ID_FK AND VADF_ID_FK = " + PPOPINACT;
        } else {
            return null;
        }

        con = getJNDIConnection();
        if (con == null) {
            LOG.error(MSG_CONN_NULL);
            return list;
        } 

        ResultSet rs = null;

        try {
            pstmt = con.prepareStatement(query); // create a statement
            rs = pstmt.executeQuery();

            // extract data from the ResultSet
            while (rs.next()) {
                try {
                    list.add(rs.getLong(1));
                } catch (Exception e) {
                    LOG.error(validateStringInput(e.getMessage(), LOG_FORGING));
                }
            }
        } catch (Exception e) {
            LOG.error(validateStringInput(e.getMessage(), LOG_FORGING));
        } finally {
            safeClose(rs);
            safeClose(pstmt);
            safeClose(con);
        }

        return list;

    }

    
    /**
     * Setup and return a connection
     *
     * @return Connection
     */
    protected Connection setupConnection() {
        Connection con = null; //NOPMD

        con = getJNDIConnection();
        
        if (con == null){
            LOG.error("Connection is null");
        }

        return con;
    }



    /**
     * Uses JNDI and Datasource (preferred style).
     * 
     * @return Connection
     */
    private Connection getJNDIConnection() {

        // Obtain our environment naming context
        Context initCtx;
        Connection conn = null; //NOPMD

        try {
            ConfigFileUtility cfu = new ConfigFileUtility();
            initCtx = new InitialContext();
            Hashtable<String, String> ht = new Hashtable<String, String>();

            ht.put(Context.INITIAL_CONTEXT_FACTORY, "weblogic.jndi.WLInitialContextFactory");

            ht.put(Context.PROVIDER_URL, "t3://localhost:" + cfu.getPort());

            initCtx = new InitialContext(ht);

            // Lookup this DataSouce at the top level of the WebLogic JNDI tree
            DataSource datasource = (DataSource) initCtx.lookup("datasource.NationalEPL");

            conn = datasource.getConnection();

        } catch (NamingException ex) {
            LOG.error("Cannot get connection: " + validateStringInput(ex.toString(), LOG_FORGING));
        } catch (SQLException ex) {
            LOG.error("Cannot get connection: " + validateStringInput(ex.toString(), LOG_FORGING));
        }

        return conn;
    }


}
