/**
 * Source file created in 2008 by Southwest Research Institute
 */


package gov.va.med.pharmacy.peps.domain.common.capability;


import java.util.Date;
import java.util.List;

import gov.va.med.pharmacy.peps.common.vo.NationalSettingVo;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.domain.common.dao.EplNationalSettingDao;
import gov.va.med.pharmacy.peps.domain.common.utility.converter.NationalSettingConverter;


/**
 * Perform CRUD operations on NationalSetting
 */
public interface NationalSettingDomainCapability {

    /**
     * Creates the FDB add file
     * 
     * @param nationalSettingVo migration file
     * @param user provides user information
     * @return NationalSettingVo FDB file returned
     */
    NationalSettingVo create(NationalSettingVo nationalSettingVo, UserVo user);

    /**
     * Retrieves a list of National Setting files
     * 
     * @return List<NationalSettingVo> list of National Setting files returned
     */
    List<NationalSettingVo> retrieve();
    
    /**
     * retrieve
     *
     * @param id Long
     * @return NationalSettingVo
     */
    NationalSettingVo retrieve(Long id);
    
    /**
     * retrieve
     *
     * @param keyName String
     * @return NationalSettingVo
     */
    NationalSettingVo retrieve(String keyName);

    /**
     * Retrieves the Boolean value for the selected National Setting.
     * 
     * @param keyName the key of the National Setting to retriev
     * @return Boolean
     */
    Boolean retrieveBoolean(String keyName);
    
    /**
     * Retrieves the Date value for the selected National Setting.
     * 
     * @param keyName the key of the National Setting to retrieve
     * @return Boolean
     */
    Date retrieveDate(String keyName);
    
    /**
     * Retrieves the Integer value for the selected National Setting.
     * 
     * @param keyName the key of the National Setting to retrieve
     * @return Long
     */
    Long retrieveInteger(String keyName);
    
    /**
     * Retrieves the String value for the selected National Setting.
     * 
     * @param keyName the key of the National Setting to retrieve
     * @return String
     */
    String retrieveString(String keyName);
    
    /**
     * Updates the FDB file
     * 
     * @param nationalSettingVo
     *            migration file
     * @param user
     *            provides user information
     * @return nationalSettingVo FDB file returned
     */
    NationalSettingVo update(NationalSettingVo nationalSettingVo, UserVo user);

    /**
     * Updates the Boolean value for a National Setting with the specified key.
     * 
     * @param key the key of the National Setting
     * @param value the Boolean value to set
     * @param user the user performing the update.
     */
    NationalSettingVo update(String key, Boolean value, UserVo user);

    /**
     * Updates the Date value for a National Setting with the specified key.
     * 
     * @param key the key of the National Setting
     * @param value the Date value to set
     * @param user the user performing the update.
     */
    NationalSettingVo update(String key, Date value, UserVo user);

    /**
     * Updates the Integer value for a National Setting with the specified key.
     * 
     * @param key the key of the National Setting
     * @param value the Decimal value to set
     * @param user the user performing the update.
     */
    NationalSettingVo update(String key, Long value, UserVo user);

    /**
     * Updates the String value for a National Setting with the specified key.
     * 
     * @param key the key of the National Setting
     * @param value the String value to set
     * @param user the user performing the update.
     */
    NationalSettingVo update(String key, String value, UserVo user);

    /**
     * Deletes all of the NationalSettingVo's.
     */
    void deleteAll();
    
    /**
     * delete a specific setting
     *
     * @param id Long
     */
    void delete(Long id);
    
    /**
     * delete a specific setting
     *
     * @param keyName String
     */
    void delete(String keyName);

    /**
     * set the Local console info
     * 
     * @param nationalSettingDao EplNationalSettingDao
     */
    void setEplNationalSettingDao(EplNationalSettingDao nationalSettingDao);

    /**
     * setNationalSettingConverter
     * 
     * @param nationalSettingConverter
     *            NationalSettingConverter property
     */
    void setNationalSettingConverter(NationalSettingConverter nationalSettingConverter);
}
