/**
 * Source file created in 2007 by Southwest Research Institute
 */


package gov.va.med.pharmacy.peps.common.vo.validator.fields;


import java.util.regex.Matcher;
import java.util.regex.Pattern;

import gov.va.med.pharmacy.peps.common.utility.PPSConstants;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.validator.AbstractValidator;
import gov.va.med.pharmacy.peps.common.vo.validator.ErrorKey;
import gov.va.med.pharmacy.peps.common.vo.validator.Errors;


/**
 * validates the GCN Sequence Number
 */
public class GCNSequenceNumberValidator extends AbstractValidator<String> {

    /**
     * validates the GCN Sequence Number
     * 
     * @param target is the GCN Sequence Number
     * @param errors is the errors collection
     */
    @Override
    public void validate(String target, Errors errors) {
        if (isNullOrEmpty(target)) {
            rejectIfNullOrEmpty(target, errors, FieldKey.GCN_SEQUENCE_NUMBER);

            return;
        }
        
        //no decimal digits allowed
        Pattern p1 = Pattern.compile("^\\d+$");
        Matcher matcher1 = p1.matcher(target);
        boolean matchFound1 = matcher1.matches();

        if (!matchFound1) {
            errors.addError(FieldKey.GCN_SEQUENCE_NUMBER, ErrorKey.COMMON_NOT_NUMERIC,
                new Object[] {FieldKey.GCN_SEQUENCE_NUMBER});

            return;
        }
        
        // Defect 227281 - Verify the GCNSEQNO does not have more than 6 decimal digits.  Just in case so that a 
        // NumberFormatException is not thrown when trying to convert to a Long down below.
        rejectIfTrue(isLongerThanMax(target, PPSConstants.I6), ErrorKey.COMMON_MAX_MIN_VALUE_INCLUSIVE, errors, 
                FieldKey.GCN_SEQUENCE_NUMBER, 0, PPSConstants.I999999);
        if ( errors.size() > 0 ) {
            return;
        }

        rejectIfOutsideRangeInclusive(Long.valueOf(target), 0, PPSConstants.I999999, errors, FieldKey.GCN_SEQUENCE_NUMBER);
    }
}
