package gov.va.med.pharmacy.peps.common.vo;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

/**
 * Represents the status of an Update File as it goes through its lifecycle.
 */
public class StatusVo extends ValueObject {

    /**
     * Serial version id of the class.
     */
    private static final long serialVersionUID = 1L;
    /**
     * Status - Initiated - 1.
     */
    public static final Long INITIATED = Long.valueOf(1L);

    /**
     * Status - Created - 2.
     */
    public static final Long CREATED = Long.valueOf(2L);

    /**
     * Status - PPS-N/NDF Update Test File Transmitted to the test sFTP - 3.
     */
    public static final Long READY_FOR_TESTING = Long.valueOf(3L);
    
    /**
     * Status - Start of National VistA Processing - 4.
     */
    public static final Long START_OF_NATIONAL_PROCESSING = Long.valueOf(4L);
    
    /**
     * Status - PPS-N/NDF Update Test File Install Successful - 5.
     */
    public static final Long NATIONAL_INSTALL_SUCCESS = Long.valueOf(5L);
    
    /**
     * Status - PPS-N/NDF Update Test File Approved By PBM - 6.
     */
    public static final Long APPROVED = Long.valueOf(6L);
    
    /**
     * Status - PPS-N/NDF Update Test File Rejected By PBM - 7.
     */
    public static final Long REJECTED_BY_QA = Long.valueOf(7L);
    
    /**
     * Status - Transmitted to Production VistA - 8.
     */
    public static final Long TRANSMITTED_TO_PRODUCTION = Long.valueOf(8L);
    
    /**
     * Status - VistA Error - 98.
     */
    public static final Long VISTA_ERROR = Long.valueOf(98L);
    
    /**
     * Status - NDF Update Process Error in PPS-N - 99.
     */
    public static final Long PPSN_ERROR = Long.valueOf(99L);

    /**
     * Set of statuses that are in a final state, meaning there will be no further action on that item. 
     */
    public static final Set<Long> FINAL_STATE_STATUSES = 
            new HashSet<Long>(Arrays.asList(REJECTED_BY_QA, TRANSMITTED_TO_PRODUCTION, VISTA_ERROR, PPSN_ERROR));

    /**
     * Set of statuses that are NOT in a final state, meaning there could be further action on that item.
     */
    public static final Set<Long> NON_FINAL_STATE_STATUSES = 
            new HashSet<Long>(Arrays.asList(INITIATED, CREATED, READY_FOR_TESTING, START_OF_NATIONAL_PROCESSING, 
                    NATIONAL_INSTALL_SUCCESS, APPROVED));   
    
    /** The status id. */
    private Long statusId;
    
    /** The status name. */
    private String statusName;
    
    /** The status description. */
    private String statusDesc;

    /**
     * Zero-arg constructor.
     */
    public StatusVo() {
        super();
    }

    /**
     * Constructor that specifies a Status Id.  Note that the StatusVo would not be fully populated, caller needs to further 
     * populate or only use this in situations where only the status id is needed.
     * @param statusId
     */
    public StatusVo(Long statusId) {
        super();
        this.statusId = statusId;
    }

    /**
     * Get Status Id.
     * @return the status id
     */
    public Long getStatusId() {

        return statusId;
    }

    /**
     * Set Status Id.
     * @param statusId new status id
     */
    public void setStatusId(Long statusId) {

        this.statusId = statusId;
    }

    /**
     * Get Status Name.
     * @return the status name
     */
    public String getStatusName() {

        return statusName;
    }

    /**
     * Set Status Name.
     * @param statusName new status name
     */
    public void setStatusName(String statusName) {

        this.statusName = statusName;
    }

    /**
     * Get Status Description.
     * @return the status description
     */
    public String getStatusDesc() {

        return statusDesc;
    }

    /**
     * Set Status Description.
     * @param statusDesc new status description
     */
    public void setStatusDesc(String statusDesc) {

        this.statusDesc = statusDesc;
    }

    /**
     * Returns true if the status is {@code READY_FOR_TESTING}; false if it is not.
     * @return boolean true if the status is {@code READY_FOR_TESTING}; false if it is not
     */
    public boolean isReadyForTesting() {
        return getStatusId().equals(READY_FOR_TESTING);
    }

    /**
     * Returns true if the status is {@code REJECTED_BY_QA}; false if it is not.
     * @return boolean true if the status is {@code REJECTED_BY_QA}; false if it is not
     */
    public boolean isRejectedByQA() {
        return getStatusId().equals(REJECTED_BY_QA);
    }

    /**
     * Returns true if the status is {@code TRANSMITTED_TO_PRODUCTION}; false if it is not.
     * @return boolean true if the status is {@code TRANSMITTED_TO_PRODUCTION}; false if it is not
     */
    public boolean isTransmittedToProduction() {
        return getStatusId().equals(TRANSMITTED_TO_PRODUCTION);
    }

    /**
     * Returns true if the status is {@code VISTA_ERROR}; false if it is not.
     * @return boolean true if the status is {@code VISTA_ERROR}; false if it is not
     */
    public boolean isVistaError() {
        return getStatusId().equals(VISTA_ERROR);
    }    
 
}
