/**
 * Source file created in 2011 by Southwest Research Institute
 */


package gov.va.med.pharmacy.peps.common.utility;


import static gov.va.med.pharmacy.peps.common.utility.IOUtility.safeClose;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.springframework.stereotype.Component;


/**
 * The ConfigFileUtility class is used to setup the configuration file 
 *
 */
@Component
public class ConfigFileUtility {

    /** IMAGE_PATH */
    public static final String DEFAULT_FDB_IMAGE_PATH = "/FDB_Images/";

    /** DEFAULT_PORT */
    public static final int DEFAULT_PORT = 7001;

    /** DEFAULT_TIMEOUT */
    public static final int DEFAULT_TIMEOUT = 60;

    /** DEFAULT_PROXY */
    public static final int DEFAULT_PROXY = 360;

    /** DEFAULT_DIVISION */
    public static final int DEFAULT_DIVISION = 521;

    /** DEFAULT_CONNSPEC */
    public static final String DEFAULT_CONNSPEC = "DUZ";

    /** DEFAULT_RPCTIMEOUT */
    public static final int DEFAULT_RPCTIMEOUT = 60000;

//    private static final Logger LOG = LogManager.getLogger(ConfigFileUtility.class);

    private static String START_PORT_TAG = "<NationalPort>";
    private static String STOP_PORT_TAG = "</NationalPort>";
    private static String START_TIMEOUT_TAG = "<TransactionTimeout>";
    private static String STOP_TIMEOUT_TAG = "</TransactionTimeout>";
    private static String START_PROXY_TAG = "<NDFProxyUserIEN>";
    private static String STOP_PROXY_TAG = "</NDFProxyUserIEN>";
    private static String START_DIV_TAG = "<NDFDivision>";
    private static String STOP_DIV_TAG = "</NDFDivision>";
    private static String START_CONNSPEC_TAG = "<ConnectionSpecName>";
    private static String STOP_CONNSPEC_TAG = "</ConnectionSpecName>";
    private static String START_RPCTIMEOUT_TAG = "<RPCTimeout>";
    private static String STOP_RPCTIMEOUT_TAG = "</RPCTimeout>";
    private static String START_FDBLOC_TAG = "<FDBImageLocation>";
    private static String STOP_FDBLOC_TAG = "</FDBImageLocation>";
    private static String START_STS_WEBSERVICE_TAG = "<STSWebServiceURL>";
    private static String STOP_STS_WEBSERVICE_TAG = "</STSWebServiceURL>";

    private int port;
    private int timeout;
    private int ndfProxyUserDUZ;
    private int division;
    private int rpcTimeout;
    private String connectionSpecName;
    private String fdbImageLocation;
    private String stsWebServiceURL;

    /**
     * Default Constructor
     */
    public ConfigFileUtility() {

        init();
    }

    /**
     * getConnectionSpecName
     * @return connectionSpecName
     */
    public String getConnectionSpecName() {

        return connectionSpecName;
    }

    /**
     * setConnectionSpecName
     * @param connectionSpecName : connectionSpecName
     */
    public void setConnectionSpecName(String connectionSpecName) {

        this.connectionSpecName = connectionSpecName;
    }

    /**
     * getRpcTimeout
     * @return rpcTimeout
     */
    public int getRpcTimeout() {

        return rpcTimeout;
    }

    /**
     * setRpcTimeout
     * @param rpcTimeout : rpcTimeout
     */
    public void setRpcTimeout(int rpcTimeout) {

        this.rpcTimeout = rpcTimeout;
    }

    /**
     * getPort
     * @return port
     */
    public int getPort() {

        return port;
    }

    /**
     * setPort
     * @param port : port
     */
    public void setPort(int port) {

        this.port = port;
    }

    /**
     * getTimeout    
     * @return : timeout
     */
    public int getTimeout() {

        return timeout;
    }

    /**
     * setTimeout
     * @param timeout : timeout
     */
    public void setTimeout(int timeout) {

        this.timeout = timeout;
    }

    /**
     * getNdfProxyUserDUZ
     * @return ndfProxyUserDUZ
     */
    public int getNdfProxyUserDUZ() {

        return ndfProxyUserDUZ;
    }

    /**
     * setNdfProxyUserDUZ
     * @param ndfProxyUserDUZ : ndfProxyUserDUZ
     */
    public void setNdfProxyUserDUZ(int ndfProxyUserDUZ) {

        this.ndfProxyUserDUZ = ndfProxyUserDUZ;
    }

    /**
     * getDivision
     * @return division
     */
    public int getDivision() {

        return division;
    }

    /**
     * setDivision
     * @param division division
     */
    public void setDivision(int division) {

        this.division = division;
    }

    /**
     * get the fdbImageLocation
     * 
     * @return the fdbImageLocation
     */
    public String getFdbImageLocation() {

        return fdbImageLocation;
    }

    /**
     * sets fdbImageLocation field
     * 
     * @param fdbImageLocation the fdbImageLocation to set
     */
    public void setFdbImageLocation(String fdbImageLocation) {

        this.fdbImageLocation = fdbImageLocation;
    }

    /**
     * Returns the STS WebService URL.
     * 
     * @return String
     */
    public String getStsWebServiceURL() {
    
        return stsWebServiceURL;
    }

    /**
     * Sets the STS WebService URL.
     * 
     * @param stsWebServiceURL the STS WebService URL
     */
    public void setStsWebServiceURL(String stsWebServiceURL) {
    
        this.stsWebServiceURL = stsWebServiceURL;
    }

    /**
     * init()
     */
    private void init() {

        port = DEFAULT_PORT;
        timeout = DEFAULT_TIMEOUT;
        ndfProxyUserDUZ = DEFAULT_PROXY;
        division = DEFAULT_DIVISION;
        connectionSpecName = DEFAULT_CONNSPEC;
        rpcTimeout = DEFAULT_RPCTIMEOUT;
        fdbImageLocation = DEFAULT_FDB_IMAGE_PATH;

        String s = new String();

        InputStream is = this.getClass().getClassLoader().getResourceAsStream("xml/PPS-NConfig.xml");
        InputStreamReader isr = new InputStreamReader(is);
        BufferedReader reader = new BufferedReader(isr);
        try {
            int iPort;
            int iTimeout;
            int iNDFProxy;
            int iDivision;
            int iRPCTimeout;
            String iConnSpec;
            String iFdbLocation;

            while ((s = reader.readLine()) != null) {
                try {
                    iPort = this.getPropertyInt(s, START_PORT_TAG, STOP_PORT_TAG, DEFAULT_PORT);
                    port = iPort;
                } catch (PropertyNotFoundInXmlLineException e) {
                    log(e.getMessage());
                }

                try {
                    iTimeout = this.getPropertyInt(s, START_TIMEOUT_TAG, STOP_TIMEOUT_TAG, DEFAULT_TIMEOUT);
                    timeout = iTimeout;
                } catch (PropertyNotFoundInXmlLineException e) {
                    log(e.getMessage());
                }

                try {
                    iNDFProxy = this.getPropertyInt(s, START_PROXY_TAG, STOP_PROXY_TAG, DEFAULT_PROXY);
                    ndfProxyUserDUZ = iNDFProxy;
                } catch (PropertyNotFoundInXmlLineException e) {
                    log(e.getMessage());
                }

                try {
                    iDivision = this.getPropertyInt(s, START_DIV_TAG, STOP_DIV_TAG, DEFAULT_DIVISION);
                    division = iDivision;
                } catch (PropertyNotFoundInXmlLineException e) {
                    log(e.getMessage());
                }

                try {
                    iRPCTimeout = this.getPropertyInt(s, START_RPCTIMEOUT_TAG, STOP_RPCTIMEOUT_TAG, DEFAULT_RPCTIMEOUT);
                    rpcTimeout = iRPCTimeout;
                } catch (PropertyNotFoundInXmlLineException e) {
                    log(e.getMessage());
                }

                try {
                    iConnSpec = getProperty(s, START_CONNSPEC_TAG, STOP_CONNSPEC_TAG, null);
                    connectionSpecName = iConnSpec;
                } catch (PropertyNotFoundInXmlLineException e) {
                    log(e.getMessage());
                }

                try {
                    iFdbLocation = getProperty(s, START_FDBLOC_TAG, STOP_FDBLOC_TAG, null);
                    fdbImageLocation = iFdbLocation;
                } catch (PropertyNotFoundInXmlLineException e) {
                    log(e.getMessage());
                }

                try {
                    String value = getProperty(s, START_STS_WEBSERVICE_TAG, STOP_STS_WEBSERVICE_TAG, null);
                    setStsWebServiceURL(value);
                } catch (PropertyNotFoundInXmlLineException e) {
                    log(e.getMessage());
                }
            }

            
        } catch (Exception e) {
            log("Exception: " + e);
            log("Setting defaults port:" + port + " timeout:" + timeout + " ndfDUZ:" + ndfProxyUserDUZ + " division: "
                + division);
        }
        finally{
            safeClose(reader);
        }
    }

    /**
     * getPropertyInt - gets an integer property from the xml line
     *
     * @param s String
     * @param startTag String
     * @param stopTag String
     * @param defaultValue int
     * @return int
     * @throws PropertyNotFoundInXmlLineException exception
     */
    private int getPropertyInt(String s, String startTag, String stopTag, int defaultValue)
        throws PropertyNotFoundInXmlLineException {

        int rv = defaultValue;

        if (s.contains(startTag) && s.contains(stopTag)) {
            int beginIndex = s.indexOf(startTag) + startTag.length();
            int endIndex = s.indexOf(stopTag);
            String strPort = s.substring(beginIndex, endIndex);

            try {
                rv = Integer.valueOf(strPort).intValue();
            } catch (NumberFormatException nfe) {
                rv = defaultValue;
            }

            return rv;
        }

        throw new PropertyNotFoundInXmlLineException();
    }

    /**
     * getProperty - gets a value from the xml line (a string)
     *
     * @param s String
     * @param startTag String
     * @param stopTag String
     * @param defaultValue String
     * @return String
     * @throws PropertyNotFoundInXmlLineException exception
     */
    private String getProperty(String s, String startTag, String stopTag, String defaultValue)
        throws PropertyNotFoundInXmlLineException {

        String rv = defaultValue;

        if (s.contains(startTag) && s.contains(stopTag)) {
            int beginIndex = s.indexOf(startTag) + startTag.length();
            int endIndex = s.indexOf(stopTag);
            rv = s.substring(beginIndex, endIndex);

            return rv;
        }

        throw new PropertyNotFoundInXmlLineException();
    }

    /**
     * log 
     * @param s : String to Log
     */
    private void log(String s) {

    }

    /**
     * this exception class is used to skip the assignment of a property.
     */
    private class PropertyNotFoundInXmlLineException extends Exception {

        /** serialVersionUID */
        private static final long serialVersionUID = 8015312492996833208L;
        private static final String DEFAULT_MSG = "Property not found in input string";

        /**
         * Constructor
         *
         */
        public PropertyNotFoundInXmlLineException() {

            super(DEFAULT_MSG);
        }

        /**
         * Constructor
         *
         * @param message String
         */
        @SuppressWarnings("unused")
        public PropertyNotFoundInXmlLineException(String message) {

            super(message);
        }

    }
}
