package gov.va.med.pharmacy.peps.common.email;

import java.util.Arrays;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.mail.internet.MimeMessage;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.PropertySource;
import org.springframework.core.env.Environment;
import org.springframework.core.io.FileSystemResource;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;


@Configuration
@PropertySource("classpath:gov.va.med.pharmacy.peps.emailservice.properties")
public class EmailServiceImpl implements EmailService 
{
    
    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(EmailServiceImpl.class);

    private static String EMAIL_RESEND_INTERVAL_DEFAULT = "5";
    private static String EMAIL_RESEND_MAXIMUM_TIME_DEFAULT = "1";
    
	JavaMailSenderImpl ppsMailSender;

	@Autowired
	private Environment environment;
	
	 private ScheduledExecutorService exec = Executors.newScheduledThreadPool(1);
	
    /**
     * @param ppsMailSender
     */
    public void setPpsMailSender(JavaMailSenderImpl ppsMailSender) {
    
        this.ppsMailSender = ppsMailSender;
    }


    /** 
	 * This is method is intended to just send email without attachment (including cc). 
	 * 
	 */
	@Override
	public void sendEmail(Email email) {
		

		StringBuffer subjectAppender = new StringBuffer(email.getSubject());		
		
		String emailHostName = null;
		String fromEmailAddress = null;
		
		
        try {
                       
            emailHostName =  environment.getRequiredProperty("host_fqdn");
            fromEmailAddress =  environment.getRequiredProperty("default_from_email");
                  
        } 
        catch (Exception e) {
           LOG.error("Error Loading email properties.");
        }
		
	
        if(StringUtils.isNotEmpty(fromEmailAddress))
        {
            email.setFrom(fromEmailAddress);
        }
        
		LOG.debug("Email to :"+ Arrays.toString(email.getTo()));
		LOG.debug("Email from :"+ email.getFrom());
		LOG.debug("Email subject :"+ subjectAppender.toString());
		LOG.debug("Email message body :"+ email.getEmailMessage());
		LOG.debug("Sending email to host :"+ emailHostName);
		
		SimpleMailMessage message = new SimpleMailMessage();

		message.setFrom(email.getFrom());
		message.setTo(email.getTo());
		
		LOG.debug("emailBean.getCc():"+ Arrays.toString(email.getCc()));
		
		if (email.hasCc()) 
		{
			try 
			{
				message.setCc(email.getCc());
			} catch(Exception exec) 
			{
				LOG.debug("Ignoring sendEmail cc exception");
			}
		}		
		
		message.setSubject(subjectAppender.toString());
		message.setText(email.getEmailMessage());
		
		ppsMailSender.setHost(emailHostName);
		
		ppsMailSender.send(message);		
	}
	
	/**
	 * This method sends an rich email (including cc and attachment)
	 */
	@Override
	public void sendRichEmail(RichEmail richEmailBean) throws Exception {
		
	    LOG.info("Entering sendRichEmail");
		
	    MimeMessage message = null;
	    
	    String emailHostName = null;
	    String fromEmailAddress = null;
        
        try {
           
            emailHostName =  environment.getRequiredProperty("host_fqdn");
            fromEmailAddress =  environment.getRequiredProperty("default_from_email");
            
        } 
        catch (Exception e) {
           LOG.error("Error Loading email properties.", e);
        }
	    
		
		try{			
				    
			message = ppsMailSender.createMimeMessage();
			
			LOG.info("Created Mime message successfully");
			
			StringBuffer subjectAppender = new StringBuffer(richEmailBean.getSubject());
			
			
			
			MimeMessageHelper helper = new MimeMessageHelper(message, true);
			
			LOG.info("Created helper instance successfully");
			
			 if(StringUtils.isNotEmpty(fromEmailAddress))
		     {
				 helper.setFrom(fromEmailAddress);
		     }
			
			 
			// No need to append anything for production
			if(StringUtils.isNotEmpty(richEmailBean.getFrom())){
			
				helper.setFrom(richEmailBean.getFrom());
			}
			
			
			LOG.debug("richEmailBean.getCc(): "+ Arrays.toString(richEmailBean.getCc()));
			
			if (richEmailBean.hasCc()) {
				try {
					helper.setCc(richEmailBean.getCc());
				} catch(Exception exec) {
					LOG.debug("Ignoring cc exception ", exec);
				}
			}
			helper.setTo(richEmailBean.getTo());
			helper.setSubject(subjectAppender.toString());
			helper.setText(richEmailBean.getEmailMessage());

			FileSystemResource file = new FileSystemResource(richEmailBean.getAttachmentPath());
			
			LOG.debug("File name:"+ file.getFilename());
			
			String attachmentName = richEmailBean.getAttachmentName();
			if (attachmentName == null){
				attachmentName = file.getFilename();
				LOG.debug("Please speficy the attachment name or else default name will be used");
			}
			
			helper.addAttachment(attachmentName, file);
						
			LOG.debug("Email to :"+ Arrays.toString(richEmailBean.getTo()));
			LOG.debug("Email cc :"+ Arrays.toString(richEmailBean.getCc()));
			LOG.debug("Email from :"+ richEmailBean.getFrom());
			LOG.debug("Email subject :"+ subjectAppender.toString());
			LOG.debug("Email message body :"+ richEmailBean.getEmailMessage());
			LOG.debug("Email attachment name :"+ richEmailBean.getAttachmentName());
			LOG.debug("Sending email to host :"+ emailHostName);
			
			ppsMailSender.setHost(emailHostName);
			LOG.info("Sending Email");
			
		}catch (Exception e) {
			LOG.error("Exception while sending Email ",e);
		}
		ppsMailSender.send(message);
		LOG.info("Email sent successfully");
		LOG.info("Exiting sendRichEmail");
	}


	 /** 
     * This is method is intended to send an email multiple times until it is successful 
     * or a configurable amount of time has passed.    
     */
    @Override
    public void sendEmailMutlipleTimes(Email email) {
        new RepeatedSchedule(exec, email).submit();       
    }
    
    /**
     * Shuts down the ScheduledExecutorService to stop any other threads running when Spring destroys this instance. 
     */
    public final void destroy(){
        exec.shutdownNow();
    }    
    
   
    private class RepeatedSchedule implements Runnable {
        
        private ScheduledExecutorService ses;        
        private Future<?> control;
        private Email email;
        private Calendar intialTime;
        private long interval = 5;
        private int maximumTime = 1;

        public RepeatedSchedule(ScheduledExecutorService ses, Email email) {
            this.ses = ses;
            this.email = email;
            this.intialTime = new GregorianCalendar();
            this.interval = 
                new Long (environment.getProperty("email_resend_interval", EMAIL_RESEND_INTERVAL_DEFAULT)).longValue();
            this.maximumTime = 
                new Integer(environment.getProperty("email_resend_maximum_time", EMAIL_RESEND_MAXIMUM_TIME_DEFAULT)).intValue();            
        }

        public void submit() {
            
            // We submit this, not the received command
            this.control = this.ses.scheduleWithFixedDelay(this, 0, interval, TimeUnit.MINUTES);            
        }

        @Override
        public void run() {
             Calendar currentTime = new GregorianCalendar();
             Calendar maxExecutionTime = ((Calendar)intialTime.clone());
             maxExecutionTime.add(Calendar.HOUR, maximumTime);
             
             if (currentTime.after(maxExecutionTime)){
                 LOG.error("The application has been trying to send an email message for the last " + maximumTime + 
                            " hour(s) with no success. The application will no longer try to send it." );
                 this.control.cancel(true);
             }
             else {
                 try {
                     sendEmail(this.email);
                     this.control.cancel(true);
                 }catch (Throwable ex){
                     LOG.error("Error sending e-mail. Will try again in " + interval +" more minutes", ex);
                 }
             }
        }

    }


    

}
