package gov.va.med.authentication.kernel.sspi.authentication.manageable;

/**
 * Represents a user in the database.  The user is cached in memory.
 *
 * 
 * @author VHIT Security and Other Common Services (S&OCS)
 * @version 1.1.0.002
 */

/*package*/ class KaajeeUserEntry extends KaajeePrincipalEntry
{
  // All user database keys are prefixed by this:
  private static final String PREFIX = "user.";

  // The user's password:
  private String password;

  /**
   * The constructor.
   *
   * @param name a String containing the user's name.
   *
   * @param database a UserGroupDatabase containing the
   * users and groups for the manageable sample authenticator.
   *
   * @throws IllegalArgumentException if the name is
   * invalid (eg. starts with whitespace).
   */
  /*package*/ KaajeeUserEntry(String name, KaajeeUserGroupDatabase database)
  {
    super(PREFIX, name, database);
    if (exists()) {
      // The user exists.  Get's its persistent state from the database.
      String value = getValue();

      // A user's persistent state the password followed by a
      // comma separated list of the names of the groups immediately
      // containing this user as a member.
      // Extract the user's password and groups from its persisten state.
      int commaIndex = value.indexOf(",");
      if (commaIndex >= 0) {
        password = value.substring(0, commaIndex);
        String groupsString = value.substring(commaIndex+1);
        initializeGroups(groupsString);
      } else {
        password = value;
      }
    }
  }

  /**
   * Extract the user's name from its database key.
   *
   * If a database key for a group is passed in, return null.
   *
   * @return a String containing the user's name.
   */
  protected static String getNameFromKey(String key)
  {
    return (key.startsWith(PREFIX)) ? key.substring(PREFIX.length()) : null;
  }

  /**
   * Add this user to the database if it doesn't already exist.
   *
   * @param password a String containing this user's password.
   *
   * @throws IllegalArgumentException if the password is
   * illegal (eg. starts with whitespace)
   */
  /*package*/ void create(String password)
  {
    if (!exists()) {
      setPassword(password);
    }
  }

  /**
   * Write this user's persistent state to the database.
   *
   * A user's persistent state the password followed by a
   * comma separated list of the names of the groups immediately
   * containing this user as a member.
   */
  protected void update()
  {
    String value = getPassword();
    String groupsString = getGroupsString();
    if (groupsString.length() > 0) {
      value = value + "," + groupsString;
    }
    setValue(value);
  }

  /**
   * Get this user's password.
   *
   * @return a String containing this user's password.
   */
  /*package*/ String getPassword()
  {
    return password;
  }

  /**
   * Set this user's password and write it to the database.
   *
   * @param password a String containing the user's password.
   *
   * @throws IllegalArgumentException if the password is
   * illegal (eg. starts with whitespace)
   */
  /*package*/ void setPassword(String password)
  {
    Utils.checkVal(password);
    this.password = password;
    update();
  }
}
