package gov.va.med.pharmacy.hazardous.parser;

import gov.va.med.pharmacy.hazardous.db.HazardousDBManager;
import gov.va.med.pharmacy.hazardous.util.Log;
import gov.va.med.pharmacy.hazardous.util.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.Properties;
import java.util.Objects;


/**
 * Reads in a comma-separated file containing VUID, EPL ID, Product Name, Primary
 * EPA Code, Waste Sort Code, DOT Shipping Name, updates product records, and
 * inserts outgoing differences to get picked up in a subsequent NDF Update
 * File.
 */
public class HazardousWasteParser extends HazardousDBManager implements IHazardousParser {
	
    private static final String EPL_PRODUCT_ID = "EPL_ID";
    private static final String EPL_PRODUCT_VUID = "VUID";
    private static final String EPL_PRODUCT_NAME = "VA_PRODUCT_NAME"; 
    private static final String EPL_PRIMARY_EPA = "PRIMARY_EPA";
    private static final String EPL_HAZARDOUS_TO_HANDLE = "HAZARDOUS_TO_HANDLE";
    private static final String EPL_HAZARDOUS_TO_DISPOSE = "HAZARDOUS_TO_DISPOSE";
    private static final String EPL_WASTE_SORT_CODE = "WASTE_SORT_CODE";
    private static final String EPL_DOT_SHIPPING_NAME = "DOT_SHIPPING_NAME";
    private static final String H2D_YES = "Y";

    private static final int VUID_INX = 0;
    private static final int PRODUCT_NAME_INX = 4;
    private static final int HAZARDOUS_TO_HANDLE_INX = 8;
    private static final int HAZARDOUS_TO_DISPOSE_INX = 9;
    private static final int PRIMARY_EPA_CODE_INX = 10;
    private static final int WASTE_SORT_CODE_INX = 11;
    private static final int DOT_SHIPPING_NAME_INX = 12;
    
    private int recordsUnaffectedCount = 0;
     
    // csv file with quote delimiters
    private static final String REG_EX_INPUT_LINE_PATTERN = ",(?=([^\"]*\"[^\"]*\")*[^\"]*$)";

    private static final String QUERY_BY_VUID_SQL = 
            "Select EPL_ID, VUID, VA_PRODUCT_NAME, HAZARDOUS_TO_DISPOSE, HAZARDOUS_TO_HANDLE, PRIMARY_EPA, WASTE_SORT_CODE, DOT_SHIPPING_NAME from EPL_PRODUCTS where VUID = ?";
    private PreparedStatement vuidPreparedStatement;
     
    private static final String COMMON_UPDATE_SQL = "Update EPL_PRODUCTS SET HAZARDOUS_TO_HANDLE = ?, HAZARDOUS_TO_DISPOSE = ?,"
            + " PRIMARY_EPA = ?, WASTE_SORT_CODE = ?, DOT_SHIPPING_NAME = ?, LAST_MODIFIED_BY = ?," 
            + " LAST_MODIFIED_DTM = SYSDATE";

    private PreparedStatement updatePreparedStatementVUID;
    private static final String UPDATE_BY_VUID_SQL = COMMON_UPDATE_SQL + " where VUID = ?";
    
    /** SQL for finding product records last modified by 'HAZWASTERUN1'. */
    private final String QUERY_FOR_DIFFERENCES = "select NDF_PRODUCT_IEN, HAZARDOUS_TO_HANDLE, HAZARDOUS_TO_DISPOSE, PRIMARY_EPA, "
            + "WASTE_SORT_CODE,DOT_SHIPPING_NAME from EPL_PRODUCTS where VUID = ?";    
    PreparedStatement queryPS = null;

    /** SQL for inserting EPL_NDF_OUTGOING_DIFFERENCES records.  Contains four placeholders:<br>
     *  <ul>
     *  <li>1- NEW_VALUE</li>
     *  <li>2- OLD_VALUE</li>
     *  <li>3- VISTA_FIELD_NUMBER</li>
     *  <li>4- VISTA_IEN</li>
     *  */
    private final String INSERT_DIFFERENCES_SQL = "insert into EPL_NDF_OUTGOING_DIFFERENCES " 
            + "(ACTION_TYPE, CREATED_BY, CREATED_DTM, NDC_UPDATE_FILE_FK, NDF_OUTGOING_DIFFERENCES_ID, " 
            + "NEW_VALUE, OLD_VALUE, VISTA_FIELD_NUMBER, VISTA_FILE_NUMBER, VISTA_IEN) VALUES " 
            + "('M', ?, SYSDATE, NULL, EPL_NDF_OUTGOING_DIFFERENS_SEQ.nextval, ?, ?, ?, '50.68', ?)";
    PreparedStatement insertDifferencePS = null;
    
    /** SQL for inserting EPL_ITEM_AUDIT_HISTORY records.  Contains four placeholders:<br>
     *  <ul>
     *  <li>1- EPL_ID (generated)</li>
     *  <li>2- AUDIT_ITEM_ID</li>
     *  <li>3- CREATED_BY</li>
     *  <li>4- LAST_MODIFIED_BY</li>
     *  */
    private final String INSERT_ITEM_AUDIT ="Insert into EPL_ITEM_AUDIT_HISTORY"
            + " (EPL_ID, AUDIT_ITEM_TYPE, AUDIT_ITEM_ID, SITE_NAME, EVENT_CATEGORY, "
            + "REASON, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) VALUES "
            + "(?, 'PRODUCT', ?, '999', 'SYSTEM_EVENT', 'HAZWASTELOAD', ?, SYSDATE, ?, SYSDATE)";
    PreparedStatement insertIAPreparedStatement = null;
 
    /** SQL for inserting EPL_ITEM_AUDIT_HISTORY_DETAILS records.  Contains four placeholders:<br>
     *  <ul>
     *  <li>1- COL_NM</li>
     *  <li>2- OLD_VALUE</li>
     *  <li>3- NEW_VALUE</li>
     *  <li>4- IAH_ID_FK</li>
     *  <li>3- CREATED_BY</li>
     *  <li>4- LAST_MODIFIED_BY</li>
     *  */
    private final String INSERT_ITEM_AUDIT_DETAILS = "Insert into EPL_ITEM_AUDIT_HISTORY_DETAILS "
            + "(EPL_ID, COL_NM, OLD_VALUE, NEW_VALUE, DETAIL_EVENT_CATEGORY, " 
            + "IAH_ID_FK, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) VALUES "
            + "(EPL_ITEM_AUDIT_HIST_DTL_SEQ.NEXTVAL, ?, ?, ?, 'SYSTEM_EVENT', ?, ?, SYSDATE, ?, SYSDATE)";
    PreparedStatement insertIADetailsPreparedStatement = null;

    private Properties connectionProperties;	

    private static int totalRecsUpdated=0;
    
    public HazardousWasteParser(Properties connectionProperties) {
        this.connectionProperties = connectionProperties;
    }

    @Override
    public void parse(File hazardousWasteFile, File outputExceptionFile, String lastUpdateUser) throws HazardousParserException {

        FileWriter exceptionLogWriter = null;

        BufferedReader reader = null;
        int lineCount = 0;
        String line = null;

        boolean foundMatchInSearch = false;

        try {
            exceptionLogWriter = new FileWriter(outputExceptionFile, false);
            exceptionLogWriter.write("VUID, EPL_ID, CMOP_ID, NDF_PRODUCT_IEN, VA_PRODUCT_NAME, GENERIC_NAME, SCHEDULE_NAME, " 
            	+ "DF_NAME, HAZARDOUS_TO_DISPOSE, HAZARDOUS_TO_HANDLE, PRIMARY_EPA, WASTE_SORT_CODE, DOT_SHIPPING_NAME"
            	+ "EXCEPTION\n");
            exceptionLogWriter.flush();

            connectToDb();
            reader = new BufferedReader(new FileReader(hazardousWasteFile));
            ResultSet resultsVUIDSearch = null;

            setupPreparedStatements();

            String tempVUID = null;

            // Loop through all rows in spreadsheet
            while ((line = reader.readLine()) != null) {
                lineCount++;

                logProgress(lineCount);

                String[] tokens = line.split(REG_EX_INPUT_LINE_PATTERN, 13);

                if (tokens.length != 13) {
                    //don't have to worry about log forging here, as we are already reading one line at a time 
                    Log.e("Invalid Line: " + line);
                    continue;
                }
                tokens = scrubTokens(tokens);

                try {

                    // VUID Search
                	tempVUID = (tokens[VUID_INX]); 
                    vuidPreparedStatement.setString(1, tempVUID);
                    resultsVUIDSearch = vuidPreparedStatement.executeQuery();

                    // Process the returned search results
                    foundMatchInSearch = processSearchResults(resultsVUIDSearch, tokens, 
                            updatePreparedStatementVUID, lastUpdateUser, exceptionLogWriter, line);

                    if (!foundMatchInSearch) {
                        // If VUID is not found:
		                exceptionLogWriter.write(line + ",No matches found in Search for VUID " + tempVUID + "\n");
		                exceptionLogWriter.flush();
                    }   
                } catch (SQLException e) {
                    Log.e(getClass(), "Error executing query: \n" + e);
                }
            }
        } catch (Throwable t) {
            Log.e(this.getClass(), t.toString());
            throw new HazardousParserException();
        } finally {
            Log.d("Processed " + lineCount + " records from " + hazardousWasteFile.getName() + ".");
            Log.d("Updated " + totalRecsUpdated + " records from " + hazardousWasteFile.getName() + ".");
            Log.d("Matched " + recordsUnaffectedCount + " records from " + hazardousWasteFile.getName() 
            	+ " in the EPL_PRODUCTS table.");
            closeReaderQuietly(reader);
            closeWriterQuietly(exceptionLogWriter);
            closePreparedStatementsQuietly();
            closeDbConnection();
        }
    }

    /**
     * Logs the progress of file processing every 1000 lines.
     * @param lineCount
     */
    private void logProgress(int lineCount) {
        if (lineCount % 1000 == 0) {
            Log.i("Progress: line " + lineCount);
        }
    }
    
    /**
     * Closes the specified reader quietly.
     * @param reader
     */
    private void closeReaderQuietly(BufferedReader reader) {
        if (reader != null)
            try {
                reader.close();
            } catch (IOException e) {
                // Ignore
            }
        reader = null;
    }

    /**
     * Closes all the prepared statements, squelching any exceptions.
     */
    private void closePreparedStatementsQuietly() {
        closePreparedStatementQuietly(vuidPreparedStatement);
        closePreparedStatementQuietly(updatePreparedStatementVUID);
        closePreparedStatementQuietly(queryPS);
        closePreparedStatementQuietly(insertDifferencePS);
        closePreparedStatementQuietly(insertIAPreparedStatement);
        closePreparedStatementQuietly(insertIADetailsPreparedStatement);
    }

    /**
     * Closes the specified writer, squelching any exceptions.
     * @param exceptionLogWriter
     */
    private void closeWriterQuietly(FileWriter exceptionLogWriter) {
        if (exceptionLogWriter != null) {
            try {
                exceptionLogWriter.flush();
                exceptionLogWriter.close();
            } catch (IOException e) {
                // Ignore
            }
        }
    }

    /**
     * Scrubs all the tokens for SQL Injection and validity.
     * 
     * @param tokens
     * @return tokens with Strings scrubbed
     * @throws RuntimeException
     *             if NDC Number is not a valid Long.
     */
    private String[] scrubTokens(String[] tokens) {

    	tokens[VUID_INX] = trimQuotesAndSpaces(tokens[VUID_INX]);
    	tokens[PRODUCT_NAME_INX] = trimQuotesAndSpaces(tokens[PRODUCT_NAME_INX]);
    	tokens[HAZARDOUS_TO_HANDLE_INX] = trimQuotesAndSpaces(tokens[HAZARDOUS_TO_HANDLE_INX]);
    	tokens[HAZARDOUS_TO_DISPOSE_INX] = trimQuotesAndSpaces(tokens[HAZARDOUS_TO_DISPOSE_INX]);
        tokens[PRIMARY_EPA_CODE_INX] = trimQuotesAndSpaces(tokens[PRIMARY_EPA_CODE_INX]);
        tokens[WASTE_SORT_CODE_INX] = trimQuotesAndSpaces(tokens[WASTE_SORT_CODE_INX]);
        tokens[DOT_SHIPPING_NAME_INX] = trimQuotesAndSpaces(tokens[DOT_SHIPPING_NAME_INX]);

        return tokens;
    }

    /**
     * @throws SQLException
     */
    private void setupPreparedStatements() throws SQLException {
        // Setup prepared statements
        vuidPreparedStatement = connection.prepareStatement(QUERY_BY_VUID_SQL);
        updatePreparedStatementVUID = connection.prepareStatement(UPDATE_BY_VUID_SQL);
        insertIAPreparedStatement = connection.prepareStatement(INSERT_ITEM_AUDIT);
        insertIADetailsPreparedStatement = connection.prepareStatement(INSERT_ITEM_AUDIT_DETAILS);
        queryPS = connection.prepareStatement(QUERY_FOR_DIFFERENCES);
        insertDifferencePS = connection
                .prepareStatement(INSERT_DIFFERENCES_SQL);
    }
    
    protected void printTokenLine(String[] tokens) {
        StringBuffer strb = new StringBuffer();
        int col = 0;
        for (String token : tokens) {
            strb.append("col " + col + ". " + token + " ");
            col++;
        }
        Log.d(strb.toString());
    }

    protected boolean printResultSet(final ResultSet resultSet) {
        boolean foundMatch = false;

        if (resultSet != null) {

            try {
                // Retrieve and print all fields and values
                ResultSetMetaData metaData = resultSet.getMetaData();
                // Log.m(metaData);
                StringBuffer strb = null;
                boolean first = true;
                int numberOfCols = metaData.getColumnCount();
                Log.d(getClass(), "printResultSet: columnCount: " + numberOfCols);
                while (resultSet.next()) {
                    strb = new StringBuffer();
                    foundMatch = true;
                    first = true;
                    for (int i = 1; i <= numberOfCols; i++) {
                        // Print all field values in the row
                        if (!first)
                            strb.append(" | ");
                        else
                            first = false;
                        strb.append(resultSet.getString(i));
                    }

                    Log.d(getClass(), strb.toString());
                }

            } catch (SQLException e) {
                Log.e(getClass(), "SQL exception: " + e.toString());
            }
        }
        return foundMatch;
    }

    protected boolean isConnectedToDb() {
        return super.isConnected();
    }

    protected void connectToDb() {
        super.connect(connectionProperties);
    }

    protected void closeDbConnection() {
        super.close();
    }

    protected boolean processSearchResults(final ResultSet resultSet, String[] tokens, PreparedStatement updatePreparedStatement, 
            String lastUpdateUser, FileWriter exceptionLogWriter, String inputData) {

    	String vuidValue = tokens[VUID_INX];
    	boolean foundMatch = false;
        try {
            // Scroll through result set and set the hazardous waste fields
            if (resultSet != null) {
                String inputProductName = null;
                while (resultSet.next()) {
					foundMatch = true;
                    String dbProductName = resultSet.getString(EPL_PRODUCT_NAME);
                    inputProductName = trimQuotesAndSpaces(tokens[PRODUCT_NAME_INX]);
                    
                	if (dbProductName.equals(inputProductName)) {
						String h2hFlag = tokens[HAZARDOUS_TO_HANDLE_INX];
						String h2dFlag = tokens[HAZARDOUS_TO_DISPOSE_INX];
						String primaryEPA = tokens[PRIMARY_EPA_CODE_INX];
						String wasteSortCode = tokens[WASTE_SORT_CODE_INX];
						String dotShippingName = tokens[DOT_SHIPPING_NAME_INX]; 						

	                    String dbHazardousToHandle = resultSet.getString(EPL_HAZARDOUS_TO_HANDLE);
	                    String dbHazardousToDispose = resultSet.getString(EPL_HAZARDOUS_TO_DISPOSE);
	                    String dbPrimaryEPA = resultSet.getString(EPL_PRIMARY_EPA);
	                    String dbWasteSortCode = resultSet.getString(EPL_WASTE_SORT_CODE);
	                    String dbDOTShippingName = resultSet.getString(EPL_DOT_SHIPPING_NAME);
                
						boolean hasH2DFields =(!StringUtils.isBlank(primaryEPA)||
								!StringUtils.isBlank(dotShippingName)||
								!StringUtils.isBlank(wasteSortCode));
							
						//compare input fields to database fields and don't update if they are the same
						if (Objects.equals(h2hFlag, dbHazardousToHandle) && Objects.equals(h2dFlag, dbHazardousToDispose) && Objects.equals(primaryEPA, dbPrimaryEPA)
								&& Objects.equals(wasteSortCode, dbWasteSortCode) && Objects.equals(dotShippingName, dbDOTShippingName)) {
			    			Log.d("No Update - All Harzadous Waste fields on input file and DB file match for VUID " + vuidValue);
			    			recordsUnaffectedCount++;
						}
						
						else if (!hasH2DFields) {
							processRecordUpdate(resultSet, tokens, updatePreparedStatement, lastUpdateUser);
						}
						else if (!StringUtils.isBlank(h2dFlag) && h2dFlag.equals(H2D_YES)) {
							processRecordUpdate(resultSet, tokens, updatePreparedStatement, lastUpdateUser);
						}
						else {
							//log sub-field populated w/o h2d exception
			                exceptionLogWriter.write(inputData + ",Hazardous waste sub-fields are populated when Hazardous to Dispose is not true" + "\n");
			                exceptionLogWriter.flush();
						}
					}
                	else {
                		// log Product Name exception
                		exceptionLogWriter.write(inputData + ",No database record found with input Product Name" + "\n");
                		exceptionLogWriter.flush();
                	}
                }
            }
            
        } catch (SQLException e) {
            Log.e(getClass(), "SQL exception: " + e.toString());

        } catch (Throwable e) {
            Log.e(getClass(), "Exception: " + e.toString());

        } finally {
            try {
                if (resultSet != null && !resultSet.isClosed()) {
                    resultSet.close();
                }
            } catch (SQLException e) {
                Log.e(getClass(), "Error closing resultset:" + e);
            }            
        }    	
        return foundMatch;
    }

	private int processRecordUpdate(final ResultSet resultProducts, String[] tokens, 
			PreparedStatement updatePreparedStatement, String lastUpdateUser) throws SQLException {
		String tempValue = null;
		String compareVuid = null;
        int recordsAffectedCount = 0;
        Long eplID = resultProducts.getLong(EPL_PRODUCT_ID);
		{
			//Process Record Update
			compareVuid = resultProducts.getString(EPL_PRODUCT_VUID);
		
			tempValue = trimQuotesAndSpaces(tokens[HAZARDOUS_TO_HANDLE_INX]);
			//Log.d("Hazardous to Handle: " + tempValue);
			// Set Hazardous to Handle
			updatePreparedStatement.setString(1, tempValue);  
			
			tempValue = trimQuotesAndSpaces(tokens[HAZARDOUS_TO_DISPOSE_INX]);
			//Log.d("Hazardous to Dispose: " + tempValue);
			// Set Hazardous to Handle
			updatePreparedStatement.setString(2, tempValue); 

			tempValue = trimQuotesAndSpaces(tokens[PRIMARY_EPA_CODE_INX]);
			//Log.d("EPA Code: " + tempValue);
			// Set EPA Code
			updatePreparedStatement.setString(3, tempValue);

			tempValue = trimQuotesAndSpaces(tokens[WASTE_SORT_CODE_INX]);
			//Log.d("Waste Sort Code: " + tempValue);
			// Set Waste Sort Code
			updatePreparedStatement.setString(4, tempValue);

			tempValue = trimQuotesAndSpaces(tokens[DOT_SHIPPING_NAME_INX]);
			//Log.d("DOT Shipping Name: " + tempValue);
			// Set DOT Shipping Name
			updatePreparedStatement.setString(5, tempValue);
		
			updatePreparedStatement.setString(6, lastUpdateUser);

			// Set VUID
			updatePreparedStatement.setString(7, compareVuid);

			Log.d("Updating Hazardous Waste fields for VUID: " + compareVuid);
			
        	queryPS.setString(1, compareVuid);
            ResultSet resultsDifferences = queryPS.executeQuery();
            createDifferences(lastUpdateUser, resultsDifferences, resultProducts, insertDifferencePS, insertIAPreparedStatement, insertIADetailsPreparedStatement, tokens, eplID);
			recordsAffectedCount = updatePreparedStatement.executeUpdate();
		}
		Log.d("Update Complete: " + compareVuid);
		totalRecsUpdated++;
		return recordsAffectedCount;
	}

    /**
     * Trims surrounding double-quote characters from the String and then Trims
     * leading and trailing spaces. Since the input data originally comes from
     * MS Excel, some cell values are surrounded by quotes.
     * 
     * @param token
     *            String potentially containing surrounding quotes and extra
     *            leading/trailing spaces
     * @return String with surrounding quotes and leading/trailing spaces
     *         removed
     */
    private String trimQuotesAndSpaces(String token) {
        if (StringUtils.isBlank(token)) {
            return null;
        }

        String trimmedToken = token.replaceAll("^\"|\"$", "");
        trimmedToken = trimmedToken.trim();
        return trimmedToken;
    }
    
	private static void createDifferences(String lastUpdateUser, ResultSet resultDifferences, ResultSet resultProducts, PreparedStatement insertDifferencePS, 
    		PreparedStatement insertIAPreparedStatement, PreparedStatement insertIADetails, String[] tokens, Long eplID) {
        Log.i("Running Create Differences");
        HazardousDifferenceCreator creator = new HazardousDifferenceCreator();
        try{
            creator.create(lastUpdateUser, resultProducts, resultDifferences, insertDifferencePS, insertIAPreparedStatement, insertIADetails, tokens, eplID);
        } catch (HazardousParserException e) {
            Log.e("*** ERROR *** createDifference: " + e);
        }            
    }
}
