package gov.va.med.pharmacy.hazardous.parser;

import gov.va.med.pharmacy.hazardous.db.HazardousDBManager;
import gov.va.med.pharmacy.hazardous.util.Log;
import gov.va.med.pharmacy.hazardous.util.StringUtils;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Objects;

/**
 * HazardousDifferenceCreator creates EPL_NDF_OUTGOING_DIFFERENCE, EPL_ITEM_AUDIT_HISTORY, AND EPL_ITEM_AUDIT_HISTORY_DETAILS table rows for all hazardous waste fields loaded by the 
 * Hazardous Waste Loader program.  It identifies records by the last_modified_by field = "HAZWASTELOADER" and will generate 
 * difference records AND item audit records for any of the five hazardous waste fields that are not null on those products.
 * 
 * Note that the PPS-N application normally also sends ingredients any time a product is modified, but testing concluded that 
 * this was not necessary for this program. 
 * @param <UserVo>
 */
public class HazardousDifferenceCreator<UserVo> extends HazardousDBManager {

private static final String SPACE = " ";
/*    *//** The connection properties. *//*
    private Properties connectionProperties;
*/
    private static final int HAZARDOUS_TO_HANDLE_INX = 8;
    private static final int HAZARDOUS_TO_DISPOSE_INX = 9;
    private static final int PRIMARY_EPA_CODE_INX = 10;
    private static final int WASTE_SORT_CODE_INX = 11;
    private static final int DOT_SHIPPING_NAME_INX = 12;
    private static final String NATIONAL = "999";
	int rowsInserted = 0;

    /**
     * Main method - queries for products last modified by HAZWASTELOAD and creates EPL_NDF_OUTGOING_DIFFERENCES records.
     *
     * @throws HazardousParserException the hazardous parser exception
     */
    public void create(String lastUpdateUser, ResultSet resultsProducts, ResultSet resultsDifferences, PreparedStatement insertDifferencePS, 
    		PreparedStatement insertIAPreparedStatement, PreparedStatement insertIADetails, String[] tokens, Long eplID) throws HazardousParserException {

        try {
            processSearchResults(lastUpdateUser, resultsDifferences, insertDifferencePS, insertIAPreparedStatement, insertIADetails, tokens, eplID);
        } catch (Throwable t) {
            Log.e(this.getClass(), t.toString());
            throw new HazardousParserException();
        }
    }

    /**
     * Checks if is connected to db.
     *
     * @return true, if is connected to db
     */
/*    protected boolean isConnectedToDb() {
        return super.isConnected();
    }*/

    /**
     * Connect to db.
     */
/*    protected void connectToDb() {
        super.connect(connectionProperties);
    }*/

    /**
     * Close db connection.
     */
    protected void closeDbConnection() {
        super.close();
    }

    /**
     * Process the found products, creating a difference record for each field that has a non-null value.
     * Pass the prepared statements for the ITEM_AUDIT_HISTORY and ITEM_AUDIT_HISTORY_DETAILS table insertions
     * 
     * @param lastUpdateUser
     * @param resultSet the result set containing the products modified by HAZWASTELOAD
     * @param insertDifferencePS - Prepared Statement to insert EPL_NDF_OUTGOING_DIFFERENCES records
     * @param insertIAPreparedStatement - Prepared Statement to insert ITEM_AUDIT_HISTORY records
     * @param insertIADetails - Prepared Statement to insert ITEM_AUDIT_HISTORY_DETAILS records
     * @param tokens - input fields
     * @param eplID - EPL_ID from the products table
     * @return
     */
    protected boolean processSearchResults(String lastUpdateUser, ResultSet resultSet, PreparedStatement insertDifferencePS, 
    		PreparedStatement insertIAPreparedStatement, PreparedStatement insertIADetails, String[] tokens, Long eplID) {
    	rowsInserted = 0;
    	String histEplId = null;
        try {
            if (resultSet != null) {
                while (resultSet.next()) {
					String h2hFlag = tokens[HAZARDOUS_TO_HANDLE_INX];
					String h2dFlag = tokens[HAZARDOUS_TO_DISPOSE_INX];
					String primaryEPA = tokens[PRIMARY_EPA_CODE_INX];
					String wasteSortCode = tokens[WASTE_SORT_CODE_INX];
					String dotShippingName = tokens[DOT_SHIPPING_NAME_INX]; 
					
					//create the ITEM_AUDIT_HISTORY and return the generated EPL_ID
					histEplId = createItemAuditRecords(lastUpdateUser, insertIAPreparedStatement, tokens, eplID);
					
                    //check individual fields for differences between the input record and database record
					processField(lastUpdateUser, resultSet, insertDifferencePS, insertIADetails, "HAZARDOUS_TO_HANDLE", "101", h2hFlag, histEplId);
                    processField(lastUpdateUser, resultSet, insertDifferencePS, insertIADetails, "HAZARDOUS_TO_DISPOSE", "102", h2dFlag, histEplId);
                    processField(lastUpdateUser, resultSet, insertDifferencePS, insertIADetails, "PRIMARY_EPA", "103", primaryEPA, histEplId);
                    processField(lastUpdateUser, resultSet, insertDifferencePS, insertIADetails, "WASTE_SORT_CODE", "104", wasteSortCode, histEplId);
                    processField(lastUpdateUser, resultSet, insertDifferencePS, insertIADetails, "DOT_SHIPPING_NAME", "105", dotShippingName, histEplId);
                }
            }
        } catch (SQLException e) {
            Log.e(getClass(), "SQL exception: " + e.toString());

        } catch (Throwable e) {
            Log.e(getClass(), "Exception: " + e.toString());

        } finally {
            try {
                if (resultSet != null && !resultSet.isClosed()) {
                	resultSet.close();
                }
            } catch (SQLException e) {
                Log.e(getClass(), "Error closing resultset:" + e);
            }
        }
        return true;
    }

    /**
     * Process the specified field and creates a difference record if the field has a non-null value.
     * Pass arguments to the processItemAuditDetail method to create ITEM_AUDIT_HISTORY_DETAILS records
     *
     * @param lastUpdateUser
     * @param resultSet
     * @param insertDifferencePS Prepared statement for differences update
     * @param insertIADetails Prepared statement for ITEM_AUDIT_HISTORY_DETAILS update 
     * @param fieldName field updated on the Products table
     * @param vistaFieldNumber
     * @param token
     * @param histEplId
     * @throws HazardousParserException the hazardous parser exception
     */

    private void processField(String lastUpdateUser, ResultSet resultSet, PreparedStatement insertDifferencePS, PreparedStatement insertIADetails, String fieldName, 
                String vistaFieldNumber, String token, String histEplId) throws HazardousParserException {
        try {
            String dbFieldValue = resultSet.getString(fieldName);
            String inputFieldValue = token;
            //String ienValue = null;
            
            if (!Objects.equals(inputFieldValue, dbFieldValue)) { 
            	String ndfProductIen = resultSet.getString("NDF_PRODUCT_IEN");
            	if (!StringUtils.isBlank(inputFieldValue)){
            		if (fieldName.equals("HAZARDOUS_TO_HANDLE") || fieldName.equals("HAZARDOUS_TO_DISPOSE")){
            			inputFieldValue = convertYesNoToInteger(inputFieldValue);
            		}
            	}
            	insertDifferencePS.setString(1, lastUpdateUser);
            	insertDifferencePS.setString(2, (inputFieldValue == null) ? SPACE : inputFieldValue);
            	insertDifferencePS.setString(3, (dbFieldValue == null) ? SPACE : dbFieldValue);
            	insertDifferencePS.setString(4, vistaFieldNumber);
            	insertDifferencePS.setString(5, ndfProductIen);

            	Log.d("Inserting row in Outgoing Differences File. Product IEN: " + ndfProductIen + " VistA Field: " + vistaFieldNumber + " Value:" + inputFieldValue);
            	rowsInserted = insertDifferencePS.executeUpdate();

            	processItemAuditDetail(lastUpdateUser, resultSet, insertIADetails, fieldName, inputFieldValue, dbFieldValue, histEplId); 
            } 
        } catch (SQLException e) {
            Log.e("Error encountered during Outgoing Differences insert: " + e);
        }               
    }

    /**
     * Create the ITEM_AUDIT_HISTORY record.
     * EPL_ID is generated
     * @param lastUpdateUser
     * @param insertIAPreparedStatement
     * @param tokens
     * @param eplId
     * @return
     */
    private String createItemAuditRecords(String lastUpdateUser, PreparedStatement insertIAPreparedStatement, String[] tokens, Long eplId) {
    	//int iaRowsInserted = 0; 
        String generatedId = generateItemAuditHistoryId(lastUpdateUser);
    	try {
            	insertIAPreparedStatement.setString(1, generatedId);
            	insertIAPreparedStatement.setLong(2, eplId);
            	insertIAPreparedStatement.setString(3, lastUpdateUser);
            	insertIAPreparedStatement.setString(4, lastUpdateUser);

            	Log.d("Inserting row in Item Audit Record for EPL_ID " + generatedId);
            	insertIAPreparedStatement.executeUpdate();

        } catch (SQLException e) {
            Log.e("Error encountered during Item Audit History insert: " + e);
        } 
    	return generatedId;
    }
    private void processItemAuditDetail(String lastUpdateUser, ResultSet resultSet, PreparedStatement insertIADetailsPreparedStatement, String fieldName, String newValue, String oldValue, String histEplId) {

    	try { 
    		if (oldValue == null){
    			oldValue = "No Value Specified";
    		}
    		if (newValue == null){
    			newValue = "No Value Specified";
    		}
            	insertIADetailsPreparedStatement.setString(1, fieldName);
            	insertIADetailsPreparedStatement.setString(2, oldValue);
            	insertIADetailsPreparedStatement.setString(3, newValue);
            	insertIADetailsPreparedStatement.setString(4, histEplId);
            	insertIADetailsPreparedStatement.setString(5, lastUpdateUser);
            	insertIADetailsPreparedStatement.setString(6, lastUpdateUser);

            	Log.d("Inserting row in Item Audit History Details for Item Audit: " + histEplId);
            	insertIADetailsPreparedStatement.executeUpdate();

        } catch (SQLException e) {
            Log.e("Error encountered during Item Audit History Details insert: " + e);
        } 
    }
    
    /**
     * @param user
     * @return
     */
    public String generateItemAuditHistoryId(String user) {
        String generatedId = NATIONAL + getId("EPL_ITEM_AUDIT_HISTORY", user);

        return (generatedId);
    }
    /**
     * Retrieve id for a table from ep_seq_num
     * 
     * @param tableName String
     * @param user performing the action
     * @return long
     * A possibility that this could cause data conflict if another process is procuring ID concurrently
     */
    private synchronized long getId(final String tableName, final String user) {
        long value = System.currentTimeMillis();

        long t0;
        long t1;
        t0 = System.currentTimeMillis();

        do {
            t1 = System.currentTimeMillis();
        } while (t1 - t0 < 1);

        return value;
    }
    
    /**
     * Converts Y/N to 1/0.
     * @param value
     * @return
     */
    public static String convertYesNoToInteger(String value){
        return "Y".equals(value) ? "1" : "0";
        
    }
}
