SET SERVEROUTPUT ON
SET LINESIZE 2000
SET PAGESIZE 60
SPOOL PPSN_3_migration.log

SELECT 'Migration Started: ' || CURRENT_TIMESTAMP FROM DUAL;

SET ECHO ON

REM ###########################################################################
REM #  Description: CREATE PPSN v3 DATABASE OBJECTS
REM #
REM #   Usage:
REM #   Input Parameters:
REM #   1) None
REM #   Output:
REM #   1) Terminal
REM #   Internal Script Parameters:
REM #   1) None
REM ###########################################################################
REM # Change History
REM #
REM # DATE       User Name         DESCRIPTION
REM # ---------- ----------------- -------------------------------------------
REM # 07/6	 David Lumb        Created script
REM ##########################################################################

DECLARE

   cc_schema_name   CONSTANT VARCHAR2 (35) := 'PPSNEPL';

   vc_object_name            VARCHAR2 (30);
   vc_sql_statement          VARCHAR2 (4000);
   vc_errm                   VARCHAR2 (64);
   vn_code                   NUMBER;
   vn_continue               NUMBER;
   
   vn_row_count              NUMBER;

   -----------------------------------------------------------------------------
   -- NAME:         check_for_PPSN_3
   -- TYPE:         Function
   -- DESCRIPTION:  Check to see if PPSN 3 objects already created
   --               - currently based on sequence count - only 1 in prior releases
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION check_for_PPSN_3
      RETURN NUMBER
   IS
   BEGIN
      SELECT COUNT (*)
        INTO vn_row_count
        FROM dba_sequences
       WHERE sequence_owner = cc_schema_name
          AND sequence_name IN ('EPL_ATC_CANISTERS_SEQ');

      IF vn_row_count > 1
      THEN
         RETURN (1);
      END IF;

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         DBMS_OUTPUT.put_line (
            'check_for_PPSN_3: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
         RETURN (1);
   END check_for_PPSN_3;

   -----------------------------------------------------------------------------
   -- NAME:         insert_update_EPL_IEN_GEN
   -- TYPE:         Function
   -- DESCRIPTION:  Insert/update EPL_EIN_GENERATOR based on maxval + 1 of associated table
   --               This is only used for the package sizes IEN row insert now for insert,
   --               and all others for simple reset. Note well, package sizes still not
   --               populated at this time
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
       
    FUNCTION insert_update_EPL_IEN_GEN
       RETURN NUMBER
    IS
       vc_ien_drug_unit 	NUMBER;
       vc_ien_dose_form 	NUMBER;
       vc_ien_ingredient 	NUMBER;
       vc_ien_manufacturer 	NUMBER;
       vc_ien_ndc 		NUMBER;
       vc_ien_package_type 	NUMBER;
       vc_ien_product 		NUMBER;
       vc_ien_dispense_unit 	NUMBER;
       vc_ien_drug_classes 	NUMBER;
       vc_ien_gen_name 		NUMBER;
       vc_ien_package_size 	NUMBER;

    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  insert_update_EPL_IEN_GEN'); 
    
      SELECT NVL(MAX(NDF_DRUGUNIT_IEN),1) 	INTO vc_ien_drug_unit 		FROM PPSNEPL.EPL_DRUG_UNITS ;
      SELECT NVL(MAX(DOSAGEFORM_IEN),1) 	INTO vc_ien_dose_form 		FROM PPSNEPL.EPL_DOSAGE_FORMS ;
      SELECT NVL(MAX(NDF_INGREDIENT_IEN),1) 	INTO vc_ien_ingredient 		FROM PPSNEPL.EPL_INGREDIENTS ;
      SELECT NVL(MAX(NDF_MANUFACTURER_IEN),1) 	INTO vc_ien_manufacturer 	FROM PPSNEPL.EPL_MANUFACTURERS ;
      SELECT NVL(MAX(NDF_NDC_IEN),1) 		INTO vc_ien_ndc 		FROM PPSNEPL.EPL_NDCS ;
      SELECT NVL(MAX(NDF_PACKAGETYPE_IEN),1) 	INTO vc_ien_package_type 	FROM PPSNEPL.EPL_PACKAGE_TYPES ;
      SELECT NVL(MAX(NDF_PRODUCT_IEN),1) 	INTO vc_ien_product 		FROM PPSNEPL.EPL_PRODUCTS ;
      SELECT NVL(MAX(NDF_DISPENSEUNIT_IEN),1) 	INTO vc_ien_dispense_unit 	FROM PPSNEPL.EPL_VA_DISPENSE_UNITS ;
      SELECT NVL(MAX(NDF_CLASS_IEN),1) 		INTO vc_ien_drug_classes 	FROM PPSNEPL.EPL_VA_DRUG_CLASSES ;
      SELECT NVL(MAX(NDF_GENERIC_IEN),1) 	INTO vc_ien_gen_name 		FROM PPSNEPL.EPL_VA_GEN_NAMES ;
      SELECT NVL(MAX(NDF_PACKAGE_SIZE_IEN),1) 	INTO vc_ien_package_size 	FROM PPSNEPL.EPL_PACKAGE_SIZES ;
     
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_drug_unit
         || ' WHERE CONCEPT = ''EPL_DRUG_UNITS'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
             
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_dose_form
         || ' WHERE CONCEPT = ''EPL_DOSAGE_FORMS'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
    
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = ' 
         ||vc_ien_ingredient
         || ' WHERE CONCEPT = ''EPL_INGREDIENTS'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
    
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_manufacturer
         || ' WHERE CONCEPT = ''EPL_MANUFACTURERS'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_ndc
         || ' WHERE CONCEPT = ''EPL_NDCS'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_package_type
         || ' WHERE CONCEPT = ''EPL_PACKAGE_TYPES'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_product
         || ' WHERE CONCEPT = ''EPL_PRODUCTS'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_dispense_unit
         ||' WHERE CONCEPT = ''EPL_VA_DISPENSE_UNITS'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
    
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_drug_classes
         ||' WHERE CONCEPT = ''EPL_VA_DRUG_CLASSES'' ';
         
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'UPDATE PPSNEPL.EPL_IEN_GENERATOR SET LAST_USED_IEN = '
         ||vc_ien_gen_name
         ||' WHERE CONCEPT = ''EPL_VA_GEN_NAMES'' ';

      EXECUTE IMMEDIATE vc_sql_statement;
       
      vc_sql_statement := 'INSERT INTO PPSNEPL.EPL_IEN_GENERATOR '
         || '(CONCEPT,LAST_USED_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) '
         || 'VALUES(''EPL_PACKAGE_SIZES'','||vc_ien_package_size||',''DEV'',SYSDATE,''DEV'',SYSDATE)' ;
    
      EXECUTE IMMEDIATE vc_sql_statement;
       
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  insert_update_EPL_IEN_GEN');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'insert_update_EPL_IEN_GEN: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'insert_update_EPL_IEN_GEN: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'insert_update_EPL_IEN_GEN: ' || vc_sql_statement);
         RETURN (1);
   END insert_update_EPL_IEN_GEN; 
     
     
   -----------------------------------------------------------------------------
   -- NAME:         create_PPSNEPL_UPDATE_ROLE
   -- TYPE:         Function
   -- DESCRIPTION:  Create update role
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION create_PPSNEPL_UPDATE_ROLE
      RETURN NUMBER
   IS
        
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_PPSNEPL_UPDATE_ROLE'); 
      
/***** No DROP required (doesn't exist)
      vc_sql_statement := 'DROP ROLE PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed DROP ROLE PPSNEPL_UPDATE_ROLE') ; 
*/
      
      vc_sql_statement := 'CREATE ROLE PPSNEPL_UPDATE_ROLE NOT IDENTIFIED' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE ROLE PPSNEPL_UPDATE_ROLE NOT IDENTIFIED') ; 

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_PPSNEPL_UPDATE_ROLE');

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_UPDATE_ROLE: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_UPDATE_ROLE: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_UPDATE_ROLE: ' || vc_sql_statement);
         RETURN (1);
   END create_PPSNEPL_UPDATE_ROLE;  

  
   -----------------------------------------------------------------------------
   -- NAME:         create_PPSNEPL_READ_ONLY_ROLE
   -- TYPE:         Function
   -- DESCRIPTION:  Create read only role
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION create_PPSNEPL_READ_ONLY_ROLE
      RETURN NUMBER
   IS
        
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_PPSNEPL_READ_ONLY_ROLE'); 
      
      vc_sql_statement := 'DROP ROLE PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed DROP ROLE PPSNEPL_READ_ONLY_ROLE') ; 
            
      vc_sql_statement := 'CREATE ROLE PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE ROLE PPSNEPL_READ_ONLY_ROLE NOT IDENTIFIED') ; 

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_PPSNEPL_READ_ONLY_ROLE');

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_READ_ONLY_ROLE: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_READ_ONLY_ROLE: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_READ_ONLY_ROLE: ' || vc_sql_statement);
         RETURN (1);
   END create_PPSNEPL_READ_ONLY_ROLE;  

    
   -----------------------------------------------------------------------------
   -- NAME:         create_PPSNEPL_UPDATE_USER
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION create_PPSNEPL_UPDATE_USER
      RETURN NUMBER
   IS
       
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_PPSNEPL_UPDATE_USER'); 
      
      vc_sql_statement :=
            'CREATE USER PPSNEPL_UPDATE_USER '
         || 'IDENTIFIED BY "ppsnepl_UPDATE123" '
         || 'DEFAULT TABLESPACE PPSNEPL_DATA '
         || 'TEMPORARY TABLESPACE TEMP '
         || 'PROFILE SERVICE_ACCOUNT '
         || 'ACCOUNT UNLOCK ';
	
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATED USER PPSNEPL_UPDATE_USER') ; 
 
      vc_sql_statement := 'GRANT PPSNEPL_UPDATE_ROLE TO PPSNEPL_UPDATE_USER' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT PPSNEPL_UPDATE_ROLE TO PPSNEPL_UPDATE_USER') ; 
      
      vc_sql_statement := 'GRANT PPSNEPL_READ_ONLY_ROLE TO PPSNEPL_UPDATE_USER' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT PPSNEPL_READ_ONLY_ROLE TO PPSNEPL_UPDATE_USER') ; 
      
      vc_sql_statement := 'GRANT CONNECT TO PPSNEPL_UPDATE_USER' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT CONNECT TO PPSNEPL_UPDATE_USER') ; 
      
      vc_sql_statement := 'ALTER USER PPSNEPL_UPDATE_USER DEFAULT ROLE ALL' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed ALTER USER PPSNEPL_UPDATE_USER DEFAULT ROLE ALL') ; 
      
      vc_sql_statement := 'GRANT UNLIMITED TABLESPACE TO PPSNEPL_UPDATE_USER' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT UNLIMITED TABLESPACE TO PPSNEPL_UPDATE_USER') ; 
      
      vc_sql_statement := 'ALTER USER PPSNEPL_UPDATE_USER QUOTA UNLIMITED ON PPSNEPL_DATA' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed ALTER USER PPSNEPL_UPDATE_USER QUOTA UNLIMITED ON PPSNEPL_DATA') ; 
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_PPSNEPL_UPDATE_USER');

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_UPDATE_USER: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_UPDATE_USER: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_PPSNEPL_UPDATE_USER: ' || vc_sql_statement);
         RETURN (1);
   END create_PPSNEPL_UPDATE_USER;  
      
  
   -----------------------------------------------------------------------------
   -- NAME:         EPL_ATC_CANISTERS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION create_EPL_ATC_CANISTERS
      RETURN NUMBER
   IS
      sequence_id NUMBER;
       
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_EPL_ATC_CANISTERS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_ATC_CANISTERS;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_ATC_CANISTERS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_ATC_CANISTERS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_ATC_CANISTERS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_ATC_CANISTERS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_ATC_CANISTERS_SEQ FOR PPSNEPL.EPL_ATC_CANISTERS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_ATC_CANISTERS_SEQ FOR PPSNEPL.EPL_ATC_CANISTERS_SEQ') ; 

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_ATC_CANISTERS');

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_ATC_CANISTERS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_ATC_CANISTERS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_ATC_CANISTERS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_ATC_CANISTERS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_HOSPITAL_LOCATIONS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_HOSPITAL_LOCATIONS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_HOSPITAL_LOCATIONS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_HOSPITAL_LOCATIONS ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_HOSPITAL_LOCATIONS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_HOSPITAL_LOCATIONS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_HOSPITAL_LOCATIONS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_HOSPITAL_LOCATIONS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_HOSPITAL_LOCATIONS_SEQ FOR PPSNEPL.EPL_HOSPITAL_LOCATIONS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_HOSPITAL_LOCATIONS_SEQ FOR PPSNEPL.EPL_HOSPITAL_LOCATIONS_SEQ') ; 
   
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_HOSPITAL_LOCATIONS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_HOSPITAL_LOCATIONS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_HOSPITAL_LOCATIONS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_HOSPITAL_LOCATIONS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_HOSPITAL_LOCATIONS;  
   
   -----------------------------------------------------------------------------
   -- NAME:         EPL_IFCAP_ITEM_NUMBERS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_IFCAP_ITEM_NUMBERS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_IFCAP_ITEM_NUMBERS'); 
 
      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_IFCAP_ITEM_NUMBERS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_IFCAP_ITEM_NUMBERS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_IFCAP_ITEM_NUMBERS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_IFCAP_ITEM_NUMBERS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_IFCAP_ITEM_NUMBERS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_IFCAP_ITEM_NUMBERS_SEQ FOR PPSNEPL.EPL_IFCAP_ITEM_NUMBERS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_IFCAP_ITEM_NUMBERS_SEQ FOR PPSNEPL.EPL_IFCAP_ITEM_NUMBERS_SEQ') ; 
   
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_IFCAP_ITEM_NUMBERS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_IFCAP_ITEM_NUMBERS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_IFCAP_ITEM_NUMBERS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_IFCAP_ITEM_NUMBERS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_IFCAP_ITEM_NUMBERS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_ITEM_AUDIT_HIST_DTL
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_ITEM_AUDIT_HIST_DTL
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_ITEM_AUDIT_HIST_DTL'); 

      SELECT NVL(MAX(EPL_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_ITEM_AUDIT_HISTORY_DETAILS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_ITEM_AUDIT_HIST_DTL_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_ITEM_AUDIT_HIST_DTL_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_ITEM_AUDIT_HIST_DTL_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_ITEM_AUDIT_HIST_DTL_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_ITEM_AUDIT_HIST_DTL_SEQ FOR PPSNEPL.EPL_ITEM_AUDIT_HIST_DTL_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_ITEM_AUDIT_HIST_DTL_SEQ FOR PPSNEPL.EPL_ITEM_AUDIT_HIST_DTL_SEQ') ; 
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_ITEM_AUDIT_HIST_DTL');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_ITEM_AUDIT_HIST_DTL: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_ITEM_AUDIT_HIST_DTL: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_ITEM_AUDIT_HIST_DTL: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_ITEM_AUDIT_HIST_DTL;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_LABS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_LABS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_LABS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_LABS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_LABS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_LABS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_LABS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_LABS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_LABS_SEQ FOR PPSNEPL.EPL_LABS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_LABS_SEQ FOR PPSNEPL.EPL_LABS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_LABS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_LABS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_LABS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_LABS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_LABS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_LOCAL_POSSIBLE_DOSAGES
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_LOCAL_POSS_DOSAGE
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_LOCAL_POSS_DOSAGE'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_LOCAL_POSSIBLE_DOSAGES_SEQ FOR PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_LOCAL_POSSIBLE_DOSAGES_SEQ FOR PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES_SEQ') ;  
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_LOCAL_POSS_DOSAGE');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_LOCAL_POSS_DOSAGE: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_LOCAL_POSS_DOSAGE: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_LOCAL_POSS_DOSAGE: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_LOCAL_POSS_DOSAGE;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_MED_INSTRUCT_WARDS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_MED_INSTRUCT_WARDS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
       
    BEGIN    
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_MED_INSTRUCT_WARDS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_MED_INSTRUCT_WARDS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_MED_INSTRUCT_WARDS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_MED_INSTRUCT_WARDS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_MED_INSTRUCT_WARDS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_MED_INSTRUCT_WARDS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MED_INSTRUCT_WARDS_SEQ FOR PPSNEPL.EPL_MED_INSTRUCT_WARDS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_MED_INSTRUCT_WARDS_SEQ FOR PPSNEPL.EPL_MED_INSTRUCT_WARDS_SEQ') ;  
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_MED_INSTRUCT_WARDS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_MED_INSTRUCT_WARDS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_MED_INSTRUCT_WARDS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_MED_INSTRUCT_WARDS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_MED_INSTRUCT_WARDS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_NATIONAL_POSSIBLE_DOSAGES
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_NATIONAL_POSS_DOSAG
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_NATIONAL_POSS_DOSAG'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_NATIONAL_POSSIBLE_DOSAGES ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_NATIONAL_POSSIBLE_DSGS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_NATIONAL_POSSIBLE_DSGS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_NATIONAL_POSSIBLE_DSGS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_NATIONAL_POSSIBLE_DSGS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NATIONAL_POSSIBLE_DSGS_SEQ FOR PPSNEPL.EPL_NATIONAL_POSSIBLE_DSGS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_NATIONAL_POSSIBLE_DSGS_SEQ FOR PPSNEPL.EPL_NATIONAL_POSSIBLE_DSGS_SEQ') ;  
 
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_NATIONAL_POSS_DOSAG');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NATIONAL_POSS_DOSAG: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NATIONAL_POSS_DOSAG: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NATIONAL_POSS_DOSAG: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_NATIONAL_POSS_DOSAG;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_NDC_BY_OUTPATIENT_SITE_NDC
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_NDC_BY_OTP_SITE_NDC
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_NDC_BY_OTP_SITE_NDC'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_NDC_BY_OUTPATIENT_SITE_NDC ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_NDC_BY_OUTPAT_SITE_NDC_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_NDC_BY_OUTPAT_SITE_NDC_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_NDC_BY_OUTPAT_SITE_NDC_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_NDC_BY_OUTPAT_SITE_NDC_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NDC_BY_OUTPAT_SITE_NDC_SEQ FOR PPSNEPL.EPL_NDC_BY_OUTPAT_SITE_NDC_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_NDC_BY_OUTPAT_SITE_NDC_SEQ FOR PPSNEPL.EPL_NDC_BY_OUTPAT_SITE_NDC_SEQ') ;  
   
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_NDC_BY_OTP_SITE_NDC');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NDC_BY_OTP_SITE_NDC: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NDC_BY_OTP_SITE_NDC: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NDC_BY_OTP_SITE_NDC: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_NDC_BY_OTP_SITE_NDC;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_PARTIAL_SAVE_MGT
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_PARTIAL_SAVE_MGT
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_PARTIAL_SAVE_MGT'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_PARTIAL_SAVE_MGT ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_PARTIAL_SAVE_MGT_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_PARTIAL_SAVE_MGT_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_PARTIAL_SAVE_MGT_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_PARTIAL_SAVE_MGT_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PARTIAL_SAVE_MGT_SEQ FOR PPSNEPL.EPL_PARTIAL_SAVE_MGT_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_PARTIAL_SAVE_MGT_SEQ FOR PPSNEPL.EPL_PARTIAL_SAVE_MGT_SEQ') ;  
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_PARTIAL_SAVE_MGT');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PARTIAL_SAVE_MGT: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PARTIAL_SAVE_MGT: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PARTIAL_SAVE_MGT: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_PARTIAL_SAVE_MGT;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_PRODUCT_LABS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_PRODUCT_LABS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_PRODUCT_LABS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_PRODUCT_LABS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_PRODUCT_LABS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_PRODUCT_LABS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_PRODUCT_LABS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_PRODUCT_LABS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PRODUCT_LABS_SEQ FOR PPSNEPL.EPL_PRODUCT_LABS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_PRODUCT_LABS_SEQ FOR PPSNEPL.EPL_PRODUCT_LABS_SEQ') ;  
    
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_PRODUCT_LABS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PRODUCT_LABS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PRODUCT_LABS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PRODUCT_LABS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_PRODUCT_LABS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_PRODUCT_VITALS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_PRODUCT_VITALS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_PRODUCT_VITALS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_PRODUCT_VITALS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_PRODUCT_VITALS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_PRODUCT_VITALS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_PRODUCT_VITALS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_PRODUCT_VITALS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PRODUCT_VITALS_SEQ FOR PPSNEPL.EPL_PRODUCT_VITALS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_PRODUCT_VITALS_SEQ FOR PPSNEPL.EPL_PRODUCT_VITALS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_PRODUCT_VITALS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PRODUCT_VITALS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PRODUCT_VITALS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PRODUCT_VITALS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_PRODUCT_VITALS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_REQUEST_DETAILS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_REQUEST_DETAILS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_REQUEST_DETAILS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_REQUEST_DETAILS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_REQUEST_DETAILS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_REQUEST_DETAILS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_REQUEST_DETAILS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_REQUEST_DETAILS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_REQUEST_DETAILS_SEQ FOR PPSNEPL.EPL_REQUEST_DETAILS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_REQUEST_DETAILS_SEQ FOR PPSNEPL.EPL_REQUEST_DETAILS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_REQUEST_DETAILS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REQUEST_DETAILS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REQUEST_DETAILS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REQUEST_DETAILS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_REQUEST_DETAILS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_REQUESTS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_REQUESTS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_REQUESTS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_REQUESTS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_REQUESTS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_REQUESTS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_REQUESTS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_REQUESTS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_REQUESTS_SEQ FOR PPSNEPL.EPL_REQUESTS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_REQUESTS_SEQ FOR PPSNEPL.EPL_REQUESTS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_REQUESTS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REQUESTS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REQUESTS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REQUESTS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_REQUESTS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_SITE_UPDATE_SCHEDULES
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_SITE_UPDT_SCHEDULES
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_SITE_UPDT_SCHEDULES'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_SITE_UPDATE_SCHEDULES ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_SITE_UPDATE_SCHEDULES_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_SITE_UPDATE_SCHEDULES_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_SITE_UPDATE_SCHEDULES_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_SITE_UPDATE_SCHEDULES_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SITE_UPDATE_SCHEDULES_SEQ FOR PPSNEPL.EPL_SITE_UPDATE_SCHEDULES_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_SITE_UPDATE_SCHEDULES_SEQ FOR PPSNEPL.EPL_SITE_UPDATE_SCHEDULES_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_SITE_UPDT_SCHEDULES');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_SITE_UPDT_SCHEDULES: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_SITE_UPDT_SCHEDULES: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_SITE_UPDT_SCHEDULES: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_SITE_UPDT_SCHEDULES;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_SYNONYMS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_SYNONYMS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_SYNONYMS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_SYNONYMS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_SYNONYMS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_SYNONYMS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_SYNONYMS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_SYNONYMS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SYNONYMS_SEQ FOR PPSNEPL.EPL_SYNONYMS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_SYNONYMS_SEQ FOR PPSNEPL.EPL_SYNONYMS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_SYNONYMS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_SYNONYMS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_SYNONYMS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_SYNONYMS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_SYNONYMS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_VITALS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_VITALS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_VITALS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_VITALS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_VITALS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_VITALS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_VITALS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_VITALS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VITALS_SEQ FOR PPSNEPL.EPL_VITALS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_VITALS_SEQ FOR PPSNEPL.EPL_VITALS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_VITALS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VITALS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VITALS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VITALS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_VITALS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_WARDS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_WARDS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_WARDS'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_WARDS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_WARDS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_WARDS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_WARDS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_WARDS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_WARDS_SEQ FOR PPSNEPL.EPL_WARDS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_WARDS_SEQ FOR PPSNEPL.EPL_WARDS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_WARDS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_WARDS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_WARDS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_WARDS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_WARDS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_STATUS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_STATUS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_STATUS'); 

      SELECT NVL(MAX(STATUS_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_STATUS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_STATUS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_STATUS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_STATUS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_STATUS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_STATUS_SEQ FOR PPSNEPL.EPL_STATUS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_STATUS_SEQ FOR PPSNEPL.EPL_STATUS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_STATUS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_STATUS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_STATUS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_STATUS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_STATUS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_VISTA_STATUS
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_VISTA_STATUS
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_VISTA_STATUS'); 

      SELECT NVL(MAX(VISTA_STATUS_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_VISTA_STATUS ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_VISTA_STATUS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_VISTA_STATUS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_VISTA_STATUS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_VISTA_STATUS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VISTA_STATUS_SEQ FOR PPSNEPL.EPL_VISTA_STATUS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_VISTA_STATUS_SEQ FOR PPSNEPL.EPL_VISTA_STATUS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_VISTA_STATUS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VISTA_STATUS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VISTA_STATUS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VISTA_STATUS: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_VISTA_STATUS;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_VISTA_UPDATE
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_VISTA_UPDATE
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_VISTA_UPDATE'); 

      SELECT NVL(MAX(VISTA_UPDATE_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_VISTA_UPDATE ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_VISTA_UPDATE_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_VISTA_UPDATE_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_VISTA_UPDATE_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_VISTA_UPDATE_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VISTA_UPDATE_SEQ FOR PPSNEPL.EPL_VISTA_UPDATE_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_VISTA_UPDATE_SEQ FOR PPSNEPL.EPL_VISTA_UPDATE_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_VISTA_UPDATE');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VISTA_UPDATE: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VISTA_UPDATE: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_VISTA_UPDATE: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_VISTA_UPDATE;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_DRUG_DRUG_INTERACTION
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_DRUG_DRUG_INTERACT
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_DRUG_DRUG_INTERACT'); 

      SELECT NVL(MAX(EPL_DRUG_DRUG_INTERACTION_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_DRUG_DRUG_INTERACTION ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_DRUG_DRUG_INTERACTION_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_DRUG_DRUG_INTERACTION_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_DRUG_DRUG_INTERACTION_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_DRUG_DRUG_INTERACTION_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DRUG_DRUG_INTERACTION_SEQ FOR PPSNEPL.EPL_DRUG_DRUG_INTERACTION_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_DRUG_DRUG_INTERACTION_SEQ FOR PPSNEPL.EPL_DRUG_DRUG_INTERACTION_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_DRUG_DRUG_INTERACT');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_DRUG_DRUG_INTERACT: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_DRUG_DRUG_INTERACT: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_DRUG_DRUG_INTERACT: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_DRUG_DRUG_INTERACT;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_MATCH_TYPE
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_MATCH_TYPE
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_MATCH_TYPE'); 
  
      SELECT NVL(MAX(MATCH_TYPE_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_MATCH_TYPE ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_MATCH_TYPE_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_MATCH_TYPE_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_MATCH_TYPE_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_MATCH_TYPE_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MATCH_TYPE_SEQ FOR PPSNEPL.EPL_MATCH_TYPE_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_MATCH_TYPE_SEQ FOR PPSNEPL.EPL_MATCH_TYPE_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_MATCH_TYPE');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_MATCH_TYPE: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_MATCH_TYPE: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_MATCH_TYPE: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_MATCH_TYPE;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_NDF_OUTGOING_DIFFERENCES
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_NDF_OUTGOING_DIFFER
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_EPL_NDF_OUTGOING_DIFFER'); 
 
      SELECT NVL(MAX(NDF_OUTGOING_DIFFERENCES_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_NDF_OUTGOING_DIFFERENCES ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_NDF_OUTGOING_DIFFERENS_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_NDF_OUTGOING_DIFFERENS_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_NDF_OUTGOING_DIFFERENS_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_NDF_OUTGOING_DIFFERENS_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NDF_OUTGOING_DIFFERENS_SEQ FOR PPSNEPL.EPL_NDF_OUTGOING_DIFFERENS_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_NDF_OUTGOING_DIFFERENS_SEQ FOR PPSNEPL.EPL_NDF_OUTGOING_DIFFERENS_SEQ') ;  

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_NDF_OUTGOING_DIFFER');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NDF_OUTGOING_DIFFER: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NDF_OUTGOING_DIFFER: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_NDF_OUTGOING_DIFFER: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_NDF_OUTGOING_DIFFER;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_PACKAGE_SIZES
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_PACKAGE_SIZES
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  create_EPL_PACKAGE_SIZES'); 

      SELECT NVL(MAX(EPL_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_PACKAGE_SIZES ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_PACKAGE_SIZES_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_PACKAGE_SIZES_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_PACKAGE_SIZES_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_PACKAGE_SIZES_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PACKAGE_SIZES_SEQ FOR PPSNEPL.EPL_PACKAGE_SIZES_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_PACKAGE_SIZES_SEQ FOR PPSNEPL.EPL_PACKAGE_SIZES_SEQ') ;    

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_PACKAGE_SIZES');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PACKAGE_SIZES: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PACKAGE_SIZES: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_PACKAGE_SIZES: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_PACKAGE_SIZES;  

   -----------------------------------------------------------------------------
   -- NAME:         EPL_REMATCH_SUGGESTION
   -- TYPE:         Function
   -- DESCRIPTION:  Create sequence based on maxval + 1 of associated table PK
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_EPL_REMATCH_SUGGESTION
       RETURN NUMBER
    IS
       sequence_id NUMBER;
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_EPL_REMATCH_SUGGESTION'); 

      SELECT NVL(MAX(EPL_REMATCH_SUGGESTION_ID), 0) + 1 INTO sequence_id FROM PPSNEPL.EPL_REMATCH_SUGGESTION ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'CREATE SEQUENCE PPSNEPL.EPL_REMATCH_SUGGESTION_SEQ MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH ' || sequence_id || ' NOCACHE ';
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE SEQUENCE PPSNEPL.EPL_REMATCH_SUGGESTION_SEQ INCREMENT BY 1 START WITH ' || sequence_id ) ;
      
      vc_sql_statement := 'GRANT SELECT, ALTER ON PPSNEPL.EPL_REMATCH_SUGGESTION_SEQ TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed GRANT SELECT, ALTER ON PPSNEPL.EPL_REMATCH_SUGGESTION_SEQ TO PPSNEPL_UPDATE_ROLE') ;    
      
      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_REMATCH_SUGGESTION_SEQ FOR PPSNEPL.EPL_REMATCH_SUGGESTION_SEQ' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      DBMS_OUTPUT.put_line('Executed CREATE OR REPLACE PUBLIC SYNONYM EPL_REMATCH_SUGGESTION_SEQ FOR PPSNEPL.EPL_REMATCH_SUGGESTION_SEQ') ;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_EPL_REMATCH_SUGGESTION');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REMATCH_SUGGESTION: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REMATCH_SUGGESTION: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_EPL_REMATCH_SUGGESTION: ' || vc_sql_statement);
         RETURN (1);
   END create_EPL_REMATCH_SUGGESTION; 
   
   -----------------------------------------------------------------------------
   -- NAME:         create_replace_SYNS
   -- TYPE:         Function
   -- DESCRIPTION:  Create synonyms yet required
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION create_replace_SYNS
       RETURN NUMBER
    IS
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_replace_SYNS'); 

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_ADMIN_SCHEDULES FOR PPSNEPL.EPL_ADMIN_SCHEDULES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_ATC_CANISTERS FOR PPSNEPL.EPL_ATC_CANISTERS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_CLASS_TYPES FOR PPSNEPL.EPL_CLASS_TYPES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_CLASS_TYPES_TEMP FOR PPSNEPL.EPL_CLASS_TYPES_TEMP' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_CMOP_ID_GENERATOR FOR PPSNEPL.EPL_CMOP_ID_GENERATOR' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_CMOP_ID_HISTORY FOR PPSNEPL.EPL_CMOP_ID_HISTORY' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_CS_FED_SCHEDULES FOR PPSNEPL.EPL_CS_FED_SCHEDULES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DDI_SEVERITY FOR PPSNEPL.EPL_DDI_SEVERITY' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DF_MED_RT_DF_ASSOCS FOR PPSNEPL.EPL_DF_MED_RT_DF_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DF_NOUNS FOR PPSNEPL.EPL_DF_NOUNS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DF_UNITS FOR PPSNEPL.EPL_DF_UNITS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DISPENSE_UNITS_PER_DOSE FOR PPSNEPL.EPL_DISPENSE_UNITS_PER_DOSE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DOSAGE_FORMS FOR PPSNEPL.EPL_DOSAGE_FORMS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DOSE_UNITS FOR PPSNEPL.EPL_DOSE_UNITS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DOSE_UNIT_SYNONYMS FOR PPSNEPL.EPL_DOSE_UNIT_SYNONYMS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DRUG_DRUG_INTERACTION FOR PPSNEPL.EPL_DRUG_DRUG_INTERACTION' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DRUG_TEXT FOR PPSNEPL.EPL_DRUG_TEXT' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DRUG_UNITS FOR PPSNEPL.EPL_DRUG_UNITS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_DT_SYNONYMS FOR PPSNEPL.EPL_DT_SYNONYMS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_ADD FOR PPSNEPL.EPL_FDB_ADD' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_AUTO_ADD FOR PPSNEPL.EPL_FDB_AUTO_ADD' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_AUTO_UPDATE FOR PPSNEPL.EPL_FDB_AUTO_UPDATE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_DOSAGE_FORM FOR PPSNEPL.EPL_FDB_DOSAGE_FORM' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_DRUG_CLASS FOR PPSNEPL.EPL_FDB_DRUG_CLASS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_DRUG_INGREDIENT FOR PPSNEPL.EPL_FDB_DRUG_INGREDIENT' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_DRUG_UNITS FOR PPSNEPL.EPL_FDB_DRUG_UNITS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_GENERIC_NAME FOR PPSNEPL.EPL_FDB_GENERIC_NAME' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_NDC FOR PPSNEPL.EPL_FDB_NDC' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_PRODUCT FOR PPSNEPL.EPL_FDB_PRODUCT' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_FDB_UPDATE FOR PPSNEPL.EPL_FDB_UPDATE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_HOSPITAL_LOCATIONS FOR PPSNEPL.EPL_HOSPITAL_LOCATIONS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_IEN_GENERATOR FOR PPSNEPL.EPL_IEN_GENERATOR' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_IFCAP_ITEM_NUMBERS FOR PPSNEPL.EPL_IFCAP_ITEM_NUMBERS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_INGREDIENTS FOR PPSNEPL.EPL_INGREDIENTS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_INTENDED_USES FOR PPSNEPL.EPL_INTENDED_USES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_INTERFACE_COUNTERS FOR PPSNEPL.EPL_INTERFACE_COUNTERS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_ITEM_AUDIT_HISTORY FOR PPSNEPL.EPL_ITEM_AUDIT_HISTORY' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_ITEM_AUDIT_HISTORY_DETAILS FOR PPSNEPL.EPL_ITEM_AUDIT_HISTORY_DETAILS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_LABS FOR PPSNEPL.EPL_LABS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_LOCAL_CONSOLE_INFO FOR PPSNEPL.EPL_LOCAL_CONSOLE_INFO' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_LOCAL_MED_ROUTES FOR PPSNEPL.EPL_LOCAL_MED_ROUTES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_LOCAL_POSSIBLE_DOSAGES FOR PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MANUFACTURERS FOR PPSNEPL.EPL_MANUFACTURERS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MATCH_TYPE FOR PPSNEPL.EPL_MATCH_TYPE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MEDICATION_INSTRUCTIONS FOR PPSNEPL.EPL_MEDICATION_INSTRUCTIONS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MED_INSTRUCT_WARDS FOR PPSNEPL.EPL_MED_INSTRUCT_WARDS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MIGRATION_CONTROL FOR PPSNEPL.EPL_MIGRATION_CONTROL' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MIGRATION_ERROR FOR PPSNEPL.EPL_MIGRATION_ERROR' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MIGRATION_FILE FOR PPSNEPL.EPL_MIGRATION_FILE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_MULTI_TEXT FOR PPSNEPL.EPL_MULTI_TEXT' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NATIONAL_POSSIBLE_DOSAGES FOR PPSNEPL.EPL_NATIONAL_POSSIBLE_DOSAGES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NATIONAL_SETTINGS FOR PPSNEPL.EPL_NATIONAL_SETTINGS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NDCS FOR PPSNEPL.EPL_NDCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NDC_BY_OUTPATIENT_SITE_NDC FOR PPSNEPL.EPL_NDC_BY_OUTPATIENT_SITE_NDC' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NDF_OUTGOING_DIFFERENCES FOR PPSNEPL.EPL_NDF_OUTGOING_DIFFERENCES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NDF_SYNCH_QUEUE FOR PPSNEPL.EPL_NDF_SYNCH_QUEUE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NDF_UPDATE_FILE FOR PPSNEPL.EPL_NDF_UPDATE_FILE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NOTIFICATIONS FOR PPSNEPL.EPL_NOTIFICATIONS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NOTIFICATION_STATUS FOR PPSNEPL.EPL_NOTIFICATION_STATUS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_NOTIF_USER_PREFS FOR PPSNEPL.EPL_NOTIF_USER_PREFS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_OI_ADMIN_SCHED_ASSOCS FOR PPSNEPL.EPL_OI_ADMIN_SCHED_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_OI_DRUG_TEXT_L_ASSOCS FOR PPSNEPL.EPL_OI_DRUG_TEXT_L_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_OI_DRUG_TEXT_N_ASSOCS FOR PPSNEPL.EPL_OI_DRUG_TEXT_N_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_OI_MED_ROUTE_ASSOCS FOR PPSNEPL.EPL_OI_MED_ROUTE_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_OI_SCHEDULE_TYPES FOR PPSNEPL.EPL_OI_SCHEDULE_TYPES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_ORDERABLE_ITEMS FOR PPSNEPL.EPL_ORDERABLE_ITEMS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_ORDER_UNITS FOR PPSNEPL.EPL_ORDER_UNITS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PACKAGE_SIZES FOR PPSNEPL.EPL_PACKAGE_SIZES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PACKAGE_TYPES FOR PPSNEPL.EPL_PACKAGE_TYPES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PACKAGE_USAGES FOR PPSNEPL.EPL_PACKAGE_USAGES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PACKAGE_USE_ASSOCS FOR PPSNEPL.EPL_PACKAGE_USE_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PARTIAL_SAVE_MGT FOR PPSNEPL.EPL_PARTIAL_SAVE_MGT' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PHARMACY_SYSTEM FOR PPSNEPL.EPL_PHARMACY_SYSTEM' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PRINT_FIELDS FOR PPSNEPL.EPL_PRINT_FIELDS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PRINT_TEMPLATES FOR PPSNEPL.EPL_PRINT_TEMPLATES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PRODUCTS FOR PPSNEPL.EPL_PRODUCTS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PRODUCT_LABS FOR PPSNEPL.EPL_PRODUCT_LABS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PRODUCT_VITALS FOR PPSNEPL.EPL_PRODUCT_VITALS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PROD_DRUG_CLASS_ASSOCS FOR PPSNEPL.EPL_PROD_DRUG_CLASS_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PROD_DRUG_TEXT_L_ASSOCS FOR PPSNEPL.EPL_PROD_DRUG_TEXT_L_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PROD_DRUG_TEXT_N_ASSOCS FOR PPSNEPL.EPL_PROD_DRUG_TEXT_N_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PROD_INGREDIENT_ASSOCS FOR PPSNEPL.EPL_PROD_INGREDIENT_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PROD_SPEC_HANDLING_ASSOCS FOR PPSNEPL.EPL_PROD_SPEC_HANDLING_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PROD_WARN_LABEL_L_ASSOCS FOR PPSNEPL.EPL_PROD_WARN_LABEL_L_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_PROD_WARN_LABEL_N_ASSOCS FOR PPSNEPL.EPL_PROD_WARN_LABEL_N_ASSOCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_REDUCED_COPAY FOR PPSNEPL.EPL_REDUCED_COPAY' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_REMATCH_SUGGESTION FOR PPSNEPL.EPL_REMATCH_SUGGESTION' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_REQUESTS FOR PPSNEPL.EPL_REQUESTS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_REQUEST_DETAILS FOR PPSNEPL.EPL_REQUEST_DETAILS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SCHEDULE_TYPES FOR PPSNEPL.EPL_SCHEDULE_TYPES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SEARCH_CRITERIA FOR PPSNEPL.EPL_SEARCH_CRITERIA' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SEARCH_TEMPLATES FOR PPSNEPL.EPL_SEARCH_TEMPLATES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SEQ_NUMS FOR PPSNEPL.EPL_SEQ_NUMS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SESSION_PREFERENCES FOR PPSNEPL.EPL_SESSION_PREFERENCES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SITE_CONFIGS FOR PPSNEPL.EPL_SITE_CONFIGS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SITE_UPDATE_SCHEDULES FOR PPSNEPL.EPL_SITE_UPDATE_SCHEDULES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SPECIAL_HANDLING FOR PPSNEPL.EPL_SPECIAL_HANDLING' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_STANDARD_MED_ROUTES FOR PPSNEPL.EPL_STANDARD_MED_ROUTES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_STATUS FOR PPSNEPL.EPL_STATUS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_SYNONYMS FOR PPSNEPL.EPL_SYNONYMS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_USERS FOR PPSNEPL.EPL_USERS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VADF_ASSOC_VALUES FOR PPSNEPL.EPL_VADF_ASSOC_VALUES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VADF_EDITABLE_PROPERTIES FOR PPSNEPL.EPL_VADF_EDITABLE_PROPERTIES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VADF_LOVS FOR PPSNEPL.EPL_VADF_LOVS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VADF_NONLIST_VALUES FOR PPSNEPL.EPL_VADF_NONLIST_VALUES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VADF_OWNERS FOR PPSNEPL.EPL_VADF_OWNERS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VADF_VALUES_VIEW FOR PPSNEPL.EPL_VADF_VALUES_VIEW' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VA_DFS FOR PPSNEPL.EPL_VA_DFS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VA_DISPENSE_UNITS FOR PPSNEPL.EPL_VA_DISPENSE_UNITS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VA_DRUG_CLASSES FOR PPSNEPL.EPL_VA_DRUG_CLASSES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VA_GEN_NAMES FOR PPSNEPL.EPL_VA_GEN_NAMES' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VISTA_STATUS FOR PPSNEPL.EPL_VISTA_STATUS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VISTA_UPDATE FOR PPSNEPL.EPL_VISTA_UPDATE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VITALS FOR PPSNEPL.EPL_VITALS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VUID_STATUS_HISTORY FOR PPSNEPL.EPL_VUID_STATUS_HISTORY' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_WARDS FOR PPSNEPL.EPL_WARDS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_WARN_LABELS FOR PPSNEPL.EPL_WARN_LABELS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM HIBERNATE_UNIQUE_KEY FOR PPSNEPL.HIBERNATE_UNIQUE_KEY' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM STATS_TABLE FOR PPSNEPL.STATS_TABLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM TEMP_EPL_NDCS FOR PPSNEPL.TEMP_EPL_NDCS' ;
      EXECUTE IMMEDIATE vc_sql_statement;

-- Additional missed in PPSN-2 migration script

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM UPDATE_FILE_SEQUENCE FOR PPSNEPL.UPDATE_FILE_SEQUENCE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM EPL_VADF_VALUES_VIEW FOR PPSNEPL.EPL_VADF_VALUES_VIEW' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM FDB_GCNSEQNO_PEM_V FOR PPSNEPL.FDB_GCNSEQNO_PEM_V' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM FDB_GCNSEQNO_PLBLW_V FOR PPSNEPL.FDB_GCNSEQNO_PLBLW_V' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM FDB_MONOGRAPH_PEM_V FOR PPSNEPL.FDB_MONOGRAPH_PEM_V' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'CREATE OR REPLACE PUBLIC SYNONYM FDB_PLBLWARNINGS_V FOR PPSNEPL.FDB_PLBLWARNINGS_V' ;
      EXECUTE IMMEDIATE vc_sql_statement;
     
      DBMS_OUTPUT.put_line('Executed create_replace_SYNS ...multiple') ;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_replace_SYNS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_replace_SYNS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_replace_SYNS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_replace_SYNS: ' || vc_sql_statement);
         RETURN (1);
   END create_replace_SYNS; 
   
   -----------------------------------------------------------------------------
   -- NAME:         grant_TO_ROLES
   -- TYPE:         Function
   -- DESCRIPTION:  Grants for new roles/tables etc 
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION grant_TO_ROLES
       RETURN NUMBER
    IS
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  grant_TO_ROLES'); 

-- READ_ONLY_ROLE HERE

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_ADMIN_SCHEDULES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_ATC_CANISTERS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_CLASS_TYPES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_CMOP_ID_GENERATOR TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_CMOP_ID_HISTORY TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_CS_FED_SCHEDULES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DDI_SEVERITY TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DF_MED_RT_DF_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DF_NOUNS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DF_UNITS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DISPENSE_UNITS_PER_DOSE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DOSAGE_FORMS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DOSE_UNITS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DOSE_UNIT_SYNONYMS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DRUG_DRUG_INTERACTION TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DRUG_TEXT TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DRUG_UNITS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_DT_SYNONYMS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_ADD TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_AUTO_ADD TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_AUTO_UPDATE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_DOSAGE_FORM TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_DRUG_CLASS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_DRUG_INGREDIENT TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_DRUG_UNITS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_GENERIC_NAME TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_NDC TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_PRODUCT TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_FDB_UPDATE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_HOSPITAL_LOCATIONS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_IEN_GENERATOR TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_IFCAP_ITEM_NUMBERS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_INGREDIENTS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_INTENDED_USES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_INTERFACE_COUNTERS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_ITEM_AUDIT_HISTORY TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_ITEM_AUDIT_HISTORY_DETAILS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_LABS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_LOCAL_CONSOLE_INFO TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_LOCAL_MED_ROUTES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_MANUFACTURERS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_MATCH_TYPE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_MEDICATION_INSTRUCTIONS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_MED_INSTRUCT_WARDS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_MIGRATION_CONTROL TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_MIGRATION_ERROR TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_MIGRATION_FILE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_MULTI_TEXT TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NATIONAL_POSSIBLE_DOSAGES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NATIONAL_SETTINGS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NDCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NDC_BY_OUTPATIENT_SITE_NDC TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NDF_OUTGOING_DIFFERENCES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
 
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NDF_SYNCH_QUEUE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NDF_UPDATE_FILE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NOTIFICATIONS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NOTIFICATION_STATUS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_NOTIF_USER_PREFS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_OI_ADMIN_SCHED_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_OI_DRUG_TEXT_L_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_OI_DRUG_TEXT_N_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_OI_MED_ROUTE_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_OI_SCHEDULE_TYPES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_ORDERABLE_ITEMS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_ORDER_UNITS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PACKAGE_SIZES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PACKAGE_TYPES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PACKAGE_USAGES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PACKAGE_USE_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PARTIAL_SAVE_MGT TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PHARMACY_SYSTEM TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PRINT_FIELDS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PRINT_TEMPLATES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PRODUCTS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PRODUCT_LABS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PRODUCT_VITALS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PROD_DRUG_CLASS_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PROD_DRUG_TEXT_L_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PROD_DRUG_TEXT_N_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PROD_INGREDIENT_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PROD_SPEC_HANDLING_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PROD_WARN_LABEL_L_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_PROD_WARN_LABEL_N_ASSOCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_REDUCED_COPAY TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_REMATCH_SUGGESTION TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_REQUESTS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_REQUEST_DETAILS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SCHEDULE_TYPES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SEARCH_CRITERIA TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SEARCH_TEMPLATES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SEQ_NUMS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SESSION_PREFERENCES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SITE_CONFIGS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SITE_UPDATE_SCHEDULES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SPECIAL_HANDLING TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_STANDARD_MED_ROUTES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_STATUS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_SYNONYMS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_USERS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VADF_ASSOC_VALUES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VADF_EDITABLE_PROPERTIES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VADF_LOVS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VADF_NONLIST_VALUES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VADF_OWNERS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VA_DFS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VA_DISPENSE_UNITS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VA_DRUG_CLASSES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VA_GEN_NAMES TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VISTA_STATUS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VISTA_UPDATE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VITALS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VUID_STATUS_HISTORY TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_WARDS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_WARN_LABELS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.HIBERNATE_UNIQUE_KEY TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.TEMP_EPL_NDCS TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

-- UPDATE_USER_ROLE_HERE

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_ADMIN_SCHEDULES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_ATC_CANISTERS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_CLASS_TYPES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_CMOP_ID_GENERATOR TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_CMOP_ID_HISTORY TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_CS_FED_SCHEDULES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DDI_SEVERITY TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DF_MED_RT_DF_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DF_NOUNS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DF_UNITS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DISPENSE_UNITS_PER_DOSE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DOSAGE_FORMS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DOSE_UNITS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DOSE_UNIT_SYNONYMS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DRUG_DRUG_INTERACTION TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DRUG_TEXT TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DRUG_UNITS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_DT_SYNONYMS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_ADD TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_AUTO_ADD TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_AUTO_UPDATE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_DOSAGE_FORM TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_DRUG_CLASS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_DRUG_INGREDIENT TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_DRUG_UNITS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_GENERIC_NAME TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_NDC TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_PRODUCT TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_FDB_UPDATE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_HOSPITAL_LOCATIONS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_IEN_GENERATOR TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_IFCAP_ITEM_NUMBERS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_INGREDIENTS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_INTENDED_USES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_INTERFACE_COUNTERS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_ITEM_AUDIT_HISTORY TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_ITEM_AUDIT_HISTORY_DETAILS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_LABS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_LOCAL_CONSOLE_INFO TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_LOCAL_MED_ROUTES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_LOCAL_POSSIBLE_DOSAGES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_MANUFACTURERS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_MATCH_TYPE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_MEDICATION_INSTRUCTIONS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_MED_INSTRUCT_WARDS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_MIGRATION_CONTROL TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_MIGRATION_ERROR TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_MIGRATION_FILE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_MULTI_TEXT TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NATIONAL_POSSIBLE_DOSAGES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NATIONAL_SETTINGS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NDCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NDC_BY_OUTPATIENT_SITE_NDC TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NDF_OUTGOING_DIFFERENCES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
 
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NDF_SYNCH_QUEUE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NDF_UPDATE_FILE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NOTIFICATIONS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NOTIFICATION_STATUS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_NOTIF_USER_PREFS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_OI_ADMIN_SCHED_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_OI_DRUG_TEXT_L_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_OI_DRUG_TEXT_N_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_OI_MED_ROUTE_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_OI_SCHEDULE_TYPES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_ORDERABLE_ITEMS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_ORDER_UNITS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PACKAGE_SIZES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PACKAGE_TYPES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PACKAGE_USAGES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PACKAGE_USE_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PARTIAL_SAVE_MGT TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PHARMACY_SYSTEM TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PRINT_FIELDS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PRINT_TEMPLATES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PRODUCTS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PRODUCT_LABS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PRODUCT_VITALS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PROD_DRUG_CLASS_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PROD_DRUG_TEXT_L_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PROD_DRUG_TEXT_N_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PROD_INGREDIENT_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PROD_SPEC_HANDLING_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PROD_WARN_LABEL_L_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_PROD_WARN_LABEL_N_ASSOCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_REDUCED_COPAY TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_REMATCH_SUGGESTION TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_REQUESTS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_REQUEST_DETAILS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SCHEDULE_TYPES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SEARCH_CRITERIA TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SEARCH_TEMPLATES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SEQ_NUMS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SESSION_PREFERENCES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SITE_CONFIGS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SITE_UPDATE_SCHEDULES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SPECIAL_HANDLING TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_STANDARD_MED_ROUTES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_STATUS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_SYNONYMS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_USERS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VADF_ASSOC_VALUES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VADF_EDITABLE_PROPERTIES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VADF_LOVS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VADF_NONLIST_VALUES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VADF_OWNERS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VA_DFS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VA_DISPENSE_UNITS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VA_DRUG_CLASSES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VA_GEN_NAMES TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VISTA_STATUS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VISTA_UPDATE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VITALS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_VUID_STATUS_HISTORY TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_WARDS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.EPL_WARN_LABELS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.HIBERNATE_UNIQUE_KEY TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT INSERT, UPDATE, DELETE ON PPSNEPL.TEMP_EPL_NDCS TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT ALTER, SELECT ON PPSNEPL.UPDATE_FILE_SEQUENCE TO PPSNEPL_UPDATE_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
   
-- Additional missed in PPSN-2 migration script and others for new role needs access

      vc_sql_statement := 'GRANT SELECT ON EPL_VADF_VALUES_VIEW TO PPSNEPL_READ_ONLY' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.UPDATE_FILE_SEQUENCE TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.FDB_PLBLWARNINGS_V TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.FDB_MONOGRAPH_PEM_V TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.FDB_GCNSEQNO_PLBLW_V TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.FDB_GCNSEQNO_PEM_V TO PPSNEPL_READ_ONLY_ROLE' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'GRANT SELECT ON PPSNEPL.EPL_VADF_VALUES_VIEW TO PPSNEPL_READ_ONLY_ROLE' ;      
      EXECUTE IMMEDIATE vc_sql_statement;
      
      DBMS_OUTPUT.put_line('Executed GRANT ... to ROLES ...multiple') ;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  grant_TO_ROLES');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'grant_TO_ROLES: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'grant_TO_ROLES: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'grant_TO_ROLES: ' || vc_sql_statement);
         RETURN (1);
   END grant_TO_ROLES; 
         
   -----------------------------------------------------------------------------
   -- NAME:         ALTER_EPL_NDF_OUTG_DIF
   -- TYPE:         Function
   -- DESCRIPTION:  ALTER TABLE EPL_NDF_DIFFERENCES resize 2 columns
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION alter_EPL_NDF_OUTG_DIF
       RETURN NUMBER
    IS
           
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  alter_EPL_NDF_OUTG_DIF'); 

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_NDF_OUTGOING_DIFFERENCES MODIFY OLD_VALUE VARCHAR2(4000 BYTE)' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_NDF_OUTGOING_DIFFERENCES MODIFY NEW_VALUE VARCHAR2(4000 BYTE)' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  alter_EPL_NDF_OUTG_DIF');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_NDF_OUTG_DIF: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_NDF_OUTG_DIF: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_NDF_OUTG_DIF: ' || vc_sql_statement);
         RETURN (1);
   END alter_EPL_NDF_OUTG_DIF; 
     
   -----------------------------------------------------------------------------
   -- NAME:         ALTER_EPL_VADF_LOVS
   -- TYPE:         Function
   -- DESCRIPTION:  ALTER TABLE EPL_VADF_LOVS add one column
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------

    FUNCTION alter_EPL_VADF_LOVS
       RETURN NUMBER
     IS
              
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  alter_EPL_VADF_LOVS'); 
   
      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_VADF_LOVS ADD DISPLAY_VALUE VARCHAR2(100 BYTE)' ;
      EXECUTE IMMEDIATE vc_sql_statement;
       
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  alter_EPL_VADF_LOVS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_VADF_LOVS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_VADF_LOVS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_VADF_LOVS: ' || vc_sql_statement);
         RETURN (1);
  END alter_EPL_VADF_LOVS; 
   
   -----------------------------------------------------------------------------
   -- NAME:         ALTER_EPL_PRODUCTS_RX
   -- TYPE:         Function
   -- DESCRIPTION:  ALTER TABLE EPL_PRODUCTS - RX NORM
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION alter_EPL_PRODUCTS_RX
       RETURN NUMBER
    IS
   
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  alter_EPL_PRODUCTS_RX'); 

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_PRODUCTS ADD RX_NORM_LAST_UPDATE_DATE TIMESTAMP' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_PRODUCTS ADD RX_NORM_EXCLUDE_FROM_UPDATE CHAR(1)' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_PRODUCTS MODIFY RX_NORM_NAME VARCHAR2(4000 BYTE)' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  alter_EPL_PRODUCTS_RX');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_PRODUCTS_RX: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_PRODUCTS_RX: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_PRODUCTS_RX: ' || vc_sql_statement);
         RETURN (1);
   END alter_EPL_PRODUCTS_RX; 
   
   -----------------------------------------------------------------------------
   -- NAME:         ALTER_EPL_PRODUCTS_HZ
   -- TYPE:         Function
   -- DESCRIPTION:  ALTER TABLE EPL_PRODUCTS - HAZARDOUS WASTE
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION alter_EPL_PRODUCTS_HZ
       RETURN NUMBER
    IS
      
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  alter_EPL_PRODUCTS_HZ'); 

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_PRODUCTS ADD HAZARDOUS_TO_DISPOSE CHAR(1)' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_PRODUCTS ADD HAZARDOUS_TO_HANDLE CHAR(1)' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_PRODUCTS ADD PRIMARY_EPA VARCHAR2(50 BYTE)' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_PRODUCTS ADD WASTE_SORT_CODE VARCHAR2(50 BYTE)' ;
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_PRODUCTS ADD DOT_SHIPPING_NAME VARCHAR2(200 BYTE)' ;
      EXECUTE IMMEDIATE vc_sql_statement;
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  alter_EPL_PRODUCTS_HZ');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_PRODUCTS_HZ: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_PRODUCTS_HZ: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'alter_EPL_PRODUCTS_HZ: ' || vc_sql_statement);
         RETURN (1);
   END alter_EPL_PRODUCTS_HZ; 

   -----------------------------------------------------------------------------
   -- NAME:         INSERT_EPL_VA_DFS_N_LOVS_RX
   -- TYPE:         Function
   -- DESCRIPTION:  INSERT INTO TABLE EPL_VA_DFS and EPL_VA_LOVS - SUPPRESS DML
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION insert_EPL_VA_DFS_N_LOVS_RX
       RETURN NUMBER
    IS

    max_id_dfs_plus_1 NUMBER;
    max_id_lovs       NUMBER;
        
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  insert_EPL_VA_DFS_N_LOVS_RX'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO max_id_dfs_plus_1 FROM PPSNEPL.EPL_VA_DFS;
      
      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_VA_DFS '
      || '(ID,VADF_NAME,VADF_TYPE,MULTI_SELECT_YN,DEFAULT_VALUE,VADF_USAGES,CREATED_BY,CREATED_DTM) '
      || 'VALUES ('||max_id_dfs_plus_1||',''rx.norm.suppress'',''LIST'',''N'',NULL,''NPO'',''DEVELOPER'',SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      SELECT ID INTO max_id_lovs FROM PPSNEPL.EPL_VA_DFS WHERE VADF_NAME = 'rx.norm.suppress';

      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_VADF_LOVS '
      || '(VADF_ID_FK,LIST_VALUE,DISPLAY_VALUE,DEFAULT_VALUE,CREATED_BY,CREATED_DTM) '
	  || 'VALUES ('||max_id_lovs||',''N'', ''N-Not Suppressible'', ''N'', ''DEVELOPER'', SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_VADF_LOVS '
      || '(VADF_ID_FK,LIST_VALUE,DISPLAY_VALUE,DEFAULT_VALUE,CREATED_BY,CREATED_DTM) '
	  || 'VALUES ('||max_id_lovs||',''O'', ''O-Obsolete'', ''N'', ''DEVELOPER'', SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_VADF_LOVS '
      || '(VADF_ID_FK,LIST_VALUE,DISPLAY_VALUE,DEFAULT_VALUE,CREATED_BY,CREATED_DTM) '
	  || 'VALUES ('||max_id_lovs||',''Y'', ''Y-Suppressed'', ''N'', ''DEVELOPER'', SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
          
      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_VADF_LOVS '
      || '(VADF_ID_FK,LIST_VALUE,DISPLAY_VALUE,DEFAULT_VALUE,CREATED_BY,CREATED_DTM) '
	  || 'VALUES ('||max_id_lovs||',''E'', ''E-Editor-Assigned Suppressiblity'', ''N'', ''DEVELOPER'', SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  insert_EPL_VA_DFS_N_LOVS_RX');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VA_DFS_N_LOVS_RX: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VA_DFS_N_LOVS_RX: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VA_DFS_N_LOVS_RX: ' || vc_sql_statement);
         RETURN (1);
   END insert_EPL_VA_DFS_N_LOVS_RX; 
          

   -----------------------------------------------------------------------------
   -- NAME:         INSERT_EPL_VA_DFS_HZ
   -- TYPE:         Function
   -- DESCRIPTION:  INSERT INTO TABLE EPL_VA_DFS - HAZARDOUS WASTE DML
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION insert_EPL_VA_DFS_HZ
       RETURN NUMBER
    IS
 	   
 	   max_id_dfs_hz_1   NUMBER;
 	   max_id_dfs_hz_2   NUMBER;
 	   max_id_dfs_hz_3   NUMBER; 	    	   
 	   
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  insert_EPL_VA_DFS_HZ'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO max_id_dfs_hz_1 FROM PPSNEPL.EPL_VA_DFS;

      SELECT NVL(MAX(ID), 0) + 2 INTO max_id_dfs_hz_2 FROM PPSNEPL.EPL_VA_DFS;

      SELECT NVL(MAX(ID), 0) + 3 INTO max_id_dfs_hz_3 FROM PPSNEPL.EPL_VA_DFS;
                  
      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_VA_DFS '
      || '(ID,VADF_NAME,VADF_TYPE,MULTI_SELECT_YN,DEFAULT_VALUE,VADF_USAGES,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) '
	  || 'VALUES ('||max_id_dfs_hz_1||',''primary.epa'',''STRING'',''N'',NULL,''P'',''DEVELOPER'',SYSDATE,NULL,SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_VA_DFS '
      || '(ID,VADF_NAME,VADF_TYPE,MULTI_SELECT_YN,DEFAULT_VALUE,VADF_USAGES,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) '
	  || 'VALUES ('||max_id_dfs_hz_2||',''waste.sort.code'',''STRING'',''N'',NULL,''P'',''DEVELOPER'',SYSDATE,NULL,SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_VA_DFS '
      || '(ID,VADF_NAME,VADF_TYPE,MULTI_SELECT_YN,DEFAULT_VALUE,VADF_USAGES,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) '
	  || 'VALUES ('||max_id_dfs_hz_3||',''dot.shipping.name'',''STRING'',''N'',NULL,''P'',''DEVELOPER'',SYSDATE,NULL,SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  insert_EPL_VA_DFS_HZ');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VA_DFS_HZ: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VA_DFS_HZ: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VA_DFS_HZ: ' || vc_sql_statement);
         RETURN (1);
   END insert_EPL_VA_DFS_HZ; 

   -----------------------------------------------------------------------------
   -- NAME:         INSERT_EPL_NAT_SET_RX
   -- TYPE:         Function
   -- DESCRIPTION:  INSERT INTO TABLE EPL_NATIONAL_SETTINGS - RX NORM
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION insert_EPL_NAT_SET_RX
       RETURN NUMBER
    IS
       max_id_epl_1 NUMBER;
       max_id_epl_2 NUMBER;       
    
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  insert_EPL_NAT_SET_RX'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO max_id_epl_1 FROM PPSNEPL.EPL_NATIONAL_SETTINGS;
      
      SELECT NVL(MAX(ID), 0) + 2 INTO max_id_epl_2 FROM PPSNEPL.EPL_NATIONAL_SETTINGS;
      
      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_NATIONAL_SETTINGS (ID, KEY_NAME, CREATED_BY, CREATED_DTM) '
	  || 'VALUES ('||max_id_epl_1||',''RXNORM_UPDATE_RUN_STATE'',''DEVELOPER'',SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
   
      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_NATIONAL_SETTINGS (ID, KEY_NAME, CREATED_BY, CREATED_DTM) '
      || 'VALUES ('||max_id_epl_2||',''RXNORM_UPDATE_LAST_RUN'',''DEVELOPER'',SYSDATE)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  insert_EPL_NAT_SET_RX');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_NAT_SET_RX: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_NAT_SET_RX: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_NAT_SET_RX: ' || vc_sql_statement);
         RETURN (1);
   END insert_EPL_NAT_SET_RX; 

   -----------------------------------------------------------------------------
   -- NAME:         update_EPL_STATUS
   -- TYPE:         Function
   -- DESCRIPTION:  UPDATE TABLE EPL_STATUS - RENAMED STATUSES - updated later by Armando, re-included...
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION update_EPL_STATUS
       RETURN NUMBER
    IS

	   epl_timestamp   TIMESTAMP;
	   	 
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  update_EPL_STATUS'); 

      SELECT CURRENT_TIMESTAMP INTO epl_timestamp FROM DUAL;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_STATUS SET STATUS_NAME  = ''PPS-N/NDF Update Test File Transmitted to the test sFTP'', '
      || 'STATUS_DESC = ''The Test VistA instance has processed the PPS-N/NDF Update Test File and is ready for testing. A user may now access the Update approval functionality of PPS-N'' '
	  || 'WHERE STATUS_ID = 3';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_STATUS SET STATUS_NAME  = ''PPS-N/NDF Update Test File Install Successful'', '
      || 'STATUS_DESC = ''The Test VistA instance has processed the PPS-N/NDF Update Test File and is ready for testing. A user may now access the Update approval functionality of PPS-N'' '
	  || 'WHERE STATUS_ID = 5';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_STATUS SET STATUS_NAME  = ''PPS-N/NDF Update Test File Approved By PBM'', '
      || 'STATUS_DESC = ''The PPS-N/NDF Update Test File has been approved by PBM in the update management area of PPS-N'' '
	  || 'WHERE STATUS_ID = 6';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_STATUS SET STATUS_NAME  = ''PPS-N/NDF UPDATE Test File Rejected By PBM'', '
      || 'STATUS_DESC = ''The PPS-N/NDF Update Test File has been approved by PBM in the update management area of PPS-N'' '
	  || 'WHERE STATUS_ID = 7';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_STATUS SET '
      || 'STATUS_DESC = ''Transmission of the approved file to the production SFTP location has been completed'' '
	  || 'WHERE STATUS_ID = 8';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_STATUS SET '
      || 'STATUS_DESC = ''NDF Update Process Error in PPS-N'' '
	  || 'WHERE STATUS_ID = 99';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
	  
      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_STATUS '
      || 'VALUES (98,''VistA Error'',''An error happened in VistA while the File Update was being processed.'',''system'','''||epl_timestamp||''',NULL,NULL) ';
        
      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  update_EPL_STATUS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'update_EPL_STATUS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'update_EPL_STATUS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'update_EPL_STATUS: ' || vc_sql_statement);
         RETURN (1);
   END update_EPL_STATUS; 
      
   -----------------------------------------------------------------------------
   -- NAME:         drop_create_EPL_CLASS_TYPES
   -- TYPE:         Function
   -- DESCRIPTION:  DROP and CREATE revised EPL_CLASS_TYPES tables for code/desc split for VistA
   --               Table is expected to exist and needs drop/replace
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION drop_create_EPL_CLASS_TYPES
      RETURN NUMBER
   IS

   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  drop_create_EPL_CLASS_TYPES'); 

      vc_sql_statement := 'ALTER TABLE PPSNEPL.EPL_CLASS_TYPES DROP PRIMARY KEY CASCADE ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'DROP TABLE PPSNEPL.EPL_CLASS_TYPES CASCADE CONSTRAINTS ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;  
      
      vc_sql_statement :=
            'CREATE TABLE PPSNEPL.EPL_CLASS_TYPES( '
         || 'ID                 NUMBER(30)                 NOT NULL, '
         || 'CLASS_TYPE_CODE    VARCHAR2(1)                NOT NULL, '
         || 'CLASS_TYPE_DESC    VARCHAR2(50 BYTE)          NOT NULL, '
         || 'CREATED_BY         VARCHAR2(50 BYTE)          NOT NULL, '
         || 'CREATED_DTM        TIMESTAMP(6)               NOT NULL, '
         || 'LAST_MODIFIED_BY   VARCHAR2(50 BYTE), '
         || 'LAST_MODIFIED_DTM  TIMESTAMP(6) )'
	 || 'TABLESPACE PPSNEPL_DATA ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
            'ALTER TABLE PPSNEPL.EPL_CLASS_TYPES '
         || 'ADD (CONSTRAINT EPL_CLASS_TYPES_PK '
	 || 'PRIMARY KEY (ID) '
	 || 'USING INDEX '
	 || 'ENABLE VALIDATE)';
      
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
            'GRANT INSERT, UPDATE, DELETE '
         || 'ON PPSNEPL.EPL_CLASS_TYPES '
         || 'TO PPSNEPL_UPDATE_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;
        
      vc_sql_statement :=  
	    'GRANT SELECT '
	 || 'ON PPSNEPL.EPL_CLASS_TYPES '
	 || 'TO PPSNEPL_READ_ONLY_ROLE';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
	    'CREATE OR REPLACE PUBLIC SYNONYM '
	 || 'EPL_CLASS_TYPES '
	 || 'FOR PPSNEPL.EPL_CLASS_TYPES';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  drop_create_EPL_CLASS_TYPES');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'drop_create_EPL_CLASS_TYPES: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'drop_create_EPL_CLASS_TYPES: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'drop_create_EPL_CLASS_TYPES: ' || vc_sql_statement);
         RETURN (1);
   END drop_create_EPL_CLASS_TYPES;   

   -----------------------------------------------------------------------------
   -- NAME:         INSERT_EPL_CLASS_TYPES
   -- TYPE:         Function
   -- DESCRIPTION:  INSERT INTO TABLE EPL_CLASS_TYPES - REDEFINED TABLE FOR VistA code/desc split DML
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION insert_EPL_CLASS_TYPES
       RETURN NUMBER
    IS

    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  insert_EPL_CLASS_TYPES'); 

      vc_sql_statement :=
         'INSERT INTO EPL_CLASS_TYPES '
      || '(id, class_type_code, class_type_desc, created_by, created_dtm, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
	  || 'VALUES (1,''0'',''0-Major'',''DEVELOPER'',SYSDATE,NULL,NULL)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'INSERT INTO EPL_CLASS_TYPES '
      || '(id, class_type_code, class_type_desc, created_by, created_dtm, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
	  || 'VALUES (2,''1'',''1-Minor'',''DEVELOPER'',SYSDATE,NULL,NULL)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'INSERT INTO EPL_CLASS_TYPES '
      || '(id, class_type_code, class_type_desc, created_by, created_dtm, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
	  || 'VALUES (3,''2'',''2-Sub-Class'',''DEVELOPER'',SYSDATE,NULL,NULL)';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  insert_EPL_CLASS_TYPES');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_CLASS_TYPES: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_CLASS_TYPES: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_CLASS_TYPES: ' || vc_sql_statement);
         RETURN (1);
   END insert_EPL_CLASS_TYPES; 

   -----------------------------------------------------------------------------
   -- NAME:         delete_update_EPL_NAT_SET_SFTP
   -- TYPE:         Function
   -- DESCRIPTION:  delete and UPDATE NATIONAL SETTINGS for Secure FTP 
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION delete_update_EPL_NAT_SET_SFTP
       RETURN NUMBER
    IS

    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  delete_update_EPL_NAT_SET_SFTP'); 

      vc_sql_statement :=
         'DELETE FROM EPL_NATIONAL_SETTINGS '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_PROD_PROTOCOL'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'DELETE FROM EPL_NATIONAL_SETTINGS '
      || 'WHERE KEY_NAME = ''NDF_OUTPUT_TST_PROTOCOL'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE EPL_NATIONAL_SETTINGS SET STRING_VALUE = ''vaauspresftp01.aac.domain'' '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_PROD_HOSTNAME'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE EPL_NATIONAL_SETTINGS SET STRING_VALUE = ''vaauspresftp01.aac.domain'' '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_TST_HOSTNAME'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE EPL_NATIONAL_SETTINGS SET STRING_VALUE = ''22'' '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_PROD_PORT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE EPL_NATIONAL_SETTINGS SET STRING_VALUE = ''22'' '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_TST_PORT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE EPL_NATIONAL_SETTINGS SET STRING_VALUE = ''presftp'' '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_PROD_USERNAME'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE EPL_NATIONAL_SETTINGS SET STRING_VALUE = ''presftp'' '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_TST_USERNAME'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE EPL_NATIONAL_SETTINGS SET STRING_VALUE = ''Im@g!N3'' '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_PROD_PASSWORD'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE EPL_NATIONAL_SETTINGS SET STRING_VALUE = ''Im@g!N3'' '
	  || 'WHERE KEY_NAME = ''NDF_OUTPUT_TST_PASSWORD'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
            
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  delete_update_EPL_NAT_SET_SFTP');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'delete_update_EPL_NAT_SET_SFTP: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'delete_update_EPL_NAT_SET_SFTP: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'delete_update_EPL_NAT_SET_SFTP: ' || vc_sql_statement);
         RETURN (1);
   END delete_update_EPL_NAT_SET_SFTP; 

   -----------------------------------------------------------------------------
   -- NAME:         update_EPL_NAT_SET_RPT_MSG
   -- TYPE:         Function
   -- DESCRIPTION:  delete and UPDATE NATIONAL SETTINGS for Secure FTP 
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION update_EPL_NAT_SET_RPT_MSG
       RETURN NUMBER
    IS

       max_id_nat_set 	NUMBER;
       
    BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  update_EPL_NAT_SET_RPT_MSG'); 

      SELECT NVL(MAX(ID), 0) + 1 INTO max_id_nat_set FROM PPSNEPL.EPL_NATIONAL_SETTINGS;
     
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following interactions have been added, edited or inactivated.|These changes are the result of review and recommendations from the|NDF support group.'' ' 
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE2_DDI_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) have been flagged for exclusion from|drug-drug interaction checks.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE2_EXCLUEDEDDDI_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	   || '''The FDA Med Guide for the following VA Product(s) has been changed.|'' '
	   || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	   || 'WHERE KEY_NAME = ''MESSAGE2_MEDGUIDE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s), previously flagged for exclusion from|drug-drug interaction checks, have been changed to be included in|drug-drug interaction checks.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE2_PREVEXCLUEDEDDDI_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) have been added to the National Drug File.|You may wish to review, then match or unmatch local DRUG file (#50)|entries based on this updated information.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_ADDEDPRODUCT_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product Identifier(s) (CMOP ID) have been added or|edited.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_CMOPCHANGE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Dispense Unit(s) have been added or edited.|'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_DISPENSEUNITCHANGE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Dosage Form(s) have been edited.|'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_DOSAGEFORMCHANGE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
 	  || '''Unit(s) have been added or edited for the following VA Product(s).|'' '
 	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
 	  || 'WHERE KEY_NAME = ''MESSAGE_DRUGUNITCHANGE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) WILL be inactivated on the date listed. No|alternative VA Product(s) have been found.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_FUTUREINACTIVE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Generic Name(s) have been added or edited.|'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_GENERICNAMECHANGE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following active VA Product(s) are no longer marked for CMOP. All|local DRUG file (#50) entries matched to the VA Product(s) will be|UNMARKED for CMOP. In order to have these entries dispensed by CMOP,|any local DRUG file (#50) entries matched to these products must be|re-matched to another VA product that is marked for CMOP dispensing.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_INACTIVATEDCMOP_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) have been inactivated.|No alternative VA Product(s) have been found.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_INACTIVATEDPRODUCT_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) have been inactivated. All local DRUG|file (#50) entries matched to the VA Product(s) will be unmatched and|automatically rematched to the VA Product listed.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_INACTIVATEDREMATCHED_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	 || '''The following VA Product(s) have been inactivated. All local DRUG|file (#50) entries matched to the VA Products will be unmatched.|In order to continue to use the product(s), it is suggested the local|site rematch the local DRUG file (#50) entry(ies) to the listed|VA Product.'' '
	 || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	 || 'WHERE KEY_NAME = ''MESSAGE_INACTIVATEDWSUGG_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	 || '''The National Formulary Indicator has changed for the following|VA Products. The National Formulary Indicator will automatically be|changed in your local DRUG file (#50). Please review the local|DRUG file (#50) Formulary designations of these products and make|appropriate changes.'' '
	 || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	 || 'WHERE KEY_NAME = ''MESSAGE_NATIONALFORMULARY_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) have been edited. All local DRUG file|(#50) entries matched to the VA Products will be unmatched. No|alternative VA Product(s) have been found.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_OTHERNOREMATCH_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) have been edited. All local DRUG file|(#50) entries matched to the VA Product(s) will be unmatched and|automatically rematched to the VA Product listed.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_OTHERREMATCHED_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) have been edited. All local DRUG file|(#50) entries matched to the VA Products will be unmatched. In order|to continue to use the product(s), it is suggested the local site|rematch the local DRUG file (#50) entry(ies) to the listed VA Product.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_OTHERREMATCHSUGG_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
            
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The OVERRIDE DF DOSE CHK EXCLUSION(#31) field in the VA PRODUCT|file(#50.68) has changed for the following VA Products.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_OVERRIDEDOSECHECK_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The Auto-Create Possible Dosages settings have been edited for the|following VA Product(s). Please review your local dosages for|products matched to these entries. Edits to your site''''s possible|dosages or local possible dosages may be needed.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_POSSIBLEDOSAGE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Print Name(s) have been added or edited.|'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_PRINTNAMECHANGE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) WILL be inactivated on the date listed.|All local DRUG file (#50) entries matched to the VA Products will be|unmatched once the product is inactivated. In order to continue to|use the product(s), it is suggested the local site rematch the local|DRUG file (#50) entry(ies) to the listed VA Product.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_PROPOSEDINACTIVATEDWSUGG_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Product(s) have been reactivated.|'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_REACTIVATEDPRODUCT_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''CS Federal Schedule(s) have been added or edited for the following|VA Product(s). Please review the local DEA special handling field|and make required edits.'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_SCHEDULECHANGEALL_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''Strength(s) have been added or edited for the following VA Product(s).|'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_STRENGTH_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'UPDATE PPSNEPL.EPL_NATIONAL_SETTINGS SET STRING_VALUE = '
	  || '''The following VA Drug Class(es) have been added or edited. The VA Class|for this VA Product will be automatically updated in the local|DRUG file (#50).'' '
	  || ',LAST_MODIFIED_BY=''DEVELOPER'',LAST_MODIFIED_DTM=SYSDATE ' 
	  || 'WHERE KEY_NAME = ''MESSAGE_VACLASSCHANGE_TEXT'' ';
	 
      EXECUTE IMMEDIATE vc_sql_statement;
      
      vc_sql_statement :=
         'INSERT INTO PPSNEPL.EPL_NATIONAL_SETTINGS (ID,KEY_NAME,STRING_VALUE,CREATED_BY,CREATED_DTM) '
      || 'VALUES ('||max_id_nat_set||',''MESSAGE_NEWVADRUGCLASS_TEXT'',''The following VA Drug Class(es) have been added.'',''DEVELOPER'',SYSDATE)';
   	 	 
      EXECUTE IMMEDIATE vc_sql_statement;
                  
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  update_EPL_NAT_SET_RPT_MSG');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'update_EPL_NAT_SET_RPT_MSG: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'update_EPL_NAT_SET_RPT_MSG: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'update_EPL_NAT_SET_RPT_MSG: ' || vc_sql_statement);
         RETURN (1);
   END update_EPL_NAT_SET_RPT_MSG; 

   -----------------------------------------------------------------------------
   -- NAME:         insert_EPL_VISTA_STATUS
   -- TYPE:         Function
   -- DESCRIPTION:  UPDATE TABLE EPL_VISTA_STATUS - by Armando...
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
    FUNCTION insert_EPL_VISTA_STATUS
       RETURN NUMBER
    IS
	   
	   eplv_timestamp   TIMESTAMP;
	   
    BEGIN
        
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  insert_EPL_VISTA_STATUS'); 

      SELECT CURRENT_TIMESTAMP INTO eplv_timestamp FROM DUAL;
      
      vc_sql_statement :=
            'INSERT INTO PPSNEPL.EPL_VISTA_STATUS '
         || 'VALUES (98,''ERROR'',''Vista Error'',''system'','''||eplv_timestamp||''',''system'','''||eplv_timestamp||''')';
         
      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  insert_EPL_VISTA_STATUS');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VISTA_STATUS: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VISTA_STATUS: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'insert_EPL_VISTA_STATUS: ' || vc_sql_statement);
         RETURN (1);
   END insert_EPL_VISTA_STATUS; 
         
   
-----------------------------------------------------------------------------
-- MAIN SECTION ***
-----------------------------------------------------------------------------
BEGIN
   sys.DBMS_OUTPUT.put_line (CHR (10));
   sys.DBMS_OUTPUT.put_line (
      'INFORMATION -   Checking to see if PPSN 3 objects already exist.');

   vn_continue := check_for_PPSN_3;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -     ONE OR MORE PPSN SEQUENCES ALREADY EXIST. STOPPING MIGRATION.');
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSN 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_PPSNEPL_UPDATE_ROLE;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_PPSNEPL_READ_ONLY_ROLE;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
   
   
   vn_continue := create_PPSNEPL_UPDATE_USER;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := grant_TO_ROLES;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

/****************************************
   vn_continue := create_PUBLIC_SYNONYMS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
****************************************/

   vn_continue := create_EPL_ATC_CANISTERS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_HOSPITAL_LOCATIONS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_IFCAP_ITEM_NUMBERS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_ITEM_AUDIT_HIST_DTL;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_LABS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_LOCAL_POSS_DOSAGE;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_MED_INSTRUCT_WARDS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_NATIONAL_POSS_DOSAG;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_NDC_BY_OTP_SITE_NDC;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_PARTIAL_SAVE_MGT;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_PRODUCT_LABS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_PRODUCT_VITALS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_REQUEST_DETAILS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_REQUESTS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_SITE_UPDT_SCHEDULES;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_SYNONYMS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_VITALS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_WARDS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_STATUS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_VISTA_STATUS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_VISTA_UPDATE;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_DRUG_DRUG_INTERACT;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_MATCH_TYPE;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_NDF_OUTGOING_DIFFER;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_PACKAGE_SIZES;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := create_EPL_REMATCH_SUGGESTION;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
   

   vn_continue := insert_update_EPL_IEN_GEN;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
   

   vn_continue := create_replace_SYNS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;


   vn_continue := grant_TO_ROLES;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
   
   
   vn_continue := alter_EPL_NDF_OUTG_DIF;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;   
   
   vn_continue := alter_EPL_VADF_LOVS;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;   
  
   vn_continue := alter_EPL_PRODUCTS_RX;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;   
   
   vn_continue := alter_EPL_PRODUCTS_HZ;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;   
   
   vn_continue := insert_EPL_VA_DFS_N_LOVS_RX;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;   
   
   vn_continue := insert_EPL_VA_DFS_HZ;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;   

   vn_continue := insert_EPL_NAT_SET_RX;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;      
   
   vn_continue := update_EPL_STATUS;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;      
   
   vn_continue := drop_create_EPL_CLASS_TYPES;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;      

   vn_continue := insert_EPL_CLASS_TYPES;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;      
   
   vn_continue := delete_update_EPL_NAT_SET_SFTP;
   
   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
   
   vn_continue := update_EPL_NAT_SET_RPT_MSG;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
   
   vn_continue := create_replace_SYNS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
  
   vn_continue := insert_EPL_VISTA_STATUS;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPSNEPL 3 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
         
      -- Purge recycle bin
      
      vc_sql_statement := 'purge RECYCLEBIN';
      COMMIT;
   
      EXECUTE IMMEDIATE vc_sql_statement;
   
      -- Recompile Schema and Gather Statistics
      
      DBMS_UTILITY.compile_schema (schema => 'PPSNEPL');
      DBMS_STATS.gather_schema_stats ('PPSNEPL');
      
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'MAIN: Error code ' || vn_code || ': ' || vc_errm);
   END;
   /
      
   SELECT 'Migration Stopped: ' || CURRENT_TIMESTAMP FROM DUAL;
   
   SPOOL OFF
EXIT