package gov.va.med.mhv.vitals.web.controller;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.vitals.dto.PulseOximetryReadingDTO;
import gov.va.med.mhv.vitals.enums.PeriodEnumeration;
import gov.va.med.mhv.vitals.util.CommonUtility;
import gov.va.med.mhv.vitals.web.util.WebUtility;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.data.time.Day;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;
import org.primefaces.component.datatable.DataTable;
import org.primefaces.event.data.SortEvent;
import org.primefaces.model.StreamedContent;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

@ManagedBean
@Component
@Scope("session")
public class PulseOximetryController extends AbstractController {

	private static final long serialVersionUID = -7820928514536173243L;

	private static Logger log = LogManager.getLogger(PulseOximetryController.class);
	private List<PulseOximetryReadingDTO> poreadings = new ArrayList<PulseOximetryReadingDTO>();
	private DataTable poTable;
	private PulseOximetryReadingDTO selectedPOReading = new PulseOximetryReadingDTO();
	private PulseOximetryReadingDTO newPOReading = new PulseOximetryReadingDTO();
	private static final String TITLE = "Pulse Oximetry";
	private static final String XAXIS = "Date";
	private static final String YAXIS = "percentage";

	public void init(ComponentSystemEvent event) {
		findUser();
		userprofileId = getUserProfileId();
		poTable = (DataTable) FacesContext.getCurrentInstance().getViewRoot().findComponent("poForm:poList");
		if (!FacesContext.getCurrentInstance().isPostback()) {
			resetMessages();
			if (userprofileId != null) {
				poreadings = findPOReadings(userprofileId);
			}
			
			setRowsPerPage(10);
			setPeriod(PeriodEnumeration.ONE_MONTH.getDescription());
		}
		else{
			if(sortColumn != null && sortBy != null){
				poTable.setValueExpression("sortBy", sortColumn);
				poTable.setSortOrder(sortBy);
			} 
		}
	}
		
	public void onSort(SortEvent event){
		sortColumn=event.getSortColumn().getValueExpression("sortBy");
		sortBy=event.isAscending()?"ascending":"descending";
	}
		

	private List<PulseOximetryReadingDTO> findPOReadings(Long userprofileId) {
		List<PulseOximetryReadingDTO> dtoList = null;
		try {

			dtoList = this.vitalSignsService.getPulseOximetryReadingsForUser(userprofileId);

		} catch (Exception e) {
			log.error("Error in Find Pulse oximetry readings:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return dtoList;
	}

	public String showDetail(PulseOximetryReadingDTO pulseOximetryReadingDTO) {
		resetMessages();
		selectedPOReading = pulseOximetryReadingDTO;
		setHourMinute(selectedPOReading);
		return "pulseOximetry";
	}
	
	public String editDisplay(PulseOximetryReadingDTO pulseOximetryReadingDTO) {
		resetMessages();
		selectedPOReading = pulseOximetryReadingDTO;
		setHourMinute(selectedPOReading);
		return "editPODisplay";
	}
	
	public String resetEditDisplay(){
		return "editPODisplay";
	}

	public String deleteDisplay(PulseOximetryReadingDTO pulseOximetryReadingDTO) {
		resetMessages();
		selectedPOReading = pulseOximetryReadingDTO;
		setHourMinute(selectedPOReading);
		setDeleteOrigin("tableView");
		return "deletePODisplay";
	}

	public String deleteRecordDisplay() {
		resetMessages();
		setDeleteOrigin(null);
		return "deletePODisplay";
	}

	private void setHourMinute(PulseOximetryReadingDTO selectedPOReading) {
		String dateTime = WebUtility.dateToString(selectedPOReading.getReading(), HOUR_MINUTE);
		if (!dateTime.equals(DEFAULT_DATETIME)) {
			selectedPOReading.setHour(WebUtility.dateToString(selectedPOReading.getReading(), HOURS));
			selectedPOReading.setMinute(WebUtility.dateToString(selectedPOReading.getReading(), MINUTES));
		}

	}

	public String addDisplay() {
		resetMessages();
		newPOReading = new PulseOximetryReadingDTO();
		newPOReading.setReading(new Date());
		return "addPODisplay";
	}

	public String dashboardAddDisplay() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			poreadings = findPOReadings(userprofileId);
		}
		newPOReading = new PulseOximetryReadingDTO();
		newPOReading.setReading(new Date());
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/pulseoximetry/addPODisplay.xhtml");
		return null;
	}
	
	public String dashboardViewMore() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			poreadings = findPOReadings(userprofileId);
		}
		return "/views/pulseoximetry/pulseOximetryList";
	}

	public String showDahBoardDetail() {
		resetMessages();
		findUser();
		String poid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("poid");
		Long userProfileId = getUserProfileId();
		if (poid != null && userProfileId != null) {
			Long id = Long.valueOf(poid);
			selectedPOReading = getPOReadingById(userProfileId, id);
		}
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/pulseoximetry/pulseOximetry.xhtml");
		return null;
	}

	private PulseOximetryReadingDTO getPOReadingById(Long userProfileId, Long id) {
		PulseOximetryReadingDTO dto = null;
		try {

			dto = this.vitalSignsService.getPulseOximetryReadingByid(userProfileId, id);

		} catch (Exception e) {
			log.error("Error in Find PO reading By id:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}
		return dto;
	}

	public String delete() {
		String outcome = null;
		String poid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("poid");
		Long userProfileId = getUserProfileId();

		try {
			if (poid != null && userProfileId != null) {

				this.vitalSignsService.deletePulseOximetryReading(userProfileId, Long.valueOf(poid));
				poreadings = findPOReadings(userprofileId);
				deleteMessage = true;
				outcome = "pulseOximetryList";
			}
		} catch (Exception e) {
			log.error("Failed to Delete Pulse Oximetry record", e);

			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return outcome;
	}

	public String save() {
		String outcome = null;
		try {
			if (savePOReading().getPulseOximetryId() != null) {
				outcome = "pulseOximetryList";
				saveMessage = true;
				poreadings = findPOReadings(userprofileId);
			}
		} catch (MHVException e) {
			log.error("Failed to Save Pulse Oximetry record", e);
			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public PulseOximetryReadingDTO savePOReading() throws MHVException {
		resetMessages();
		String poid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("poid");
		Long id = null;
		PulseOximetryReadingDTO poreading = null;
		if (poid != null) {
			id = Long.valueOf(poid);
			poreading = getSelectedPOReading();
		} else {
			poreading = getNewPOReading();
		}

		poreading = preparePOReading(id, poreading);
		Long poId = this.vitalSignsService.savePulseOximetryReading(poreading);
		poreading.setPulseOximetryId(poId);
		return poreading;

	}

	private PulseOximetryReadingDTO preparePOReading(Long id, PulseOximetryReadingDTO poreading) {
		Date dateTime = null;
		if (poreading.getReading() != null) {
			if (poreading.getHour() == null && poreading.getMinute() == null) {
				dateTime = WebUtility.getDaTeTime(poreading.getReading(), DEFAULT_HOUR, DEFAULT_MINUTE);
			} else {
				dateTime = WebUtility.getDaTeTime(poreading.getReading(), poreading.getHour(), poreading.getMinute());
			}
		}
		poreading.setReading(dateTime);
		poreading.setUserprofileId(getUserProfileId());
		return poreading;
	}

	public String saveAndAdd() {
		String outcome = null;
		try {
			if (savePOReading().getPulseOximetryId() != null) {
				outcome = "addPODisplay";
				saveAndAddMessage = true;
				newPOReading = new PulseOximetryReadingDTO();
				newPOReading.setReading(new Date());
				poreadings = findPOReadings(userprofileId);
			}

		} catch (MHVException e) {
			log.error("Failed to Save Pulse Oximetry record", e);
			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public String updateLineModel() {
		return "pulseOximetryGraph";
	}

	public StreamedContent getJchart() {
		StreamedContent content = null;
		try {
			content = createChart(TITLE, XAXIS, YAXIS);
		} catch (Exception e) {
			log.error("Failed to create Chart", e);
		}
		return content;
	}

	protected XYDataset createDataset(Set<String> uniqueYears) {
		TimeSeriesCollection dataset = new TimeSeriesCollection();
		TimeSeries series1 = new TimeSeries(YAXIS);
		String period = getPeriod();
		Date startDate = null;
		List<PulseOximetryReadingDTO> graphList;
		if (!period.equals(PeriodEnumeration.ALL_DATA.getDescription())) {
			startDate = getStartGraphDate(period);
		}
		if (startDate != null) {
			graphList = getGraphlist(poreadings, startDate);
		} else {
			graphList = poreadings;
		}
		for (PulseOximetryReadingDTO poreading : graphList) {
			series1.addOrUpdate(new Day(poreading.getReading()), poreading.getOximeterReading());
			uniqueYears
					.add(CommonUtility.dateToString(poreading.getReading(), YYYY_MM_DD_HHMMSS_FORMAT).substring(0, 4));
		}
		dataset.addSeries(series1);
		return dataset;
	}

	private List<PulseOximetryReadingDTO> getGraphlist(List<PulseOximetryReadingDTO> poreadings, Date startDate) {
		List<PulseOximetryReadingDTO> graphList = new ArrayList<PulseOximetryReadingDTO>();
		Integer startdateValue = Integer.valueOf(CommonUtility.dateToString(startDate, YYYYMMDD_FORMAT));
		for (PulseOximetryReadingDTO poreading : poreadings) {
			Integer readingValue = Integer.valueOf(CommonUtility.dateToString(poreading.getReading(), YYYYMMDD_FORMAT));
			if (readingValue >= startdateValue) {
				graphList.add(poreading);
			}
		}
		return graphList;
	}

	public String printerFriendlySummary(){
		resetMessages();
		poTable = (DataTable) FacesContext.getCurrentInstance().getViewRoot().findComponent("poForm:poList");
		return "printPOSummary";
	}

	public List<PulseOximetryReadingDTO> getPoreadings() {
		return poreadings;
	}

	public void setPoreadings(List<PulseOximetryReadingDTO> poreadings) {
		this.poreadings = poreadings;
	}

	public DataTable getPoTable() {
		return poTable;
	}

	public void setPoTable(DataTable poTable) {
		this.poTable = poTable;
	}

	public PulseOximetryReadingDTO getSelectedPOReading() {
		return selectedPOReading;
	}

	public void setSelectedPOReading(PulseOximetryReadingDTO selectedPOReading) {
		this.selectedPOReading = selectedPOReading;
	}

	public PulseOximetryReadingDTO getNewPOReading() {
		return newPOReading;
	}

	public void setNewPOReading(PulseOximetryReadingDTO newPOReading) {
		this.newPOReading = newPOReading;
	}

	public static String getXaxis() {
		return XAXIS;
	}

	public static String getYaxis() {
		return YAXIS;
	}

}
