package gov.va.med.mhv.vitals.web.controller;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.data.time.Day;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;
import org.primefaces.component.datatable.DataTable;
import org.primefaces.event.data.SortEvent;
import org.primefaces.model.StreamedContent;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.vitals.dto.InrReadingDTO;
import gov.va.med.mhv.vitals.enums.PeriodEnumeration;
import gov.va.med.mhv.vitals.util.CommonUtility;
import gov.va.med.mhv.vitals.web.util.WebUtility;

@ManagedBean
@Component
@Scope("session")
public class InrController extends AbstractController {

	private static final long serialVersionUID = 2890169291805333677L;
	private static Logger log = LogManager.getLogger(InrController.class);
	private List<InrReadingDTO> inrreadings = new ArrayList<InrReadingDTO>();
	private DataTable inrTable;
	private InrReadingDTO selectedINRReading = new InrReadingDTO();
	private InrReadingDTO newINRReading = new InrReadingDTO();
	private static final String TITLE = "International Normalized Ratio (INR)";
	private static final String XAXIS = "Date";
	private static final String YAXIS = "International Normalized Ratio (INR) Value";

	public void init(ComponentSystemEvent event) {
		findUser();
		userprofileId = getUserProfileId();
		inrTable = (DataTable) FacesContext.getCurrentInstance().getViewRoot().findComponent("inrForm:inrlist");
		if (!FacesContext.getCurrentInstance().isPostback()) {
			resetMessages();
			if (userprofileId != null) {
				inrreadings = findINRReadings(userprofileId);
			}
			setRowsPerPage(10);
			setPeriod(PeriodEnumeration.ONE_MONTH.getDescription());
		}
		else{
			if(sortColumn != null && sortBy != null){
				inrTable.setValueExpression("sortBy", sortColumn);
				inrTable.setSortOrder(sortBy);
			} 
		}
	}
	
	public void onSort(SortEvent event){
		sortColumn=event.getSortColumn().getValueExpression("sortBy");
		sortBy=event.isAscending()?"ascending":"descending";
	}

	private List<InrReadingDTO> findINRReadings(Long userprofileId) {
		List<InrReadingDTO> dtoList = null;
		try {

			dtoList = this.vitalSignsService.getInrReadingsForUser(userprofileId);

		} catch (Exception e) {
			log.error("Error in Find INR readings:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return dtoList;
	}

	public String showDetail(InrReadingDTO inrReading) {
		resetMessages();
		selectedINRReading = inrReading;
		setHourMinute(selectedINRReading);
		return "inr";
	}

	public String editDisplay(InrReadingDTO inrReading) {
		resetMessages();
		selectedINRReading = inrReading;
		setHourMinute(selectedINRReading);
		return "editINRDisplay";
	}
	
	public String resetEditDisplay(){
		return "editINRDisplay";
	}

	public String deleteDisplay(InrReadingDTO inrReading) {
		resetMessages();
		selectedINRReading = inrReading;
		setHourMinute(selectedINRReading);
		setDeleteOrigin("tableView");
		return "deleteINRDisplay";
	}

	public String deleteRecordDisplay() {
		resetMessages();
		setDeleteOrigin(null);
		return "deleteINRDisplay";
	}

	private void setHourMinute(InrReadingDTO selectedINRReading) {
		String dateTime = WebUtility.dateToString(selectedINRReading.getReading(), HOUR_MINUTE);
		if (!dateTime.equals(DEFAULT_DATETIME)) {
			selectedINRReading.setHour(WebUtility.dateToString(selectedINRReading.getReading(), HOURS));
			selectedINRReading.setMinute(WebUtility.dateToString(selectedINRReading.getReading(), MINUTES));
		}

	}

	public String addDisplay() {
		resetMessages();
		newINRReading = new InrReadingDTO();
		newINRReading.setInr(new BigDecimal(new String("0.0")));
		newINRReading.setReading(new Date());
		return "addINRDisplay";
	}

	public String dashboardAddDisplay() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			inrreadings = findINRReadings(userprofileId);
		}
		newINRReading = new InrReadingDTO();
		newINRReading.setInr(new BigDecimal(new String("0.0")));
		newINRReading.setReading(new Date());
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/inr/addINRDisplay.xhtml");
		return null;
	}
	
	public String dashboardViewMore() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			inrreadings = findINRReadings(userprofileId);
		}
		return "/views/inr/inrList";
	}

	public String showDahBoardDetail() {
		resetMessages();
		findUser();
		String inrid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("inrid");
		Long userProfileId = getUserProfileId();
		if (inrid != null && userProfileId != null) {
			Long id = Long.valueOf(inrid);
			selectedINRReading = getINRReadingById(userProfileId, id);
		}
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null, "/views/inr/inr.xhtml");
		return null;
	}

	private InrReadingDTO getINRReadingById(Long userProfileId, Long id) {
		InrReadingDTO dto = null;
		try {

			dto = this.vitalSignsService.getINRReadingByid(userProfileId, id);

		} catch (Exception e) {
			log.error("Error in Find INR reading By id:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return dto;
	}

	public String delete() {
		String outcome = null;
		String inrid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("inrid");
		Long userProfileId = getUserProfileId();

		try {
			if (inrid != null && userProfileId != null) {
				Long id = Long.valueOf(inrid);

				this.vitalSignsService.deleteInrReading(userProfileId, id);
				inrreadings = findINRReadings(userprofileId);
				deleteMessage = true;
				outcome = "inrList";
			}
		} catch (Exception e) {
			log.error("Failed to Delete INR record", e);

			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return outcome;
	}

	public String save() {
		String outcome = null;
		try {
			if (saveINRReading().getInrId() != null) {
				outcome = "inrList";
				inrreadings = findINRReadings(userprofileId);
				saveMessage = true;
			}
		} catch (MHVException e) {
			log.error("Failed to Save INR record", e);
			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public InrReadingDTO saveINRReading() throws MHVException {
		resetMessages();
		String inrid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("inrid");
		Long id = null;
		InrReadingDTO inrreading = null;
		if (inrid != null) {
			id = Long.valueOf(inrid);
			inrreading = getSelectedINRReading();
		} else {
			inrreading = getNewINRReading();
		}

		inrreading = prepareINRReading(id, inrreading);
		Long inrId = this.vitalSignsService.saveInrReading(inrreading);
		inrreading.setInrId(inrId);
		return inrreading;
	}

	private InrReadingDTO prepareINRReading(Long id, InrReadingDTO inrreading) {
		Date dateTime = null;
		if (inrreading.getReading() != null) {
			if (inrreading.getHour() == null && inrreading.getMinute() == null) {
				dateTime = WebUtility.getDaTeTime(inrreading.getReading(), DEFAULT_HOUR, DEFAULT_MINUTE);
			} else {
				dateTime = WebUtility.getDaTeTime(inrreading.getReading(), inrreading.getHour(),
						inrreading.getMinute());
			}
		}
		inrreading.setReading(dateTime);
		inrreading.setUserprofileId(getUserProfileId());
		return inrreading;
	}

	public String saveAndAdd() {
		String outcome = null;
		try {
			if (saveINRReading().getInrId() != null) {
				outcome = "addINRDisplay";
				saveAndAddMessage = true;
				newINRReading = new InrReadingDTO();
				newINRReading.setInr(new BigDecimal(new String("0.0")));
				newINRReading.setReading(new Date());
				inrreadings = findINRReadings(userprofileId);
			}

		} catch (MHVException e) {
			log.error("Failed to Save INR record", e);
			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public String updateLineModel() {
		return "inrGraph";
	}

	public StreamedContent getJchart() {
		StreamedContent content = null;
		try {
			content = createChart(TITLE, XAXIS, YAXIS);
		} catch (Exception e) {
			log.error("Failed to create Chart", e);
		}
		return content;
	}

	protected XYDataset createDataset(Set<String> uniqueYears) {
		TimeSeriesCollection dataset = new TimeSeriesCollection();
		TimeSeries series1 = new TimeSeries(YAXIS);
		String period = getPeriod();
		Date startDate = null;
		List<InrReadingDTO> graphList;
		if (!period.equals(PeriodEnumeration.ALL_DATA.getDescription())) {
			startDate = getStartGraphDate(period);
		}
		if (startDate != null) {
			graphList = getGraphlist(inrreadings, startDate);
		} else {
			graphList = inrreadings;
		}
		for (InrReadingDTO inrreading : graphList) {
			series1.addOrUpdate(new Day(inrreading.getReading()), inrreading.getInr());
			uniqueYears
					.add(CommonUtility.dateToString(inrreading.getReading(), YYYY_MM_DD_HHMMSS_FORMAT).substring(0, 4));
		}
		dataset.addSeries(series1);
		return dataset;
	}

	private List<InrReadingDTO> getGraphlist(List<InrReadingDTO> inrreadings, Date startDate) {
		List<InrReadingDTO> graphList = new ArrayList<InrReadingDTO>();
		Integer startdateValue = Integer.valueOf(CommonUtility.dateToString(startDate, YYYYMMDD_FORMAT));
		for (InrReadingDTO inrreading : inrreadings) {
			Integer readingValue = Integer
					.valueOf(CommonUtility.dateToString(inrreading.getReading(), YYYYMMDD_FORMAT));
			if (readingValue >= startdateValue) {
				graphList.add(inrreading);
			}
		}
		return graphList;
	}
	
	public String printerFriendlySummary(){
		resetMessages();
		inrTable = (DataTable) FacesContext.getCurrentInstance().getViewRoot().findComponent("inrForm:inrlist");
		return "printINRSummary";
	}

	public List<InrReadingDTO> getInrreadings() {
		return inrreadings;
	}

	public void setInrreadings(List<InrReadingDTO> inrreadings) {
		this.inrreadings = inrreadings;
	}

	public DataTable getInrTable() {
		return inrTable;
	}

	public void setInrTable(DataTable inrTable) {
		this.inrTable = inrTable;
	}

	public InrReadingDTO getSelectedINRReading() {
		return selectedINRReading;
	}

	public void setSelectedINRReading(InrReadingDTO selectedINRReading) {
		this.selectedINRReading = selectedINRReading;
	}

	public InrReadingDTO getNewINRReading() {
		return newINRReading;
	}

	public void setNewINRReading(InrReadingDTO newINRReading) {
		this.newINRReading = newINRReading;
	}

}
