package gov.va.med.mhv.vitals.service.impl;

import java.util.LinkedHashMap;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.vitals.converter.BloodPressureConverter;
import gov.va.med.mhv.vitals.dto.BloodPressureReadingDTO;
import gov.va.med.mhv.vitals.model.BloodPressureReading;
import gov.va.med.mhv.vitals.repository.BloodPressureReadingRepository;
import gov.va.med.mhv.vitals.service.BloodPressureService;
import gov.va.med.mhv.vitals.validator.BloodPressureValidator;

@Component
public class BloodPressureServiceImpl implements BloodPressureService {
	
	private static Logger log = LogManager.getLogger(BloodPressureServiceImpl.class);
	
	@Autowired
	private BloodPressureReadingRepository bpRepository;
	
	@Autowired
	private BloodPressureValidator validator;

	@Override
	public ResponseUtil<Long> save(BloodPressureReadingDTO bpDTO) {
		ResponseUtil<Long> response = new ResponseUtil<Long>();
		response.setValidationErrors(new LinkedHashMap<String, String>());
		validator.bpReadingValidations(response, bpDTO);
		if(response.getValidationErrors().size() == 0){
			BloodPressureReading bpReading = BloodPressureConverter.convertBPReadingDTO(bpDTO);
			bpReading = bpRepository.save(bpReading);
			response.setSuccess(true);
			response.setPojoObject(bpReading.getId());
		}else{
			log.debug("Validation Errors in save BP reading:");
			response.setFailure(true);
		}
		return response;
	}

	@Override
	public ResponseUtil delete(Long id) {
		ResponseUtil response = new ResponseUtil();
		bpRepository.delete(id);
		response.setSuccess(true);
		return response;
	}

	@Override
	public ResponseUtil findByPrimaryKey(Long id) {
		ResponseUtil response = new ResponseUtil();
		BloodPressureReading bpReading = bpRepository.findOne(id);
		if(bpReading != null){
			BloodPressureReadingDTO bpReadingDTO = BloodPressureConverter.convertBPReadingDomain(bpReading);
			response.setPojoObject(bpReadingDTO);
			response.setSuccess(true);
		}else{
			log.debug("BP Record not found for id:");
			log.debug(id);
		}
		return response;
	}

	@Override
	public ResponseUtil<List<BloodPressureReadingDTO>> getBPReadingsForUser(Long userprofileid) {
		ResponseUtil response = new ResponseUtil();
		List<BloodPressureReading> bpReadings=bpRepository.getBPReadingsForUser(userprofileid);
		if(bpReadings != null){
			List<BloodPressureReadingDTO> dtos = BloodPressureConverter.convertBPReadings(bpReadings);
			response.setPojoObject(dtos);
		}else{
			log.debug("BP Records not found for userprofileid:");
			log.debug(userprofileid);
		}
		response.setSuccess(true);
		return response;
	}

	@Override
	public ResponseUtil<List<BloodPressureReadingDTO>> getBPDashBoardReadingsForUser(Long userprofileid) {
		ResponseUtil<List<BloodPressureReadingDTO>> response = new ResponseUtil<List<BloodPressureReadingDTO>>();
		Pageable topFive = new  PageRequest(0,5);
		List<BloodPressureReading> bpReadings=bpRepository.getDashBoardBPReadingsForUser(userprofileid, topFive);
		if(bpReadings != null){
			List<BloodPressureReadingDTO> dtos = BloodPressureConverter.convertBPReadings(bpReadings);
			response.setPojoObject(dtos);
		}else{
			log.debug("BP Records not found for userprofileid:");
			log.debug(userprofileid);
		}
		response.setSuccess(true);
		return response;
	}
	
	@Override
	public ResponseUtil<BloodPressureReadingDTO> getBPReadingByid(Long userProfileId, Long id) {
		ResponseUtil response = new ResponseUtil();
		
		List<BloodPressureReading> bpReadings = bpRepository.findByIdAndUserProfileId(id, userProfileId);
		if(bpReadings != null && bpReadings.size() == 1){
			BloodPressureReadingDTO dto=BloodPressureConverter.convertBPReadingDomain(bpReadings.get(0));
			response.setPojoObject(dto);
		}else{
			log.debug("BP Record not found for Id:");
			log.debug(id);
		}
		response.setSuccess(true);
		return response;
	}


	/**
	 * @return the bpRepository
	 */
	public BloodPressureReadingRepository getBpRepository() {
		return bpRepository;
	}

	/**
	 * @param bpRepository the bpRepository to set
	 */
	public void setBpRepository(BloodPressureReadingRepository bpRepository) {
		this.bpRepository = bpRepository;
	}

	/**
	 * @return the validator
	 */
	public BloodPressureValidator getValidator() {
		return validator;
	}

	/**
	 * @param validator the validator to set
	 */
	public void setValidator(BloodPressureValidator validator) {
		this.validator = validator;
	}

}
