package gov.va.med.mhv.usermgmt.web.controller;

import java.io.IOException;
import java.io.Serializable;

import javax.annotation.PostConstruct;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.ManagedProperty;
import javax.faces.bean.RequestScoped;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.va.med.mhv.usermgmt.common.dto.HealthInfoDTO;
import gov.va.med.mhv.usermgmt.common.enums.BloodTypeEnumeration;
import gov.va.med.mhv.usermgmt.service.UserMgmtService;
import gov.va.med.mhv.usermgmt.web.converter.HealthInfoCardConverter;
import gov.va.med.mhv.usermgmt.web.formbean.HealthInfoCard;

@ManagedBean
@RequestScoped
public class HealthInfoCardController implements Serializable {
	private static final long serialVersionUID = 1L;
	private static Logger log = LogManager.getLogger(HealthInfoCardController.class);

	// Note that the spec states 35 characters, but more characters fit
	// It does not makes sense to strip characters that would fit
	private static final int ALLERGIES_FIRST_LINE_LENGTH = 35;
	private static final int ALLERGIES_OTHER_LINES_LENGTH = 42;
	private static final int EMAIL_LENGTH = 40;

	@ManagedProperty("#{userMgmtServiceProxy}")
	private UserMgmtService userMgmtServiceProxy;

	@ManagedProperty("#{healthInfoCardConverter}")
	private HealthInfoCardConverter healthInfoCardConverter;

	private HealthInfoCard healthInfoCard;
	private String allergyLine1;
	private String allergyLine2;
	private String allergyLine3;
	
	
	@PostConstruct
	private void init() {
		log.debug("@PostConstruct");

		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext()
					.getRequest();
			PortletSession session = request.getPortletSession();

			Long userProfileId = (Long) session.getAttribute("LIFERAY_SHARED_userprofileid",
					PortletSession.APPLICATION_SCOPE);

			log.debug("userProfileId: " + userProfileId);

			if (userProfileId == null) {
				return;
			}

			HealthInfoDTO dto = userMgmtServiceProxy.getHealthInfo(userProfileId);

			healthInfoCard = healthInfoCardConverter.convert(dto);

			populateAllergies();

		} catch (Exception e) {
			log.error(e);
			healthInfoCard = null;
		}
	}

	private void populateAllergies() {

		if (healthInfoCard.getAllergies().isEmpty()) {
			return;
		}

		// have to split between 3 lines
		int lineNo = 1;
		int lineLen = ALLERGIES_FIRST_LINE_LENGTH;
		StringBuffer sb = new StringBuffer();
		for (String a : healthInfoCard.getAllergies()) {
			String allergy = a.trim().toUpperCase();
			if (sb.length() + allergy.length() > lineLen) {
				switch (lineNo) {
				case 1:
					// first line
					allergyLine1 = sb.toString();
					lineLen = ALLERGIES_OTHER_LINES_LENGTH;
					sb = new StringBuffer();
					break;
				case 2:
					// second line
					allergyLine2 = sb.toString();
					sb = new StringBuffer();
					break;
				}
				lineNo++;
			}

			if (sb.length() > 0) {
				sb.append(", ");
			}
			sb.append(allergy);
		}
		if (lineNo >= 3) {
			// third line
			allergyLine3 = sb.toString();
		}
		
		if(sb.length() <= lineLen && lineNo==1) {
			allergyLine1=sb.toString();
		}
	
	

	}

	public String getMhvId() {
		return strip(healthInfoCard.getMhvId(), 18);
	}

	public String getName() {
		return asField(healthInfoCard.getName());
	}

	public String getStreet() {
		String street = defaultString(healthInfoCard.getAddr1());
		if (!isBlank(healthInfoCard.getAddr2())) {
			street += (street.length() < 24) ? "<br/>" : ", ";
			street += healthInfoCard.getAddr2();
		}
		return asField(street);
	}

	public String getCityStateZip() {
		return asField(healthInfoCard.getCity() + ", " + healthInfoCard.getState() + " " + healthInfoCard.getZip());
	}

	public String getHomePhone() {
		return asPhoneNumber(healthInfoCard.getHomePhone());
	}

	public String getWorkPhone() {
		return asPhoneNumber(healthInfoCard.getWorkPhone());
	}

	public String getMobilePhone() {
		return asPhoneNumber(healthInfoCard.getMobilePhone());
	}

	public String getEmail() {
		return asField(healthInfoCard.getEmail(), EMAIL_LENGTH);
	}

	public String getBloodType() {
		String bloodType = "";

		try {
			bloodType = BloodTypeEnumeration.valueOf(healthInfoCard.getBloodType().toUpperCase().trim())
					.getDescription();
		} catch (Exception e) {
		}

		return asField(bloodType);
	}

	public String getAllergiesLine1() {
		return asField(allergyLine1);
	}

	public String getAllergiesLine2() {
		return asField(allergyLine2);
	}

	public String getAllergiesLine3() {
		return asField(allergyLine3);
	}

	public String getMedicalConditionsLine1() {
		return "";
	}

	public String getMedicalConditionsLine2() {
		return "";
	}

	public String getPrimaryProvider() {
		return asProviderField(healthInfoCard.getPrimaryProvider());
	}

	public String getPrimaryProviderPhone() {
		return asProviderField(addPhoneExtension(healthInfoCard.getPrimaryProviderPhone(),
				healthInfoCard.getPrimaryProviderPhoneExtension()));
	}

	public String getPrimaryInsurance() {
		return asProviderField(healthInfoCard.getPrimaryInsurance());
	}

	public String getPrimaryInsuranceId() {
		return asProviderField(healthInfoCard.getPrimaryInsuranceId());
	}

	public String getPrimaryInsuranceGroupNumber() {
		return asProviderField(healthInfoCard.getPrimaryInsuranceGroupNumber());
	}

	public String getPrimaryInsurancePhone() {
		return asProviderField(healthInfoCard.getPrimaryInsurancePhone());
	}

	public String getEmergencyContactName() {
		return asField(healthInfoCard.getEmergencyContactName(), 26);
	}

	public String getEmergencyContactHomePhone() {
		return asPhoneNumber(healthInfoCard.getEmergencyContactHomePhone());
	}

	public String getEmergencyContactWorkPhone() {
		return asPhoneNumber(addPhoneExtension(healthInfoCard.getEmergencyContactWorkPhone(),
				healthInfoCard.getEmergencyContactWorkPhoneExtension()));
	}

	public String getEmergencyContactMobilePhone() {
		return asPhoneNumber(healthInfoCard.getEmergencyContactMobilePhone());
	}

	public String getEmergencyContactEmail() {
		return strip(healthInfoCard.getEmergencyContactEmail(), 26).toUpperCase();
	}

	private String addPhoneExtension(String phone, String extension) {
		if (phone == null) {
			phone = "";
		}
		if (!isBlank(extension)) {
			phone += "x" + extension; // Refer Bug-MHV00003185(Primary Provider
										// phone extension has unusual format).
		}
		return phone;
	}

	private String asPhoneNumber(String phoneNumber) {
		return strip(phoneNumber, 18);
	}

	private String asProviderField(String field) {
		return strip(asField(field), 25);
	}

	private String strip(String str, int length) {
		if (str == null) {
			return "";
		} else if (str.length() <= length) {
			return str;
		} else {
			return str.substring(0, length);
		}
	}

	private String asField(String str, int length) {
		return strip(str, length).toUpperCase();
	}

	private String asField(String str) {
		return str != null ? str.toUpperCase() : "";
	}

	private boolean isBlank(String str) {
		return str == null || str.isEmpty();
	}

	private String defaultString(String str) {
		return str != null ? str : "";
	}

	public HealthInfoCard getHealthInfoCard() {
		return healthInfoCard;
	}

	public void setHealthInfoCard(HealthInfoCard healthInfoCard) {
		this.healthInfoCard = healthInfoCard;
	}

	public void setUserMgmtServiceProxy(UserMgmtService userMgmtServiceProxy) {
		this.userMgmtServiceProxy = userMgmtServiceProxy;
	}

	public void setHealthInfoCardConverter(HealthInfoCardConverter healthInfoCardConverter) {
		this.healthInfoCardConverter = healthInfoCardConverter;
	}
}
