package gov.va.med.mhv.usermgmt.main.service.impl;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.ws.rs.PathParam;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.dto.EmergencyContactDTO;
import gov.va.med.mhv.common.api.dto.InPersonAuthenticationDTO;
import gov.va.med.mhv.common.api.dto.PatientDTO;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.common.api.enumeration.ErrorEnum;
import gov.va.med.mhv.common.api.enumeration.FieldTestEnum;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.common.data.converter.InPersonAuthenticationConverter;
import gov.va.med.mhv.common.data.converter.PatientConverter;
import gov.va.med.mhv.common.data.converter.UserProfileConverter;
import gov.va.med.mhv.common.data.model.InPersonAuthentication;
import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.common.data.model.UserProfile;
import gov.va.med.mhv.usermgmt.common.dto.EauthHashDTO;
import gov.va.med.mhv.usermgmt.common.dto.FieldTestDTO;
import gov.va.med.mhv.usermgmt.common.dto.HealthCareProviderDTO;
import gov.va.med.mhv.usermgmt.common.dto.UserAcceptTermsHistoryDTO;
import gov.va.med.mhv.usermgmt.common.dto.HealthInfoDTO;
import gov.va.med.mhv.usermgmt.common.enums.AuthenticationStatusEnumeration;
import gov.va.med.mhv.usermgmt.common.enums.PatientCorrelationStatusEnumeration;
import gov.va.med.mhv.usermgmt.data.model.AccessControl;
import gov.va.med.mhv.usermgmt.data.model.UserAcceptTermsHistory;
import gov.va.med.mhv.usermgmt.data.repository.PatientRepository;
import gov.va.med.mhv.usermgmt.data.repository.UserAcceptTermsHistoryRepository;
import gov.va.med.mhv.usermgmt.service.AccessControlService;
import gov.va.med.mhv.usermgmt.service.AuthenticateService;
import gov.va.med.mhv.usermgmt.service.EmailService;
import gov.va.med.mhv.usermgmt.service.EmergencyContactService;
import gov.va.med.mhv.usermgmt.service.HealthCareProviderService;
import gov.va.med.mhv.usermgmt.service.UserAcceptTermsHistoryService;
import gov.va.med.mhv.usermgmt.service.InPersonAuthenticationService;
import gov.va.med.mhv.usermgmt.service.MviIntegrationService;
import gov.va.med.mhv.usermgmt.service.PatientService;
import gov.va.med.mhv.usermgmt.service.UserMgmtService;
import gov.va.med.mhv.usermgmt.service.UserPasswordService;
import gov.va.med.mhv.usermgmt.service.UserProfileService;


@Component
public class UserMgmtServiceImpl implements UserMgmtService {
	private static Logger log = LogManager.getLogger(UserMgmtServiceImpl.class);
	
	private static final String EMAIL_SUBJECT="My HealtheVet Account Activity Update";
	private static final String EMAIL_MESSAGE_CONTENT_TYPE = "text/html";
	private static final String BREAK="<br/>";


	@Autowired
	private AuthenticateService authenticateService;

	@Autowired
	private UserProfileService userProfileService;

	@Autowired
	private AccessControlService accessControlService;

	@Autowired
	private UserPasswordService userPasswordService;

	@Autowired
	private EmergencyContactService emergencyContactService;

	@Autowired
	private HealthCareProviderService healthcareProviderService;

	@Autowired
	private PatientService patientService;

	@Autowired
	private InPersonAuthenticationService ipaService;

	@Autowired
	private MviIntegrationService mviIntegrationService;

	@Autowired
	private UserAcceptTermsHistoryService userAcceptTermsHistoryService;

	@Autowired
	InPersonAuthenticationConverter inPersonAuthenticationConverter;

	@Autowired
	PatientConverter patientConverter;
	
	@Autowired
	UserProfileConverter profileConverter;

	@Autowired
	private PatientRepository patientRepository;
	
	@Autowired
	private EmailService emailService;
	
	public UserProfileDTO getUserProfile(String userName) throws MHVException {
		if(log.isDebugEnabled()) {
			log.debug("DEBUG SERVER -- GET USER PROFILE ");
		}
		UserProfileDTO profile = new UserProfileDTO();
		profile.setUserName(userName);
		try {
			profile = userProfileService.getUserProfileByUserName(userName);
		} catch (Exception e) {
			String errorMessage = "Exception Caught Getting User Profile";
			log.error(errorMessage, e);
			throw new MHVException(errorMessage, e);
		}
		return profile;
	}

	public UserProfileDTO saveUserProfile(UserProfileDTO userProfile) throws MHVException {

		if(log.isDebugEnabled()) {
			log.debug(" DEBUG SERVER -- SAVE PROFILE ");
		}

		ResponseUtil<UserProfileDTO> response = new ResponseUtil<UserProfileDTO>();

		try {
			userProfileService.updateUserProfileResponse(userProfile, response);

			if(response.isFailure()) {
				log.error(response.getFailureMessage());
				throw new MHVException(response.getInfoMessages(), response.getValidationErrors(), response.getFailureMessage());
			}

			if (log.isDebugEnabled()) {
				log.debug("SAVE PROFILE completed successful...");
			}

		} catch (MHVException ex) {
			throw ex;
		} catch (Exception e) {
			String errorMessage = "Exception Caught Updating User Profile";
			log.error(errorMessage, e);
			throw new MHVException(errorMessage, e);
		}
		return userProfile;
	}

	public void deleteEauthHashByUserProfileId(Long userProfileId) throws MHVException {
		log.debug(" DEBUG SERVER -- Delete EauthHsh by userprofileid ");
		try {
			userProfileService.deleteEauthHashByUserProfileId(userProfileId);
		} catch (Exception e) {
			String errorMessage = "Exception Caught deleting EauthHash";
			log.error(errorMessage, e);
			throw new MHVException(errorMessage, e);
		}
	}

	public List<EauthHashDTO> findEAuthHashByUserProfileId(@PathParam("userProfileId") Long userProfileId) throws MHVException {
		if(log.isDebugEnabled()) {
			log.debug(" DEBUG SERVER -- GET EauthHsh by userprofileid ");
		}
		ResponseUtil<List<EauthHashDTO>> response = null;
		try {
			response = userProfileService.findEAuthHashByUserProfileId(userProfileId);
		} catch (Exception e) {
			String errorMessage = "Exception Caught Getting EauthHash";
			log.error(errorMessage, e);
			throw new MHVException(errorMessage, e);
		}
		return response.getPojoObject();
	}

	public UserProfileDTO getUserProfileById(Long id) throws MHVException {
		if(log.isDebugEnabled()) {
			log.debug("DEBUG SERVER -- GET USER PROFILE By ID ");
		}

		ResponseUtil<UserProfileDTO> response = null;
		try {
			response = userProfileService.getUserProfileByID(id);
		} catch (Exception e) {
			String errorMessage = "Exception Caught Getting User Profile By id";
			log.error(errorMessage, e);
			throw new MHVException(errorMessage, e);
		}
		return response.getPojoObject();
	}

	public List<FieldTestDTO> getFieldTestGroupsByUserName(String userName) throws MHVException {
		if(log.isDebugEnabled()) {
			log.debug("Fetching the Field Test Groups for user " + userName);
		}

		if(userName.isEmpty()) {
			throw new MHVException("The field userName cannot be empty");
		}

		ResponseUtil<List<AccessControl>> response = new ResponseUtil<List<AccessControl>>();

		try {
			accessControlService.findAccessControls(userName, response);

			List<FieldTestDTO> dtoList= new ArrayList<FieldTestDTO>();

			List<String> names = FieldTestEnum.toNameList();

			for(AccessControl ac: response.getPojoObject()) {
				if( !names.contains(ac.getFeature().getName()) ) {
					if(log.isDebugEnabled()) {
						log.debug("Skipping this feature found in the database: '" + ac.getFeature().getName() + "' because it is not in the FieldTestEnum list: " + names );
					}
					continue;
				}

				if(log.isDebugEnabled()) {
					log.debug("Adding feature: " + ac.getFeature().getName() + " with role: " + ac.getAccessRole().getName() + " and isNational: " + ac.getAccessRole().getIsDefaultPatientRole());
				}
				dtoList.add(new FieldTestDTO(ac.getFeature().getName(), ac.getAccessRole().getName(), ac.getAccessRole().getIsDefaultPatientRole()));
			}

			return dtoList;
		} catch (Exception e) {
			String errorMessage = "Error retrieving field test groups from user.";
			log.error(errorMessage);
			throw new MHVException(errorMessage, e);
		}
 	}

	@Override
	public UserProfileDTO postLoginUpdateUser(Long userProfileId) throws MHVException {
		if(log.isDebugEnabled()) {
			log.debug(" DEBUG SERVER -- AUTHENTICATE ");
		}
		ResponseUtil<UserProfileDTO> response = null;

		try {
			response = authenticateService.postLoginUserUpdate(userProfileId);
		} catch (Exception e) {
			String errorMessage = "Exception Caught Authenticating User";
			log.error(errorMessage, e);
			throw new MHVException(errorMessage, e);
		}
		return response.getPojoObject();
	}

	@Override
	public HealthInfoDTO getHealthInfo(Long userProfileId) {
		return userProfileService.getHealthInfo(userProfileId);
	}

	@Override
	public UserProfileDTO getUserProfile(String firstName, String lastName, String gender, Date birthDate)
			throws MHVException {

		ResponseUtil<UserProfileDTO> response = new ResponseUtil<UserProfileDTO>();

		try {
			userProfileService.getProfileByNameBirthDate(firstName, lastName, gender, birthDate, response);
		}
		catch (Exception e) {
			MHVException mhvex = new MHVException(ErrorEnum.USER_NOT_FOUND_105, e);
			log.error("Exception caught during patient lookup: ", e);
			throw mhvex;
		}

		if(response.isFailure()) {
			throw new MHVException(response.getInfoMessages(), response.getValidationErrors(), response.getFailureMessage());
		}

		return response.getPojoObject();
	}
	
	@Override
	public Boolean validateHintAnswers(Long userProfileId, String answer1, String answer2)
			throws MHVException {

		ResponseUtil<UserProfileDTO> response = new ResponseUtil<UserProfileDTO>();

		try {
			userProfileService.validateHintAnswers(userProfileId, answer1, answer2,response);
		}
		catch (Exception e) {
			MHVException mhvex = new MHVException(ErrorEnum.USER_NOT_FOUND_105, e);
			log.error("Exception caught during patient lookup: ", e);
			throw mhvex;
		}

		if(response.isFailure()) {
			throw new MHVException(response.getInfoMessages(), response.getValidationErrors(), response.getFailureMessage());
		}

		return Boolean.TRUE;
	}
	

	@Override
	public UserProfileDTO changePassword(UserProfileDTO userProfile, Boolean isForgotUserPwd) throws MHVException {
		ResponseUtil<UserProfileDTO> response = new ResponseUtil<UserProfileDTO>();

		try {
			this.userPasswordService.changePassword(userProfile, response,isForgotUserPwd);
			if (response.isFailure()) {
				throw new MHVException(response);
			}
			UserProfile savedUserProfile = new UserProfile();
			savedUserProfile = profileConverter.convertDTOProfile(savedUserProfile, userProfile);
			this.userProfileService.savePasswordHistory(savedUserProfile, userProfile.getPassword(), false, response);
			if (response.isFailure()) {
				throw new MHVException(response);
			}
		} catch(MHVException e){
			throw e;
		}
		 catch(Exception ex) {
			String errorMessage = "An error was encountered while changing the users password.";
			log.error(errorMessage, ex);
			throw new MHVException(errorMessage, ex);
		}

		return userProfile;
	}

	@Override
	public List<EmergencyContactDTO> getEmergencyContacts(Long userProfileId) throws MHVException {
		try {
			log.debug("userProfileId=" + userProfileId);
			return emergencyContactService.findByUserProfileId(userProfileId);
		}
		catch (Exception e) {
			String errorMessage = "Exception Caught Getting User Profile";
			log.error(errorMessage, e);
			throw new MHVException(errorMessage, e);
		}
	}

	@Override
	public EmergencyContactDTO getEmergencyContact(Long emergencyContactId) throws MHVException {
 		try {
			log.debug("Retrieving emergency contact emergencyContactId=" + emergencyContactId);
			return emergencyContactService.getEmergencyContact(emergencyContactId);
		} catch (Exception e) {
			String errorMessage = "Exception Caught Getting Emergency Contact";
 			log.error(errorMessage, e);
 			throw new MHVException(errorMessage, e);
 		}
	}

	@Override
	public void saveEmergencyContact(EmergencyContactDTO emergencyContact) throws MHVException {
		ResponseUtil<EmergencyContactDTO> response = new ResponseUtil<EmergencyContactDTO>();
		try {
			emergencyContactService.saveEmergencyContact(emergencyContact, response);
		}
		catch (Exception e) {
			String errorMessage = "Exception Caught Getting User Profile";
			log.error(errorMessage, e);
			response.setFailure(true);
			response.setFailureMessage(e.getMessage());
		}

		if(response.isFailure()) {
			throw new MHVException(response);
		}

	}

	@Override
	public void deleteEmergencyContact(EmergencyContactDTO emergencyContact) throws MHVException {
		log.debug("deleteEmergencyContact");

		ResponseUtil<EmergencyContactDTO> response = new ResponseUtil<EmergencyContactDTO>(emergencyContact);

		try {
			emergencyContactService.deleteEmergencyContact(emergencyContact, response);
			response.setFailure(false);
		}
		catch (Exception e) {
			log.error("Exception Caught Getting User Profile", e);
			response.setFailure(true);
			response.setFailureMessage(e.getMessage());
		}

		if(response.isFailure()) {
			throw new MHVException(response);
		}
	}

	@Override
	public List<HealthCareProviderDTO> getHealthCareProviders(Long userProfileId) throws MHVException {
		try {
			return healthcareProviderService.findByUserProfileId(userProfileId);
		}
		catch (Exception e) {
			String message = "Exception Caught Getting User Profile";
			log.error(message, e);
			throw new MHVException(message, e);
		}
	}

	@Override
	public Date getLatestUserTcAcceptedTimeStamp(Long userProfileId) throws MHVException {
		try {
			List<UserAcceptTermsHistoryDTO> userAcceptTermsHistories = new ArrayList<UserAcceptTermsHistoryDTO>();
			userAcceptTermsHistories = userAcceptTermsHistoryService.getUserAcceptTermsHistByUserProfileId(userProfileId);

			// Get the userUpdatedTime from the latest HIstory record.
			for (UserAcceptTermsHistoryDTO userAcceptTermsHistory : userAcceptTermsHistories) {

				return userAcceptTermsHistory.getUserUpdatedTime();
			}
		}
		catch (Exception e) {
			String message = "Error in fetching UserAcceptTermsHistoryDTO";
			log.error(message, e);
			throw new MHVException(message, e);
		}
		return null;
	}


	@Override
	public void saveHealthCareProvider(HealthCareProviderDTO provider) throws MHVException {
		ResponseUtil<HealthCareProviderDTO> response = new ResponseUtil<HealthCareProviderDTO>();
		HealthCareProviderDTO healthCareProvider = null;

		try {
			healthcareProviderService.saveHealthCareProvider(healthCareProvider, response);
		}
		catch (Exception e) {
			String message = "Exception Caught Getting User Profile";
			log.error(message, e);
			throw new MHVException(message, e);
		}

	}

	@Override
	public void deleteHealthCareProvider(Long providerId) throws MHVException {
		ResponseUtil<Long> response = new ResponseUtil<Long>();

		try {
			healthcareProviderService.delete(providerId, response);
		}
		catch (Exception e) {
			String message = "Exception Caught Getting User Profile";
			log.error(message, e);
			throw new MHVException(message, e);
		}
	}

	@Override
	public PatientDTO getPatient(Long userProfileId) throws MHVException {

		ResponseUtil<PatientDTO> response = null;

		try {
			 response = this.patientService.getPatientByUserProfileId(userProfileId);
		} catch (Exception e) {
			String message = String.format("An unexpected error occurred while looking up patient record by user profile ID. %s", userProfileId);
			log.error(message, e);
			throw new MHVException(message, e);
		}

		if(response != null && response.isFailure()) {
			throw new MHVException(response.getInfoMessages(), response.getValidationErrors(), response.getFailureMessage());
		}

		return response.getPojoObject();
	}

	@Override
	public Boolean isPatientIPA(Long patientId) throws MHVException {

		ResponseUtil<Boolean> response = new ResponseUtil<Boolean>();
		try {
			ipaService.isPatientAuthenticated(patientId, response);
		} catch (Exception e) {
			log.error("Exception Caught checking patient in person authentication status.", e);
			response.setFailure(true);
			response.setFailureMessage(e.getMessage());
		}

		if(response != null && response.isFailure()) {
			throw new MHVException(response);
		}

		return response.getPojoObject();
	}

	@Override
	public InPersonAuthenticationDTO getPatientIPA(Long patientId) throws MHVException {
		ResponseUtil<InPersonAuthentication> response = new ResponseUtil<InPersonAuthentication>();
		try {
			ipaService.getIPAPatientById(patientId, response);
		} catch (Exception e) {
			log.error("Exception Caught getting in person authentication for patient.", e);
			response.setFailure(true);
			response.setFailureMessage(e.getMessage());
		}

		if(response != null && response.isFailure()) {
			throw new MHVException(response);
		}

		return inPersonAuthenticationConverter.convertDomainInpersonAuthentication(response.getPojoObject());
	}


	@Override
	public String DoRxIPACheck(Long patientId) throws MHVException {

		ResponseUtil<InPersonAuthentication> response = new ResponseUtil<InPersonAuthentication>();
		ResponseUtil<PatientDTO> responseUtil = new ResponseUtil<PatientDTO>();
		ResponseUtil<Patient> updateTreatmentFacilitiesResponse = new ResponseUtil<Patient>();
		if (log.isDebugEnabled()) {
			log.debug("UserMgmtServiceImpl: In side DoRxIPACheck: patient id: "+ patientId);
		}
		try {
			this.patientService.getPatientById(patientId, responseUtil);
			PatientDTO patientDto = (PatientDTO)responseUtil.getPojoObject();

			ipaService.getIPAPatientById(patientId, response);
			InPersonAuthentication ipa = (InPersonAuthentication) response.getPojoObject();
			if (null != ipa && null != ipa.getStatus()) {
				if (log.isDebugEnabled()) {
					log.debug("UserMgmtServiceImpl: IPA not null: patient id: "+ patientDto.getId());
				}

				if((ipa.getStatus().equalsIgnoreCase(AuthenticationStatusEnumeration.AUTHENTICATED.getDescription())) &&
						ipa.getPatient().getCorrelationStatus() == PatientCorrelationStatusEnumeration.UNCORRELATED.getValue() ||
						((ipa.getStatus().equals(AuthenticationStatusEnumeration.UNAUTHENTICATED.getDescription())) &&
						ipa.getPatient().getCorrelationStatus() == PatientCorrelationStatusEnumeration.UNCORRELATED.getValue()) ||
						(patientDto.getMatchedDateTime() == null && (ipa.getStatus().equals(AuthenticationStatusEnumeration.PENDINGDATAMISMATCH.getDescription())) &&
								ipa.getPatient().getCorrelationStatus() == PatientCorrelationStatusEnumeration.UNCORRELATED.getValue()) ||
						((patientDto.getMatchedDateTime() != null) &&
						ipa.getPatient().getCorrelationStatus() == PatientCorrelationStatusEnumeration.UNCORRELATED.getValue()) ){
					this.mviIntegrationService.updateTreatmentFacilitiesForExistingPatients(ipa.getPatient(), updateTreatmentFacilitiesResponse);

					if (log.isDebugEnabled()) {
						log.debug("UserMgmtServiceImpl: inside doIPACheck: patient id: "
						+ patientDto.getId());
					}
				}
			}
		} catch (Exception e) {
			log.error("Exception Caught getting in person authentication for patient.", e);
			response.setFailure(true);
			response.setFailureMessage(e.getMessage());
			return "Failure";
		}

		/*if(response != null && response.isFailure()) {
			throw new MHVException(response);
		}
		*/

		return "Success";
	}



	@Override
	public PatientDTO updateTreatmentFacilitiesForExistingPatients(Long patientId) throws MHVException {
		boolean sameDay = false;

		ResponseUtil<Patient> getPatientResponse = new ResponseUtil<Patient>();
		ResponseUtil<Patient> updateTreatmentFacilitiesResponse = new ResponseUtil<Patient>();

		try {
			ResponseUtil<PatientDTO> responseUtil = new ResponseUtil<PatientDTO>();
			this.patientService.getPatientById(patientId, responseUtil);

			if(getPatientResponse.isSuccess()) {

				PatientDTO patient = (PatientDTO)responseUtil.getPojoObject();

				//02/1/2016: We need to check to see if patientDTO exists before doing the facility update.
				if(patient != null) {
					if(patient.getLastFacilitiesUpdateDateTime() != null) {
						Calendar now = Calendar.getInstance();
						//now.setTime(now.getTime() - 1000*60*60*24);
						Calendar lastFacilitiesUpdate = Calendar.getInstance();
						lastFacilitiesUpdate.setTime(patient.getLastFacilitiesUpdateDateTime());

						sameDay = now.get(Calendar.YEAR) == lastFacilitiesUpdate.get(Calendar.YEAR) &&
							now.get(Calendar.DAY_OF_YEAR) == lastFacilitiesUpdate.get(Calendar.DAY_OF_YEAR);
					}

					if(!sameDay) {
						//Jazz Id: 230989 - adding HL7v3 Version code in MVI Request Header.
						this.mviIntegrationService.updateTreatmentFacilitiesForExistingPatients(getPatientResponse.getPojoObject(), updateTreatmentFacilitiesResponse);
					}
				} else {
					throw new MHVException(getPatientResponse);
				}
			}
		} catch (MHVException ex) {
			throw ex;
		} catch (Exception ex) {
			String msg = String.format("Error updating treatment facilities for patient: $d", patientId);
			log.error(msg, ex);
			throw new MHVException(msg, ex);
		}

		return patientConverter.convertDomainPatient(updateTreatmentFacilitiesResponse.getPojoObject());
	}
	
	@Override
	public Boolean isTempPasswdExpired(Long userProfileId) throws MHVException {
		Boolean passWordExpired=false;

		try {
			passWordExpired=this.userPasswordService.isTempPasswdExpired(userProfileId);
		}catch(Exception ex) {
			 String errorMessage = "An error was encountered while determining temp password expired or not.";
			 log.error(errorMessage, ex);
			 throw new MHVException(errorMessage, ex);
		}
		return passWordExpired;
	}

	@Override
	public Boolean isLatestPasswordInHistTableTemporay(Long userProfileId) throws MHVException {
		Boolean tempPassword=false;

		try {
			tempPassword=this.userPasswordService.isLatestPasswordInHistTableTemporay(userProfileId);
		}catch(Exception ex) {
			 String errorMessage = "An error was encountered while determining password temporary or not.";
			 log.error(errorMessage, ex);
			 throw new MHVException(errorMessage, ex);
		}
		return tempPassword;
	}

	public Boolean checkVAREligibilityForPatient(Long patientId) throws MHVException {
				int isEligible = patientRepository.varEligibleCheck(patientId);
				System.out.println("*** RESULT : : ***" + " isEligible:"+isEligible);
				boolean result = (isEligible ==0)? false:true;
				return result;
	}
	
	@Override
	public UserProfileDTO changeTemporaryPassword(UserProfileDTO userProfile, Boolean isForgotUserPwd) throws MHVException {
		 UserProfileDTO userProfileDTO=null;
		try {
		     changePassword(userProfile, isForgotUserPwd);
		     if(userProfile.getContact().getEmail() != null &&
		    	!userProfile.getContact().getEmail().isEmpty()) { 
		    	 emailService.send("PII       ", "Department of Veterans Affairs: My HealtheVet", EMAIL_SUBJECT, getEmailBody(), userProfile.getContact().getEmail().toString(), EMAIL_MESSAGE_CONTENT_TYPE);
		     }
		     
		} catch(MHVException e){
			throw e;
		}
		  catch(Exception ex) {
			String errorMessage = "An error was encountered while changing the users password.";
			log.error(errorMessage, ex);
			throw new MHVException(errorMessage, ex);
		}
		return userProfileDTO;
	}
	
	
	private String getEmailBody(){
		StringBuffer sb = new StringBuffer();
		sb.append("This is a courtesy message to notify you of recent activity involving your My HealtheVet password or User ID. If you did not recently change your password or request your User ID, please notify us by using the Contact MHV");
		sb.append(BREAK);
		sb.append("link provided on the My HealtheVet website -").append("<a href='http://www.myhealth.va.gov/'> www.myhealth.DNS   .</a>");
		sb.append(BREAK);
		sb.append("------------------------------------------------------- ");
		sb.append(BREAK);
		sb.append(" **This is an automated message.  Please do not reply** ");
		sb.append(BREAK);
		sb.append("------------------------------------------------------- ");
		sb.append(BREAK);
		sb.append("This e-mail is intended only for the person or entity to which it is addressed, and may contain information that is privileged, confidential, or otherwise protected from disclosure.  Dissemination, distribution, or copying of ");
		sb.append(BREAK);
		sb.append("this e-mail or the information herein by anyone other than the intended recipient or for official internal VHA  and/or VA business is prohibited.");
		
		return sb.toString();
	}

}


