package gov.va.med.mhv.usermgmt.main.service.impl;

import java.util.Iterator;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.usermgmt.common.enums.CorrelationType;
import gov.va.med.mhv.usermgmt.common.enums.DirectionType;
import gov.va.med.mhv.usermgmt.common.enums.PatientType;
import gov.va.med.mhv.usermgmt.common.enums.TracingStatusType;
import gov.va.med.mhv.usermgmt.data.model.MpiEvent;
import gov.va.med.mhv.usermgmt.data.model.Trace;
import gov.va.med.mhv.usermgmt.data.repository.MpiEventsRepository;
import gov.va.med.mhv.usermgmt.main.registry.BaseMessage;
import gov.va.med.mhv.usermgmt.main.registry.CorrelatePatientMessage;
import gov.va.med.mhv.usermgmt.main.registry.CorrelatePatientRequest;
import gov.va.med.mhv.usermgmt.main.registry.EventId;
import gov.va.med.mhv.usermgmt.main.registry.FacilitiesChange;
import gov.va.med.mhv.usermgmt.main.registry.FacilitiesChangeMessage;
import gov.va.med.mhv.usermgmt.main.registry.IntegrationContext;
import gov.va.med.mhv.usermgmt.main.registry.Patient;
import gov.va.med.mhv.usermgmt.main.registry.PatientChangeMessage;
import gov.va.med.mhv.usermgmt.main.registry.PersonalChangeMessage;
import gov.va.med.mhv.usermgmt.main.registry.PersonalInfo;
import gov.va.med.mhv.usermgmt.main.registry.ProcessingHelper;
import gov.va.med.mhv.usermgmt.main.registry.Result;
import gov.va.med.mhv.usermgmt.main.registry.Status;
import gov.va.med.mhv.usermgmt.main.registry.UnexpectedMessageException;
import gov.va.med.mhv.usermgmt.main.registry.UnknownMhvPatientException;
import gov.va.med.mhv.usermgmt.service.EventTracingService;
import gov.va.med.mhv.usermgmt.service.ITraceProcessor;
import gov.va.med.mhv.usermgmt.service.PatientCorrelationService;


@Component
public class PatientCorrelationServiceImpl implements PatientCorrelationService {
	
	private ProcessingHelper processingHelper;
	
	public static final String PATIENT_CORRELATION_SERVICE="patientCorrelationService";
    
    private ITraceProcessor correlateUncorrelateOutBoundTraceProcessor = null;
    
    @Autowired
    private EventTracingService eventTracingService;
    
    @Autowired
    private MpiEventsRepository  mpiEventsRepository;

	/**
     * @return Returns the correlateUncorrelateOutBoundTracePocessor.
     */
    public ITraceProcessor getCorrelateUncorrelateOutBoundTraceProcessor() {
        return correlateUncorrelateOutBoundTraceProcessor;
    }

    /**
     * @param correlateUncorrelateOutBoundTracePocessor 
     * The correlateUncorrelateOutBoundTracePocessor to set.
     */
    public void setCorrelateUncorrelateOutBoundTraceProcessor(
        ITraceProcessor correlateUncorrelateOutBoundTraceProcessor)
    {
        this.correlateUncorrelateOutBoundTraceProcessor = 
            correlateUncorrelateOutBoundTraceProcessor;
    }

   
    private Status correlatePatient(Patient patient, CorrelationType type) {
    	Result result = null;
		CorrelatePatientRequest data = new CorrelatePatientRequest();
		EventId id = new EventId();
		data.setId(id);
		data.setPatient(patient);
		data.setType(type.toString());
		Trace trace = eventTracingService.traceEvent(
				data.getId(), patient.getDfn(), data.getPatient(),
				DirectionType.out.toString(), data);
		try {
			result = correlateUncorrelateOutBoundTraceProcessor.processTrace(
                trace);
		} catch (Exception e) {
			result = processingHelper.processTraceError(trace, e);
		}
		return result.getStatus();
	}

	/* (non-Javadoc)
	 * @see gov.va.med.mhv.usermgmt.main.service.impl.PatientCorrelationService#correlatePatient(gov.va.med.mhv.usermgmt.main.registry.Patient)
	 */
	@Override
	public Status correlatePatient(Patient patient) {
		return correlatePatient(patient, CorrelationType.Correlate);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.mhv.usermgmt.main.service.impl.PatientCorrelationService#uncorrelatePatient(gov.va.med.mhv.usermgmt.main.registry.Patient)
	 */
	@Override
	public Status uncorrelatePatient(Patient patient) {
		return correlatePatient(patient, CorrelationType.Uncorrelate);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.mhv.usermgmt.main.service.impl.PatientCorrelationService#processCorrelateNotification(gov.va.med.mhv.usermgmt.main.registry.EventId, java.lang.String, gov.va.med.mhv.usermgmt.main.registry.Patient, gov.va.med.mhv.usermgmt.main.registry.Status)
	 */
	@Override
	public Result processCorrelateNotification(EventId id, String type,
			Patient patient, Status status) {
		CorrelatePatientMessage data = new CorrelatePatientMessage();
		data.setId(id);
		data.setType(type);
		data.setPatient(patient);
		data.setStatus(status);
		return processInboundTrace(id, patient, data);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.mhv.usermgmt.main.service.impl.PatientCorrelationService#processFacilitiesNotification(gov.va.med.mhv.usermgmt.main.registry.EventId, java.lang.String, gov.va.med.mhv.usermgmt.main.registry.Patient, gov.va.med.mhv.usermgmt.main.registry.FacilitiesChange)
	 */
	@Override
	public Result processFacilitiesNotification(EventId id, String type,
			Patient patient, FacilitiesChange facilitiesChange) {
		FacilitiesChangeMessage data = new FacilitiesChangeMessage();
		data.setId(id);
		data.setType(type);
		data.setPatient(patient);
		data.setFacilitiesChange(facilitiesChange);
		return processInboundTrace(id, patient, data);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.mhv.usermgmt.main.service.impl.PatientCorrelationService#processPersonalNotification(gov.va.med.mhv.usermgmt.main.registry.EventId, java.lang.String, gov.va.med.mhv.usermgmt.main.registry.Patient, gov.va.med.mhv.usermgmt.main.registry.PersonalInfo)
	 */
	@Override
	public Result processPersonalNotification(EventId id, String type,
			Patient patient, PersonalInfo info) {
		PersonalChangeMessage data = new PersonalChangeMessage();
		data.setId(id);
		data.setType(type);
		data.setPatient(patient);
		data.setInfo(info);
		return processInboundTrace(id, patient, data);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.mhv.usermgmt.main.service.impl.PatientCorrelationService#processPatientNotification(gov.va.med.mhv.usermgmt.main.registry.EventId, java.lang.String, gov.va.med.mhv.usermgmt.main.registry.Patient, gov.va.med.mhv.usermgmt.main.registry.Patient)
	 */
	@Override
	public Result processPatientNotification(EventId id, String type,
			Patient fromPatient, Patient toPatient) {
		PatientChangeMessage data = new PatientChangeMessage();
		data.setId(id);
		data.setType(type);
		data.setFromPatient(fromPatient);
		data.setToPatient(toPatient);
		Patient tmpPatient = (type.equals(PatientType.Link.toString()) ? toPatient
				: fromPatient);
		return processInboundTrace(id, tmpPatient, data);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.mhv.usermgmt.main.service.impl.PatientCorrelationService#processInboundTrace(gov.va.med.mhv.usermgmt.main.registry.EventId, gov.va.med.mhv.usermgmt.main.registry.Patient, gov.va.med.mhv.usermgmt.main.registry.BaseMessage)
	 */
	@Override
	public Result processInboundTrace(EventId id, Patient patient,
			BaseMessage data) {
		Result result = null;
		if (data.getType()==null)
		{	
			data.setType("<NULL>");
		}
		if (data.getType().equals(CorrelationType.Correlate.toString())
				|| data.getType()
						.equals(CorrelationType.Uncorrelate.toString())) {
			try
			{
				preProcessApplicationAck(id, patient, data);
			} 
			catch(UnexpectedMessageException ume)
			{
				Trace trace = getProcessingHelper().getTracingService().traceEvent(id,
						patient, DirectionType.in.toString(), data);
				return getProcessingHelper().processTraceReject(trace, ume);
			}
			catch(Exception e)
			{
				// This includes RemoteExceptions
				Trace trace = getProcessingHelper().getTracingService().traceEvent(id,
						patient, DirectionType.in.toString(), data);
				return getProcessingHelper().processTraceFail(trace, e);
			}
		}
		Trace trace = getProcessingHelper().getTracingService().traceEvent(id,
				patient, DirectionType.in.toString(), data);
		String type = data.getType();
		if (id.getId() == null) {
			id.setId(trace.getEvent().getId());
		}
		try {
			String processorId = type.toLowerCase() + "InboundProcessor";
			ITraceProcessor processor = null;
			try {
				processor = (ITraceProcessor) IntegrationContext
						.getBean(processorId);
			} catch (BeansException be) {
				processor = (ITraceProcessor) IntegrationContext
						.getBean("unknownInboundProcessor");
				be.printStackTrace();
			}
			result = processor.processTrace(trace);

			processingHelper.processTrace(trace, TracingStatusType.success, 
					trace.getEvent().getCorrelationId(),
					trace.getEvent().getUserId(), true);
		} catch (UnexpectedMessageException ume) {
			result = processingHelper.processTraceReject(trace, ume);
			ume.printStackTrace();
		} catch (UnknownMhvPatientException umpe) {
			result = processingHelper.processTraceError(trace, umpe);
			umpe.printStackTrace();
		} catch (Exception e) {
			// This includes RemoteExceptions
			result = processingHelper.processTraceFail(trace, e);
			e.printStackTrace();
		}
		return result;
	}
		
	private void preProcessApplicationAck(EventId id, Patient patient, 
        BaseMessage data) throws UnexpectedMessageException
    {
		//in the case of NACK patient is null and status is error
		if(patient==null)
		{
			patient=new Patient();
		}
		// preprocess to correlate to outgoing message
		MpiEvent event = null;
		if (id.getId() != null) {
			event = mpiEventsRepository.findOne(id.getId());
		} else if (id.getCorrelationId() != null) {
			Iterator<MpiEvent> iter = mpiEventsRepository.findByCorrelationId(id.getCorrelationId())
					.iterator();
			event = (iter.hasNext() ? iter.next() : null);
		}
		if (event == null) {
			id.setId(null);
			throw new UnexpectedMessageException("Error correlating to event with id ["
					+ id.getId() + "] and correlationId ["
					+ id.getCorrelationId() + "]");
		}
		// restore event id, so that trace is added to existing event
		id.setId(event.getId());
		Iterator<Trace> iter = event.getTraces().iterator();
		// message does not have ICN inside, restore is from out trace
		while (iter.hasNext() && patient.getIcn()==null) {
			Trace tmp = iter.next();
			BaseMessage bm = (BaseMessage) tmp.deserializeTraceData();
			if (bm instanceof CorrelatePatientRequest) {
				Patient pt = ((CorrelatePatientRequest) bm).getPatient();
				patient.setIcn(pt.getIcn());
				patient.setDfn(event.getUserId());
				((CorrelatePatientMessage)data).setPatient(patient);
			}
		}
	}

	public ProcessingHelper getProcessingHelper() {
		return processingHelper;
	}

	public void setProcessingHelper(ProcessingHelper processingHelper) {
		this.processingHelper = processingHelper;
	}

}
