package gov.va.med.mhv.api.common.service.impl;


import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import org.apache.commons.lang3.StringUtils;
import org.apache.cxf.jaxrs.ext.MessageContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.cache.CacheHandler;
import gov.va.med.mhv.common.api.enumeration.ErrorEnum;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.service.SessionAPIService;
import gov.va.med.mhv.common.api.transfer.ClientApplication;
import gov.va.med.mhv.common.api.transfer.Session;
import gov.va.med.mhv.common.api.util.ClientApplicationHelper;
import gov.va.med.mhv.common.api.util.MHVApiUtility;
import gov.va.med.mhv.common.api.util.MHVEssApiUtility;
import gov.va.med.mhv.common.data.converter.PatientConverter;
import gov.va.med.mhv.common.data.model.UserProfile;
import gov.va.med.mhv.usermgmt.data.repository.PatientRepository;
import gov.va.med.mhv.usermgmt.data.repository.UserProfileRepository;
import gov.va.med.mhv.usermgmt.service.PatientService;
import gov.va.med.mhv.usermgmt.service.UserProfileService;

@Component
public class SessionAPIServiceImpl implements SessionAPIService {
	private static Logger logger = LogManager.getLogger(SessionAPIServiceImpl.class);
	
	private static final String EXPIRES_HEADER_KEY = "Expires";
	private static final String TOKEN_HEADER_KEY = "Token";
	private static final String MHV_CORRELATION_ID_HEADER_KEY = "mhvCorrelationId";
	private static final String MHV_USERNAME_HEADER_KEY = "mhvUsername";
	
	@Context
	MessageContext mc;
	
	@Autowired
	private UserProfileService userProfileService;
	
	@Autowired
	private PatientService patientService;
	
	@Autowired
	private UserProfileRepository userProfileRepository;

	@Autowired
	private PatientRepository patientRepository;
	
	@Autowired
	private PatientConverter patientConverter;
	
//	@Autowired
//	private MviIntegrationService mviIntegrationService;
	
	/**
	 * Session Error Codes Patient:  101,102,103,104,105,106,107,111,132
	 */
	public Response createSession() throws MHVException {
		return checkUser( mc.getHttpHeaders(), mc.getUriInfo() );
	}

	/**
	 * Checks the application token, path authorized for application.
	 *
	 * Makes sure user is matching requested resource. (patient)
	 *
	 * @return
	 */
	private Response checkUser( HttpHeaders httpHeaders, UriInfo uriInfo ) throws MHVException {
		Session session = null;
		
		
		ClientApplication clientApplication = getclientApplicationUsingHeaderValue( httpHeaders, uriInfo );

		//CHECK FOR EITHER KEY PREFER username
		String username = MHVApiUtility.getValueOfHeaderElement(httpHeaders, MHV_USERNAME_HEADER_KEY);
		String correlationId = MHVApiUtility.getValueOfHeaderElement(httpHeaders, MHV_CORRELATION_ID_HEADER_KEY);
		
		if(correlationId == null && username == null) {
			throw new MHVException(ErrorEnum.MISSING_USER_CREDENTIALS_104);
		}
		
		UserProfile userProfile = null;
		
		if( username != null ) {
			userProfile = userProfileRepository.findByUserName(username);
		} else if( correlationId != null ) {
			userProfile = userProfileRepository.getUserProfileById(Long.parseLong(correlationId));
		} 
		if(clientApplication.getUriPathCheck()) {
			if(!clientApplication.isInternal()) {
	
				//Check Application Authentication First
				if( !checkClientSourcePermissions( clientApplication, httpHeaders, uriInfo ) ) {
					MHVEssApiUtility.throwException(ErrorEnum.APPLICATION_AUTHENTICATION_FAILED_101);
				}		
				//Check Application Authorization Second
				if( !checkClientAuthorizationPermissions( clientApplication, httpHeaders, uriInfo ) ) {
					MHVEssApiUtility.throwException(ErrorEnum.APPLICATION_AUTHORIZATION_FAILED_102);
				}
			}
		}	
			if (userProfile == null) {
				throw new MHVException(ErrorEnum.USER_NOT_FOUND_105);
			}
			if (userProfile.getDeactivationReason() != null ) {
				throw new MHVException(ErrorEnum.USER_ACCOUNT_DEACTIVE_113);
			}
			if (userProfile.getLockDate() != null) {
				throw new MHVException(ErrorEnum.INELIGIBLE_USER_REASON_BLOCKED_106);
			}
			//TODO: need to be implemented
	//		if (!hasTermsBeenAccepted(userProfile)) {
	//			throw new MHVException(ErrorEnum.USER_MHV_TC_STALE__114);
	//		}
			if (userProfile.getDeactivationReason() != null ) {
				throw new MHVException(ErrorEnum.USER_ACCOUNT_DEACTIVE_113);
			}
		
		
		// MHV_USERNAME_HEADER_KEY
		
		if( logger.isDebugEnabled() ) {
			logger.debug("Creating a token for this user: " + userProfile.getUserName() + " on behalf of " + clientApplication.getName());
		}

		//USER IS GOOD TO GO!
		session = new Session(userProfile.getId(), clientApplication);

		//Make session available for any caller on this thread
		CacheHandler.getInstance().setSession(session);

		String token = null;
		try {
			token = MHVApiUtility.encrypt(session);
		} catch (Exception e) {
			throw new MHVException(ErrorEnum.UNABLE_TO_CREATE_TOKEN_107);
		}

		return Response.ok().header(TOKEN_HEADER_KEY, token).header(EXPIRES_HEADER_KEY,session.getFormattedTimestamp()).build();
	}


	private boolean checkClientSourcePermissions( ClientApplication client, HttpHeaders httpHeaders, UriInfo uriInfo ) {
		String path = uriInfo.getRequestUri().getPath();
		if (client == null)
			return false;
		for( String permission: client.getPermissions()) {
			if(path.contains(permission)) {
				return true;
			}
		}
		return false;
	}
	
	private boolean checkClientAuthorizationPermissions( ClientApplication client, HttpHeaders httpHeaders, UriInfo uriInfo ) {
		String path = uriInfo.getRequestUri().getPath();
		for( String permission: client.getAuthorizations()) {
			if(path.contains(permission)) {
				return true;
			}
		}
		return false;
	}
	
	private ClientApplication getclientApplicationUsingHeaderValue( HttpHeaders httpHeaders, UriInfo uriInfo ) throws MHVException{
		String appToken = MHVApiUtility.getValueOfHeaderElement(httpHeaders, "appToken");
		ClientApplication source = null;

		if (StringUtils.isEmpty(appToken)) {
			throw new MHVException(ErrorEnum.MISSING_APPLICATION_TOKEN_132);
		} else {
			try {
				source = ClientApplicationHelper.findClientApplication(appToken);
			} catch (Exception e) {
				throw new MHVException(ErrorEnum.APPLICATION_AUTHENTICATION_FAILED_101);
			}
		}

		return source;
	}

	
}

//
// FOR FUTURE CONSUMPTION
//
//private Session checkSession() throws MHVException {
//Session session = null;
//
//String token = MHVApiUtility.getValueOfHeaderElement(httpHeaders, TOKEN_HEADER_KEY);
//if (null != token) {
//	try {
//		session = MHVApiUtility.decrypt(token);
//	} catch (Exception e) {
//		throw new MHVException(ErrorEnum.INVALID_SESSION_TOKEN_109);
//	}
//
//	//Validate that the session has data before proceeding
//	if( session.getExpirationSeconds() < 0 || session.getTimestamp() == null ) {
//		throw new MHVException(ErrorEnum.INVALID_SESSION_TOKEN_109);
//	}
//
//	//Check Application Authentication First
//	if( !checkClientSourcePermissions(session.getClientApplication()) ) {
//		throw new MHVException(ErrorEnum.APPLICATION_AUTHENTICATION_FAILED_101);
//	}
//	//Check Application Authorization First
//	if( !checkClientAuthorizationPermissions(session.getClientApplication()) ) {
//		throw new MHVException(ErrorEnum.APPLICATION_AUTHORIZATION_FAILED_102);
//	}
//
//	// TODO: REDO
////	if( !checkClientUserPermissions(session.getUserTypeEnum())){
////		SMApiUtility.throwException(ErrorEnum.INVALID_USER_PERMISSIONS_111);
////	}
//
//	if (session.isExpired()) {
//		throw new MHVException(ErrorEnum.EXPIRED_SESSION_TOKEN_110);
//	}
//
//	UserProfile userProfile = userProfileRepository.getUserProfileById(session.getUserId());
//	
//	if (userProfile.getLockDate() != null) {
//		throw new MHVException(ErrorEnum.INELIGIBLE_USER_REASON_BLOCKED_106);
//	}
//	if(userProfile.getDeactivationReason() != null) {
//		throw new MHVException(ErrorEnum.USER_ACCOUNT_DEACTIVE_113);
//	}
//	
//	//Make session available for any caller on this thread
//	CacheHandler.getInstance().setSession(session);
//
//} else {
//	throw new MHVException(ErrorEnum.MISSING_SESSION_TOKEN_108);
//}
//
//return session;
//}
//private boolean checkClientAuthorizationPermissions( ClientApplication client, HttpHeaders httpHeaders, UriInfo uriInfo ) {
//	String path = uriInfo.getRequestUri().getPath();
//	for( String permission: client.getAuthorizations()) {
//		if(path.contains(permission)) {
//			return true;
//		}
//	}
//	return false;
//}