package gov.va.med.mhv.usermgmt.service.impl;

import gov.va.med.mhv.core.messages.MessagesUtils;
import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.core.util.ServiceResponseUtils;
import gov.va.med.mhv.service.MHVAbstractService;
import gov.va.med.mhv.usermgmt.bizobj.BusinessObjectFactory;
import gov.va.med.mhv.usermgmt.bizobj.InPersonAuthenticationAssembler;
import gov.va.med.mhv.usermgmt.bizobj.InPersonAuthenticationBO;
import gov.va.med.mhv.usermgmt.bizobj.InPersonAuthenticationCriteriaBO;
import gov.va.med.mhv.usermgmt.bizobj.PatientInformationAssembler;
import gov.va.med.mhv.usermgmt.enumeration.ActivityActorTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.AuthenticationStatus;
import gov.va.med.mhv.usermgmt.enumeration.PatientCorrelationStatus;
import gov.va.med.mhv.usermgmt.messages.UserManagementMessages;
import gov.va.med.mhv.usermgmt.persist.DaoFactory;
import gov.va.med.mhv.usermgmt.persist.PatientInformationQueryDao;
import gov.va.med.mhv.usermgmt.service.InPersonAuthenticationService;
import gov.va.med.mhv.usermgmt.service.InPersonAuthenticationServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientInformationCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientServiceResponse;
import gov.va.med.mhv.usermgmt.service.ServiceFactory;
import gov.va.med.mhv.usermgmt.service.UserProfileServiceResponse;
import gov.va.med.mhv.usermgmt.service.adapter.MpiProperties;
import gov.va.med.mhv.usermgmt.service.handler.MviProperties;
import gov.va.med.mhv.usermgmt.transfer.InPersonAuthentication;
import gov.va.med.mhv.usermgmt.transfer.InPersonAuthenticationCriteria;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.util.Auditor;
import gov.va.med.mhv.usermgmt.util.InPersonAuthenticationStatusUtils;
import gov.va.med.mhv.usermgmt.util.MessageKeys;
import gov.va.med.mhv.usermgmt.util.PHRAccessControl;
import gov.va.med.mhv.usermgmt.util.PatientCorrelationStatusUtils;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.messages.MessageUtils;
import org.tigris.atlas.messages.Messages;
import org.tigris.atlas.service.BooleanServiceResponse;
import org.tigris.atlas.service.EntityServiceResponse;
import org.tigris.atlas.service.ServiceResponse;
import org.tigris.atlas.service.VoidServiceResponse;
import org.tigris.atlas.transfer.TransferObject;

/**
 * Service implementation class for the InPersonAuthentication service
 */
public class InPersonAuthenticationServiceImpl extends MHVAbstractService
    implements InPersonAuthenticationService
{

    private static Log LOG = LogFactory.getLog(
        InPersonAuthenticationServiceImpl.class);

    protected static void copyMessages(ServiceResponse response, 
        TransferObject entity) 
    {
        Precondition.assertNotNull("response", response);
        if (entity != null) { 
            response.getMessages().addMessages(entity.getAllMessages());
        }
    }

    public InPersonAuthenticationServiceResponse getAuthenticationForPatient(
        Long patientId)
    {
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();

        Collection bos = InPersonAuthenticationBO
            .getAuthenticationForPatient(patientId);
        if (!bos.isEmpty()) {
            Collection tos = InPersonAuthenticationAssembler
                .getInPersonAuthenticationCollection(bos);
            InPersonAuthentication auth = (InPersonAuthentication) tos
                .iterator().next();
            response.setInPersonAuthentication(auth);
        } else {
            InPersonAuthentication auth = TransferObjectFactory
                .createInPersonAuthentication();
            auth.setStatus(AuthenticationStatus
                .getEnum(AuthenticationStatus.UNAUTHENTICATED));
            auth.setVideoViewed(Boolean.FALSE);
            auth.setParticipationFormSigned(Boolean.FALSE);
            auth.setIdentificationPresented(Boolean.FALSE);
            auth.setApprovedForRecordsAccess(Boolean.FALSE);
            Patient patient = TransferObjectFactory.createPatient();
            patient.setId(patientId);
            auth.setPatient(patient);

            AuthenticationStatus status = AuthenticationStatus
                .getEnum(AuthenticationStatus.UNAUTHENTICATED);
            response.setInPersonAuthentication(saveIPA(auth, status));
        }

        return response;
    }

    /**
     * Execute the DeferAuthentication service
     * @return InPersonAuthenticationServiceResponse
     */
    public InPersonAuthenticationServiceResponse deferAuthentication(
        InPersonAuthentication ipa)
    {
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();

        Boolean formSigned = ipa.getParticipationFormSigned() == null 
            ? Boolean.FALSE : ipa.getParticipationFormSigned();
        Boolean idPresented = ipa.getIdentificationPresented() == null 
            ? Boolean.FALSE : ipa.getIdentificationPresented();
        AuthenticationStatus status = null;
        if (formSigned.booleanValue() && idPresented.booleanValue()) {
            status = AuthenticationStatus
                .getEnum(AuthenticationStatus.PREREQUISITESCOMPLETE);
        } else {
            status = AuthenticationStatus
                .getEnum(AuthenticationStatus.INPROCESS);
        }
        ipa.setRemovalReason(null);
        ipa = saveIPA(ipa, status);
        response.setInPersonAuthentication(ipa);
        Auditor.auditIPAInProcessEvent(ipa,
            ActivityActorTypeEnumeration.MHV_AUTHENTICATOR);
        return response;
    }

    /**
     * Execute the TerminateAuthentication service
     * @return InPersonAuthenticationServiceResponse
     */
    public InPersonAuthenticationServiceResponse terminateAuthentication(
        InPersonAuthentication ipa)
    {
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();

        AuthenticationStatus status = AuthenticationStatus
            .getEnum(AuthenticationStatus.UNAUTHENTICATED);
        ipa.setParticipationFormSigned(Boolean.FALSE);
        ipa.setIdentificationPresented(Boolean.FALSE);
        ipa.setApprovedForRecordsAccess(Boolean.FALSE);
        ipa.setDefermentReason(null);
        ipa.setAuthenticatingFacility(null);
        ipa.setRemovalReason(null);
        ipa.setAuthenticationDate(null);
        ipa.setAuthenticatedBy(null);
        ipa.setMviAuthenticationStatus(null);
        ipa = saveIPA(ipa, status);
        response.setInPersonAuthentication(ipa);

        return response;
    }

    /**
     * Execute the FinalizeUnauthentication service
     * @return InPersonAuthenticationServiceResponse
     */
    public InPersonAuthenticationServiceResponse finalizeUnauthentication(
        InPersonAuthentication ipa, String error)
    {
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();

        if (!InPersonAuthenticationStatusUtils.isPendingRemoval(ipa)) {
            addError(response, MessageKeys.PATIENT_IS_NOT_PENDING_UNCORRELATION);
        } else if (!StringUtils.isBlank(error)) {
            ipa = saveIPA(ipa, InPersonAuthenticationStatusUtils.
                REMOVAL_FAILED);
            LOG.error("Unauthentication failed for patient " +
                describePatient(ipa) + ", because MPI responded with " + error 
                + ". Status changed to " + describeStatus(ipa));
        } else {
            ipa = saveIPA(ipa, InPersonAuthenticationStatusUtils.
                UNAUTHENTICATED);
            copyMessages(response, Auditor.auditIPAUnauthenticatedEvent(ipa,
                ActivityActorTypeEnumeration.MHV_AUTHENTICATOR));
        }
        response.setInPersonAuthentication(ipa);

        return response;
    }

    public InPersonAuthenticationServiceResponse mviUnauthenticate(
            InPersonAuthentication ipa) {
        Precondition.assertNotNull("ipa", ipa);
        Precondition.assertNotNull("ipa.patient", ipa.getPatient());
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();
        AuthenticationStatus status = InPersonAuthenticationStatusUtils.UNAUTHENTICATED;
        ipa.setParticipationFormSigned(Boolean.FALSE);
        ipa.setIdentificationPresented(Boolean.FALSE);
        ipa.setApprovedForRecordsAccess(Boolean.FALSE);
        ipa.setDefermentReason(null);
        ipa.setAuthenticatingFacility(null);
        ipa.setAuthenticationDate(null);
		//JAZZ: Task#19823 - When a IPA user unauthenticates a user that is not matched and/or 
        //Correlated, then the system will not send the unauth flags to MVI since the user is not correlated
    	Patient patient = ipa.getPatient();
    	if(patient.getCorrelationStatus() == PatientCorrelationStatus.getEnum(PatientCorrelationStatus.UNCORRELATED)||
    	   patient.getCorrelationStatus() == PatientCorrelationStatus.getEnum(PatientCorrelationStatus.MVIDATAMATCH)) {
		   ipa.setMviAuthenticationStatus("OK");
	       patient.setPatientSynchronizations(null);
	       ServiceFactory.createEntityMaintenanceService().save(patient);
    	} else {
			//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - If the MVI flag is turned on call, MVI call instead of MPI call
            BooleanServiceResponse patientServiceResponse = null;
			ipa.setMviAuthenticationStatus("PENDING_UNAUTH");
			//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - If the MVI flag is turned on call, MVI call instead of MPI call
			patientServiceResponse = ServiceFactory.
            createPatientService().mviUnauthenticate(ipa.getPatient());
            if (patientServiceResponse.getBoolean()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("InPersonAuthenticationServiceImpl.mviUnauthenticate():SUCCESS unauthenticating patient with ICN:" + ipa.getPatient().getIcn());
                }
                
                //JAZZ: Task#19820 - Remove un-correlation call when user�s account is unauthenticated
    			ipa.setMviAuthenticationStatus("OK");
    			//MHV_CodeCR1918 - US12.4.x MVI Compliance Implementation - If the patient status is correleated or pending corr/uncorr
                //try to uncorrelate. Only 
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("InPersonAuthenticationServiceImpl.mviUnauthenticate():ERROR unauthenticating patient with ICN:" + ipa.getPatient().getIcn());
                }
            }
    	}
        ipa = saveIPA(ipa, status);
        response.setInPersonAuthentication(ipa);
        return response; 
    }
    
	private String getPatientFullName(Patient patient) {
		StringBuffer fullName = new StringBuffer(patient.getUserProfile().getFirstName());

		//Admin Portal_CodeCR1789 - Make sure that there is a space in between First and Last Names.
		if(patient.getUserProfile().getMiddleName()!=null) {
			fullName.append(" " + patient.getUserProfile().getMiddleName());
		}
		fullName.append(" "	+ patient.getUserProfile().getLastName());
		return fullName.toString();
	}

	/*
	 * JAZZ#112876 - US#4 Admin Portal Enhancements - Force Deactivation
	 */
    public InPersonAuthenticationServiceResponse forceUnauthenticateUnCorrelate(InPersonAuthentication ipa) {
        Precondition.assertNotNull("ipa", ipa);
        Precondition.assertNotNull("ipa.patient", ipa.getPatient());

        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();
        AuthenticationStatus status = InPersonAuthenticationStatusUtils.UNAUTHENTICATED;
        ipa.setParticipationFormSigned(Boolean.FALSE);
        ipa.setIdentificationPresented(Boolean.FALSE);
        ipa.setApprovedForRecordsAccess(Boolean.FALSE);
        ipa.setDefermentReason(null);
        ipa.setAuthenticatingFacility(null);
        ipa.setAuthenticationDate(null);
		ServiceFactory.createPatientService().mviUnauthenticate(ipa.getPatient());
    	ipa.setMviAuthenticationStatus(null);
        ipa = saveIPA(ipa, status);
        response.setInPersonAuthentication(ipa);
        
        ServiceFactory.createMviIntegrationService().forceUncorrelation(ipa.getPatient(), ipa.getPatient().getUserProfile(), "");
		return response; 
    }	
	
    /**
     * Execute the Unauthenticate service
     * @return InPersonAuthenticationServiceResponse
     */
    public InPersonAuthenticationServiceResponse unauthenticate(InPersonAuthentication ipa) {
        Precondition.assertNotNull("ipa", ipa);
        Precondition.assertNotNull("ipa.patient", ipa.getPatient());

		MviProperties mviProp = MviProperties.getInstance();
		if(mviProp.getIsMviEnabled()) {
			return mviUnauthenticate(ipa);
		}
		
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();
        
        boolean isEnabled = MpiProperties.getInstance().isUnlinkEnabled();
        boolean isCorrelated = PatientCorrelationStatusUtils.isCorrelated(
            ipa.getPatient());
        
        if (isEnabled 
            && !(isCorrelated || isPendingSynchronization(ipa.getPatient()))) 
        {
            addError(response, MessageKeys.CANNOT_UNAUTHENTICATE_PATIENT);
            return response;
        }
        InPersonAuthentication originalIpa = copyProperties(ipa, null);
        AuthenticationStatus status = (isEnabled && isCorrelated) 
            ? InPersonAuthenticationStatusUtils.PENDING_REMOVAL 
            : InPersonAuthenticationStatusUtils.UNAUTHENTICATED;
        ipa.setParticipationFormSigned(Boolean.FALSE);
        ipa.setIdentificationPresented(Boolean.FALSE);
        ipa.setApprovedForRecordsAccess(Boolean.FALSE);
        ipa.setDefermentReason(null);
        ipa.setAuthenticatingFacility(null);
        ipa.setAuthenticationDate(null);
        ipa = saveIPA(ipa, status);

        response.setInPersonAuthentication(ipa);
        copyMessages(response, ipa);
        if (!hasErrorMessages(response)) {
            VoidServiceResponse patientServiceResponse = ServiceFactory.
                createPatientService().uncorrelate(ipa.getPatient());
            copyMessages(response, patientServiceResponse);
        }
        if (hasErrorMessages(response)) {
            ipa = copyProperties(originalIpa, ipa); 
        }
        return response; 
    }
    
    private boolean isPendingSynchronization(Patient patient) { 
        return (patient != null) 
            && (patient.getPatientSynchronizations() != null)
            && (!patient.getPatientSynchronizations().isEmpty());
    }
    
    private InPersonAuthentication copyProperties(InPersonAuthentication ipa, 
        InPersonAuthentication copyTo) 
    {
        if (copyTo == null) {
            copyTo = TransferObjectFactory.createInPersonAuthentication();
        }
        copyTo.setAuthenticatingFacility(ipa.getAuthenticatingFacility());
        copyTo.setAuthenticationDate(ipa.getAuthenticationDate());
        copyTo.setStatus(ipa.getStatus());
        copyTo.setApprovedForRecordsAccess(ipa.getApprovedForRecordsAccess());
        copyTo.setIdentificationPresented(ipa.getIdentificationPresented());
        copyTo.setParticipationFormSigned(ipa.getParticipationFormSigned());
        copyTo.setDefermentReason(ipa.getDefermentReason());
        return copyTo;
    }

    /**
     * Execute the FindPatients service
     * @return PatientInformationCollectionServiceResponse
     */
    public PatientInformationCollectionServiceResponse findPatients(
        InPersonAuthenticationCriteria criteria)
    {
        PatientInformationCollectionServiceResponse response = 
            new PatientInformationCollectionServiceResponse();

        if (InPersonAuthenticationCriteriaBO.isEmpty(criteria)) {
            response.getMessages().addMessage(
                MessageUtils.createErrorMessage("empty.search.critiera", null,
                    null));
            return response;
        }

        PatientInformationQueryDao dao = (PatientInformationQueryDao) 
            DaoFactory.createDao(PatientInformationQueryDao.NAME);
        Collection bos = dao.findPatients(criteria);
        Collection tos = PatientInformationAssembler
            .getPatientInformationCollection(bos);
        response.addItems(tos);

        return response;
    }

    /**
     * Execute the AuthenticateMany service
     * @return VoidServiceResponse
     */
    public VoidServiceResponse authenticateMany(Collection patientIds) {
        VoidServiceResponse response = new VoidServiceResponse();

        for (Iterator i = patientIds.iterator(); i.hasNext();) {
            Long id = (Long) i.next();
            InPersonAuthenticationServiceResponse resp = 
                getAuthenticationForPatient(id);
            if (resp.getMessages().hasErrorMessages()) {
                response.getMessages().addMessages(resp.getMessages());
            }
            InPersonAuthentication ipa = resp.getInPersonAuthentication();
            ipa.setApprovedForRecordsAccess(Boolean.TRUE);

            if (ipa != null) {
                InPersonAuthenticationServiceResponse resp2 = authenticate(ipa);
                if (resp2.getMessages().hasErrorMessages()) {
                    response.getMessages().addMessages(resp2.getMessages());
					//Admin Portal_CodeCR1898 - Added Audit for auth many
        			Auditor.auditMviAuthenticateEvent(ipa.getPatient().getUserProfile().getId(), 
        					ActivityActorTypeEnumeration.MHV_AUTHENTICATOR, false, 
        					ipa.getAuthenticatingFacility()!=null?ipa.getAuthenticatingFacility().getName():"");
                } else {
					//Admin Portal_CodeCR1898 - Added Audit for auth many
        			Auditor.auditMviAuthenticateEvent(ipa.getPatient().getUserProfile().getId(), 
        					ActivityActorTypeEnumeration.MHV_AUTHENTICATOR, true, 
        					ipa.getAuthenticatingFacility()!=null?ipa.getAuthenticatingFacility().getName():"");
                }
                if (resp2.getInPersonAuthentication().getAllMessages()
                    .hasErrorMessages())
                {
                    response.getMessages().addMessages(
                        resp2.getInPersonAuthentication().getAllMessages());
                }
            }
        }

        return response;
    }

    public InPersonAuthenticationServiceResponse mviAuthenticate(
            InPersonAuthentication ipa) {
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();
        AuthenticationStatus status = InPersonAuthenticationStatusUtils.AUTHENTICATED;
        ipa.setAuthenticationDate(new Timestamp(new Date().getTime()));
        
		//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - If the MVI flag is turned on call, MVI call instead of MPI call
		BooleanServiceResponse patientServiceResponse = ServiceFactory.
        createPatientService().mviAuthenticate(ipa.getPatient());
        if (patientServiceResponse.getBoolean()) {
        	ipa.setMviAuthenticationStatus("OK");
			if(LOG.isInfoEnabled()){
				LOG.info("********** PatientService().mviAuthenticate ipa.setMviAuthenticationStatus:OK");
			}
        } else {
        	ipa.setMviAuthenticationStatus("PENDING_AUTH");
			if(LOG.isInfoEnabled()){
				LOG.info("********** PatientService().mviAuthenticate ipa.setMviAuthenticationStatus:PENDING_AUTH");
			}
        }

		if(LOG.isInfoEnabled()){
			LOG.info("********** PatientService().mviAuthenticate ipa.setMviAuthenticationStatus:" + ipa.getMviAuthenticationStatus());
		}
		
        ipa = saveIPA(ipa, status);
        response.setInPersonAuthentication(ipa);
        return response; 
	}

    /**
     * Execute the Authenticate service
     * @return InPersonAuthenticationServiceResponse
     */
    public InPersonAuthenticationServiceResponse authenticate(
        InPersonAuthentication ipa) {
    	
        Precondition.assertNotNull("ipa", ipa);
        Precondition.assertNotNull("ipa.patient", ipa.getPatient());
        
		MviProperties mviProp = MviProperties.getInstance();
		if(mviProp.getIsMviEnabled()) {
			return mviAuthenticate(ipa);
		}
		
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();
        
        BooleanServiceResponse accessResponse = PHRAccessControl.hasPhrAccess(
                ipa.getPatient());
        if (hasErrorMessages(accessResponse)) {
            copyMessages(response, accessResponse);
            return response;
        }
        boolean hasPhrAccess = BooleanUtils.isTrue(accessResponse.getBoolean());
        AuthenticationStatus oldStatus = ipa.getStatus(); 
        AuthenticationStatus status = 
            (MpiProperties.getInstance().isLinkEnabled() && hasPhrAccess)
            ? InPersonAuthenticationStatusUtils.PENDING_AUTHENTICATION
            // No correlation to MPI, 
            // so immediately move patient to authenticated status
            : InPersonAuthenticationStatusUtils.AUTHENTICATED;
        ipa.setAuthenticationDate(new Timestamp(new Date().getTime()));
        ipa = saveIPA(ipa, status);
        response.setInPersonAuthentication(ipa);
        if (!hasErrorMessages(response)) {
            VoidServiceResponse patientServiceResponse = ServiceFactory.
                createPatientService().correlate(ipa.getPatient());
            copyMessages(response, patientServiceResponse);
            convertMpiToIPAErrors(response);
        }
        if (hasErrorMessages(response)) {
            // Undo changes
            ipa.setStatus(oldStatus);
            ipa.setAuthenticationDate(null);
        }
        return response; 
    }
    
    /**
     * Execute the FinalizeAuthentication service
     * @return InPersonAuthenticationServiceResponse
     */
    public InPersonAuthenticationServiceResponse finalizeAuthentication(
        InPersonAuthentication ipa, String error)
    {
        InPersonAuthenticationServiceResponse response = 
            new InPersonAuthenticationServiceResponse();

        if (!InPersonAuthenticationStatusUtils.isPendingAuthentication(ipa)) {
            LOG.error("Authentication failed, because patient " +
                describePatient(ipa) + " is not pending authentication");
        } else if (!StringUtils.isBlank(error)) {
            ipa = saveIPA(ipa, InPersonAuthenticationStatusUtils.
                PENDING_FAILED);
            LOG.error("Authentication failed for patient " +
                describePatient(ipa) + ", because MPI responded with " + error 
                + ". Status changed to " + describeStatus(ipa));
        } else {
            ipa.setDefermentReason(null);
            ipa.setRemovalReason(null);
            ipa = saveIPA(ipa, InPersonAuthenticationStatusUtils.AUTHENTICATED);
            Auditor.auditIPAAuthenticatedEvent(ipa,
                ActivityActorTypeEnumeration.MHV_AUTHENTICATOR);
        }

        response.setInPersonAuthentication(ipa);
        
        return response;
    }
    
    public VoidServiceResponse saveIPA(InPersonAuthentication ipa) {
        VoidServiceResponse response = new VoidServiceResponse();
    	
        InPersonAuthenticationBO bo = BusinessObjectFactory
        .createInPersonAuthenticationBO();
        ipa = bo.saveAndUpdateStatus(ipa, ipa.getStatus());
        copyMessages(response, ipa);
        return response;
    }
    
    private InPersonAuthentication saveIPA(InPersonAuthentication ipa,
        AuthenticationStatus status)
    {
        InPersonAuthenticationBO bo = BusinessObjectFactory
            .createInPersonAuthenticationBO();
        ipa = bo.saveAndUpdateStatus(ipa, status);
        return ipa;
    }

    /**
     * @see gov.va.med.mhv.usermgmt.service.InPersonAuthenticationService#
     * isPatientAuthenticated(java.lang.Long)
     */
    public BooleanServiceResponse isPatientAuthenticated(Long patientId) {
        BooleanServiceResponse response = new BooleanServiceResponse();

        InPersonAuthenticationServiceResponse ipaResponse = 
            getAuthenticationForPatient(patientId);
        MessagesSet ipaResponseMessages = ipaResponse.getMessages();
        if (ipaResponseMessages.hasErrorMessages()) {
            response.setBoolean(Boolean.FALSE);
            LOG.error("Could not get IPA status for patient '" + patientId
                + "'!");
        } else {
            InPersonAuthentication ipaStatus = ipaResponse
                .getInPersonAuthentication();
            AuthenticationStatus status = ipaStatus.getStatus();
            AuthenticationStatus authenticated = AuthenticationStatus
                .getEnum(AuthenticationStatus.AUTHENTICATED);
            if ((status != null) && (authenticated.equals(status))) {
                response.setBoolean(Boolean.TRUE);
            } else {
                response.setBoolean(Boolean.FALSE);
            }
        }

        return response;
    }

    public PatientServiceResponse getIPAedPatientForUser(String userName) {
        Precondition.assertNotBlank("userName", userName);
        PatientServiceResponse response = new PatientServiceResponse();
        response.setPatient(null);
        
        UserProfileServiceResponse upResponse = ServiceFactory
            .createUserProfileService().getProfileForUser(userName);
        if (hasErrorMessages(upResponse)) {
            copyMessages(response, upResponse);
            return response;
        }

        PatientServiceResponse pResponse = ServiceFactory.
            createPatientService().getPatientForUser(upResponse.
            getUserProfile());
        if (hasErrorMessages(pResponse)) {
            copyMessages(response, pResponse);
            return response;
        }
        
        Patient patient = pResponse.getPatient();
        if (patient == null) {
            addError(response, MessageKeys.USER_NOT_A_PATIENT, 
            	new Object[] { userName });
            return response;
        }
        BooleanServiceResponse ipaResponse = isPatientAuthenticated(
            patient.getId());
        if (hasErrorMessages(ipaResponse)) {
            copyMessages(response, ipaResponse);
            return response;
        }
        if (BooleanUtils.isTrue(ipaResponse.getBoolean())) {
            response.setPatient(patient);
        }

        return response;
    }

    /**
     * Convert MPI messages, such that message relate to the IPA process
     * and not to the registration process.
     * @param response The response to check
     */
    private void convertMpiToIPAErrors(EntityServiceResponse response) {
        // Messages are mutially exclusive
        if (ServiceResponseUtils.findAndConvertMessage(response, 
                UserManagementMessages.PATIENT_REGISTRY_ACCESS_ERROR,
                MessagesUtils.ERROR_SEVERITY, MessageKeys.
                MPI_ACCESS_PROBLEM_AT_IPA, null))
        {
            return;
        }
        if (ServiceResponseUtils.findAndConvertMessage(response, 
                MessageKeys.PATIENT_NOT_FOUND, MessagesUtils.ERROR_SEVERITY,
                MessageKeys.PATIENT_NOT_FOUND_AT_IPA, null))
        {
            return;
        }
        ServiceResponseUtils.findAndConvertMessage(response, MessageKeys.
            PATIENT_DUPLICATE_ENTRIES, MessagesUtils.ERROR_SEVERITY, 
            MessageKeys.DUPLICATE_PATIENT_ENTRY_AT_IPA, null);
    }

    private String describePatient(InPersonAuthentication ipa) {
        if (ipa == null) {
            return "<unknown:ipa>";
        }
        if (ipa.getPatient() != null) {
            return "<unknown:ipa.patient>";
        }
        if (ipa.getPatient().getUserProfile() != null) {
            return "<unknown:ipa.patient.userProfile>";
        }
        return "'" + ipa.getPatient().getUserProfile().getUserName() + "'";
    }

    private String describeStatus(InPersonAuthentication ipa) {
        if (ipa == null) {
            return "<unknown:ipa>";
        }
        if (ipa.getStatus() != null) {
            return "<unknown:ipa.status>";
        }
        return "'" + ipa.getStatus().getName() + "'";
    }

	public InPersonAuthenticationServiceResponse getIPAPatientById(Long patientId) {
        InPersonAuthenticationServiceResponse response = new InPersonAuthenticationServiceResponse();
        Collection bos = InPersonAuthenticationBO.getAuthenticationForPatient(patientId);
        if (!bos.isEmpty()) {
        Collection tos = InPersonAuthenticationAssembler
            .getInPersonAuthenticationCollection(bos);
        InPersonAuthentication authPatient = (InPersonAuthentication) tos.iterator().next();
        response.setInPersonAuthentication(authPatient);
        } 
        return response;
	}

}