package gov.va.med.mhv.core.validate;

import gov.va.med.mhv.core.xml.LoggingErrorHandler;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.digester.Digester;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.resolver.CatalogManager;
import org.apache.xml.resolver.tools.CatalogResolver;
import org.xml.sax.EntityResolver;
import org.xml.sax.SAXException;

public final class ValidatorFactory {

	private static final ValidatorFactory INSTANCE;
	private static final String ENTITY_RESOLVER_PROPERTIES = "entityResolver.properties";
	private static final String VALIDATORS = "/validators.xml";
	private static final Log log = LogFactory.getLog( ValidatorFactory.class );
	
	static {
		INSTANCE = new ValidatorFactory();
	}
	
	public static ValidatorFactory getInstance() {
		return INSTANCE;
	}
	
	private Map validatorClasses;
	
	private ValidatorFactory() {
		super();
		
		validatorClasses = new HashMap();
		
		loadValidators();
	}
	
	private void loadValidators() {
		CatalogManager catalogManager = new CatalogManager( ENTITY_RESOLVER_PROPERTIES );
		CatalogResolver resolver = new CatalogResolver(catalogManager);
		LoggingErrorHandler errorHandler = new LoggingErrorHandler(log);
		Digester digester = getNewDigester(resolver, new File( VALIDATORS ), errorHandler);
		
		digester.push( this );
		digester.addCallMethod( "validators/validator", "addValidator", 2 );
		digester.addCallParam( "validators/validator", 0, "name");
		digester.addCallParam( "validators/validator", 1, "class");
				
		try {
			digester.parse( ValidatorFactory.class.getResourceAsStream( VALIDATORS ) );
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		}
	}
	
	private Digester getNewDigester(EntityResolver resolver, File fileToParse, LoggingErrorHandler errorHandler) {
		//Digester's javadoc specifically suggests that a new instance
		//should be created even though Digester has a clear() method:
		Digester digester = new Digester();
		digester.setEntityResolver(resolver);
		digester.setValidating(true);
		errorHandler.setFileName(fileToParse.getName());
		digester.setErrorHandler(errorHandler);
		return digester;
	}
	
	public Validator getValidator(String name) {
		Class clazz = (Class) validatorClasses.get(name);
		
		try {
			return (Validator) clazz.newInstance();
		} catch (Exception ex) {
			throw new RuntimeException("Unable to create validator for name: " + name, ex);
		}
	}
	
	public void addValidator(String name, String sClass) {
		try {
			Class clazz = Class.forName(sClass);
			validatorClasses.put(name, clazz);
		} catch (ClassNotFoundException e) {
			throw new RuntimeException("Unable to locate validator class: " + sClass, e);
		}
	}
	
}
