package gov.va.med.mhv.sm.service.impl;

import java.math.BigInteger;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.net.URL;
import java.net.MalformedURLException;
import javax.annotation.Resource;
import javax.xml.ws.BindingProvider;
import java.security.NoSuchAlgorithmException;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.PropertySource;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.core.crypto.MHVCipher;
import gov.va.med.mhv.core.crypto.ChecksumCalculator;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.common.api.dto.PatientDTO;
import gov.va.med.mhv.common.api.dto.FacilityDTO;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.common.api.exception.MHVException;
//import gov.va.med.mhv.sm.model.AbstractLoginInfo;
import gov.va.med.mhv.sm.model.PatientLoginInfo;
import gov.va.med.mhv.sm.model.SecureMessagingSettings;
import gov.va.med.mhv.sm.data.model.UserAcceptTermsHistory;
import gov.va.med.mhv.sm.data.repository.UserAcceptTermsHistoryRepository;
import gov.va.med.mhv.sm.doa.SecureMessagingProperties;
import gov.va.med.mhv.sm.delegate.SMAuthenticationDelegate;
import gov.va.med.mhv.sm.exception.MHVRuntimeException;

import gov.va.med.mhv.sm.service.MHVSMBridgeService;
import gov.va.med.mhv.usermgmt.service.UserMgmtService;


@Component
public class MHVSMBridgeServiceImpl implements MHVSMBridgeService {

	private static Logger log = LogManager.getLogger(MHVSMBridgeServiceImpl.class);

	private static SecureMessagingProperties properties = new SecureMessagingProperties();
	private SecureMessagingSettings settings = null;
	private SMAuthenticationDelegate delegate = null;

	@Autowired
	private UserAcceptTermsHistoryRepository userAcceptTermHistRepository;

	@Resource(name="userMgmtServiceProxy")
	private UserMgmtService userMgmtService;
	
	/**
	 * Constructor initialize the properties and 
	 * the Secure Messaging Authentication delegate.
	 */
	public MHVSMBridgeServiceImpl() {
		settings = properties.getSettings();
		delegate = createDelegate();
	}

	@SuppressWarnings("unused")
	@Override
	public String getAuthenticatePatient(String userName) {
		long start = System.currentTimeMillis();

		PatientLoginInfo loginInfo = null;

		try {
			UserProfileDTO userProfileDto = getCurrentUserProfile(userName);

			if (userProfileDto != null) {
				PatientDTO patientDto = getPatientForUser(userProfileDto.getId());
				
				if(patientDto != null) {
					loginInfo = createPatientLoginInfo(userProfileDto, patientDto);
					
					if(log.isInfoEnabled()){
						log.info("=====inside getAuthenticatePatient()========== start time:<<< " + userName + ">>>: "+ start);
					}					
					
					if (loginInfo != null) {
						String checksum = calculateChecksum(loginInfo);
						if (checksum == null) {
							loginInfo = null;
						} else {
							loginInfo.setChecksum(checksum);
							loginInfo.setSource(getSource());
							loginInfo.setRequestUrl(getRequestUrl());
							loginInfo.setAuthenticationURL(getSettings().getAuthenticationURL());
							loginInfo.setAuthenticationKey(getSettings().getAuthenticationKey());
						}
					}else{
						return createFailureForward("loginInfo is null:<<< " + userName + ">>>");
					}
				}else{
					return createFailureForward("patientDto is null:<<< " + userName + ">>>");
				}

				if (delegate == null) {
					return createFailureForward("Ignore unexpected call to login action:<<< " + userName + ">>>");
				}
				
				if (!authenticate(loginInfo)) {
					return createFailureForward("Failed to authenticate " + describeUser(loginInfo));
				}
			}else{
					return createFailureForward("userProfileDto is null:<<< " + userName + ">>>");
			}
		} catch (Exception e) {
			log.error("Error in createUserIdentity<<<" + userName + ">>>: ", e);
			throw new MHVRuntimeException("Error in createUserIdentity: ", e);
		}

		if (log.isInfoEnabled()) {
			log.info("===end getAuthenticatePatient=== elapsed time " + (System.currentTimeMillis() - start) + "for the user id:<<<" + userName + ">>>");
		}

		return loginInfo.getAuthenticationKey();
	}

	private SMAuthenticationDelegate createDelegate() {
    	String authenticationURL = settings.getAuthenticationURL();
    	if (authenticationURL != null) {
			try {
				return new SMAuthenticationDelegate(settings);
			} catch (MalformedURLException e) {
				// Force error to be logged
				createURL("authentication", authenticationURL);
			}
    	}
    	if (log.isDebugEnabled()) {
    		log.debug("No authenticationURL has been specified. " +
    			"Not using MHVAuthentication web service.");
    	}
    	return null;
	}

	protected PatientLoginInfo createPatientLoginInfo(UserProfileDTO userProfileDto, PatientDTO patientDto) {

		if(log.isDebugEnabled()){
			log.debug(">>>>> 101 - createLoginInfo -getUserName::"+ userProfileDto.getUserName());
		}

		PatientLoginInfo loginInfo = new PatientLoginInfo();

		loginInfo.setUserId(userProfileDto.getUserName());
		loginInfo.setFirstName((userProfileDto.getName().getFirstName() != null) ? userProfileDto.getName().getFirstName() : "");
		loginInfo.setLastName(userProfileDto.getName().getLastName());
		// Verify encryptIfRequired method for WL 12c Ciphers
		//loginInfo.setSsn(encryptIfRequired(userProfileDto.getSsn()));
		loginInfo.setSsn(userProfileDto.getSsn());
		loginInfo.setDob(userProfileDto.getBirthDate());
		loginInfo.setEmail((userProfileDto.getContact().getEmail() == null) ? "" : userProfileDto.getContact().getEmail());
		// Verify encryptIfRequired method for WL 12c Ciphers
		//loginInfo.setIcn(encryptIfRequired(patientDto.getIcn()));
		loginInfo.setIcn(patientDto.getIcn());
		Set<FacilityDTO> facilities = (Set<FacilityDTO>) patientDto.getFacilities();
			// "unchecked"
		if (facilities != null) {
			for (FacilityDTO facility: facilities) {
				loginInfo.addFacilityValue(facility.getFacilityInfo());
					if(log.isInfoEnabled()){
						log.info("User facility station number in MHV-SM Bridge:  " + facility.getFacilityInfo().getStationNumber() + " for the patient ICN: "
												+ patientDto.getIcn() );
				}
			}
		}

		//CR 1160 - Ken So - adding SM T&C info
		//Setting the SM T&C acceptance flag and timestamp
		loginInfo.setTcAccepted((userProfileDto.getAcceptSMTerms()));
		if(log.isDebugEnabled()){
			log.debug("SM T&C Flag:" + loginInfo.getTcAccepted());
		}
		if(loginInfo.getTcAccepted()!= null && loginInfo.getTcAccepted()) {
			List<UserAcceptTermsHistory> userAcceptTermsHistories = null;

			try{
				userAcceptTermsHistories = userAcceptTermHistRepository.getUserAcceptTermsHistByUserProfileId(userProfileDto.getId());
			} catch(Exception e) {
				if(userProfileDto!=null){
					log.error("Error in fetching UserAcceptTermsHistory<<<" + userProfileDto.getId(), e);
				}
				throw new MHVRuntimeException("Error in fetching UserAcceptTermsHistory " + e.getMessage(), e);
			}
			// Get the userUpdatedTime from the latest HIstory record.
			for (UserAcceptTermsHistory userAcceptTermsHistory : userAcceptTermsHistories) {
				loginInfo.setTcAcceptedTimeStamp(new java.sql.Timestamp(userAcceptTermsHistory.getUserUpdatedTime().getTime()));
				if(log.isDebugEnabled()){
					log.debug("SM T&C Timestamp:" + loginInfo.getTcAcceptedTimeStamp());
				}

				break;
			}
		}
		return loginInfo;
	}

	private String calculateChecksum(PatientLoginInfo info) {
		try {
			final String input = info + getSettings().getSeed();
			return (new ChecksumCalculator()).calculate(input);
		} catch (NoSuchAlgorithmException e) {
			log.error("Failed to create the default check sum calculator. "
				+ ChecksumCalculator.DEFAULT_ALGORITHM
				+ " provider must not be available in used JVM", e);
			//addUnexpectedError(DATA_SECURITY_FAILURE);
			return null;
		}
	}

	protected String getChecksumInput(PatientLoginInfo info) {
		StringBuilder result = new StringBuilder();
		result.append(info.getTimestamp());
		result.append(info.getLastName());
		for(String facilityValue : info.getFacilityValues()) {
			result.append(facilityValue);
		}
		return result.toString();
	}

    protected URL createURL(String description, String url) {
    	try {
    		return new URL(url);
    	} catch (MalformedURLException e) {
    		log.error("Malformed url: " +  ". Provide the appropriate " +
    			description + " URL in the SM properties.");
    		//addErrorMessage("unexpected.error.occurred", null);
    		//FacesContext.getCurrentInstance().addMessage(null, new FacesMessage(FacesMessage.SEVERITY_ERROR, UNEXPECTED_ERROR, UNEXPECTED_ERROR));
    		return null;
    	}
    }

	protected String encryptIfRequired(String input) {
		MHVCipher cipher = (getSettings() != null) ? getSettings().getCipher()
			: null;
		return ((cipher != null))
			? cipher.encrypt(input) : input;
	}

	protected String createFailureForward(String message) {
		if (!StringUtils.isBlank(message)) {
			log.error(message);
    		//addUnexpectedError(UNEXPECTED_ERROR, null);
		}
		return null;
	}

	protected String describeUser(PatientLoginInfo info) {
		return "user '" + ((info != null) ? info.getUserId() : "<null>") + "'";
	}

	protected boolean authenticate(PatientLoginInfo patientLoginInfo) {
		if(patientLoginInfo == null) {
			throw new IllegalArgumentException("No patient login information available for user.");
		}
       	return getDelegate().authenticationPatient(patientLoginInfo);
    }

	protected SMAuthenticationDelegate getDelegate() {
		return delegate;
	}

	protected SecureMessagingSettings getSettings() {
		return settings;
	}

	protected String getSource() {
		return (getSettings() != null) ? getSettings().getMhvSource() : "";
	}

	protected String getRequestUrl() {
		return (getSettings() != null) ? getSettings().getMhvUrl() : "";
	}


	/**
	 * get PatientDTO for userprofileId
	 *
	 * @param userId
	 * @return
	 */
	private PatientDTO getPatientForUser(Long userprofileId) {
		PatientDTO patientDTO = null;

		try {

			patientDTO = this.userMgmtService.getPatient(userprofileId);

		} catch (Exception e) {
			log.error("Error in fetching patient data " + e);
			throw new MHVRuntimeException(e);
		}

		return patientDTO;
	}

	/**
	 * get PatientDTO for userprofileId
	 *
	 * @param userId
	 * @return
	 */
	private boolean isPatientAuthenticated(Long patientId) {
		Boolean ipaedPatient = false;

		try {
			ipaedPatient = this.userMgmtService.isPatientIPA(patientId);
		} catch (Exception e) {
			log.error("Error in fetching patient data " + e);
			throw new MHVRuntimeException(e);
		}

		return ipaedPatient;
	}


	private UserProfileDTO getCurrentUserProfile(String userName) {
		UserProfileDTO userProfileDto = null;
		try {

			userProfileDto = this.userMgmtService.getUserProfile(userName);

		} catch (Exception e) {
			log.error("Error in fetching user data " + e);
			throw new MHVRuntimeException(e);
		}

		return userProfileDto;
	}

}
