package gov.va.med.mhv.rxrefill.service.impl.util;

import java.sql.Timestamp;
import java.util.Date;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.rxrefill.data.model.Institution;
import gov.va.med.mhv.rxrefill.data.model.Request;
import gov.va.med.mhv.rxrefill.data.model.RequestAttempt;
import gov.va.med.mhv.rxrefill.data.repository.RequestRepository;
import gov.va.med.mhv.rxrefill.enums.RequestFunctionEnumeration;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;
import gov.va.med.mhv.rxrefill.util.RxMessagesUtil;

@Component
public class RxRequestStrategy {
	private static Logger log = LogManager.getLogger(RxRequestStrategy.class);
	public static final String requests_per_day_mpi_policy = "requests-per-day";
	public static final String request_time_window_mpi_policy = "request-time-window";
	private static int MS_IN_ONE_DAY = 24 * 60 * 60 * 1000;

	@Autowired
	private RxMessagesUtil rxMessagesUtil;

	@Autowired
	private RequestRepository requestRepository;

	public long getMaxTotalRxRequests() {
		return rxMessagesUtil.getRxrefillMaxConsecutiveRequests();

	}

	public long getMaxConsecutiveRxFailures() {
		return rxMessagesUtil.getRxrefillMaxConsecutiveFailures();
	}

	public static long getMaxRxProfileFailures() {
		// TODO - write implementation
		return 1;
	}

	private boolean isBelowRequestLimit(Long rxRequestSuccessCount, Long rxRequestFailedConsecutiveFailures) {
		log.debug("In side RxRequestStrategy - isBelowRequestLimit - getMaxTotalRxRequests::"+getMaxTotalRxRequests()+" -- rxRequestSuccessCount::"+rxRequestSuccessCount);
		log.debug("In side RxRequestStrategy - isBelowRequestLimit - getMaxConsecutiveRxFailures::"+getMaxConsecutiveRxFailures()+" -- rxRequestFailedConsecutiveFailures::"+rxRequestFailedConsecutiveFailures);

		return ((getMaxTotalRxRequests() <= 0)  || (rxRequestSuccessCount < getMaxTotalRxRequests()))
			&& ((getMaxConsecutiveRxFailures() <= 0) || (rxRequestFailedConsecutiveFailures < getMaxConsecutiveRxFailures()));
	}
	
	private Timestamp getFromTimestamp(Date now) {
		Date from = UtilityMethods.getDateWithoutTime(now);
		return UtilityMethods.dateToTimestamp(from);
	}
	
	private Timestamp getToTimestamp(Date now) {
		Date from = UtilityMethods.getDateWithoutTime(now);
		return UtilityMethods.dateToTimestamp(new Date(from.getTime() + MS_IN_ONE_DAY));
	}
	
	private Long[] getSuccessFailureCount(Long userProfileId, Long institutionId, int function) {
		
		Long[] successFailureCount = new Long[2];
		Date now = new Date();
		
		try {
			successFailureCount[0] = this.requestRepository.countSuccessRequests(userProfileId, function, institutionId, getFromTimestamp(now), getToTimestamp(now));
			successFailureCount[1] = this.requestRepository.countFailedRequests(userProfileId, function, institutionId, getFromTimestamp(now), getToTimestamp(now));
		} catch (Exception ex) {
			String message = String.format("Error getting request success / failure counts. User Profile Id: %d, Insitution ID: %d, Function: %d", userProfileId, institutionId, function);
			log.error(message);
			throw new MHVRuntimeException(message, ex);
		}
		
		return successFailureCount;
	}
	
	public boolean checkSendActiveUpdateRequest(Patient patient, Institution institution) {

		Long[] profileSuccessFailureCounts = getSuccessFailureCount(patient.getUserProfileId(), institution.getId(), RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue());
		Long[] activeSuccessFailureCounts = getSuccessFailureCount(patient.getUserProfileId(), institution.getId(), RequestFunctionEnumeration.GETPRESCRIPTIONSTATUS.getValue());
		
		// check request conditions
		boolean result = isBelowRequestLimit(profileSuccessFailureCounts[0], profileSuccessFailureCounts[1]) 
				&& isBelowRequestLimit(activeSuccessFailureCounts[0], activeSuccessFailureCounts[1]);

		log.debug("In side RxRequestStrategy - checkSendActiveUpdateRequest - result of RequestLimit::"+result);

		return result;
	}

	public boolean checkSendHistoricalUpdateRequest(Patient patient, Institution institution) {

		Long[] profileSuccessFailureCounts = getSuccessFailureCount(patient.getUserProfileId(), institution.getId(), RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue());

		// check request conditions
		boolean result = isBelowRequestLimit(profileSuccessFailureCounts[0], profileSuccessFailureCounts[1]);

		log.debug("In side RxRequestStrategy - checkSendHistoricalUpdateRequest - result after isBelowRequestLimit::"+result);

		return result;
	}


	public boolean checkSendStatusRequest(Patient patient, Institution institution)
	{
		verifyPatientAndInstitution(patient, institution);
		
		Long[] profileSuccessFailureCounts = getSuccessFailureCount(patient.getUserProfileId(), institution.getId(), RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue());
		Long[] activeSuccessFailureCounts = getSuccessFailureCount(patient.getUserProfileId(), institution.getId(), RequestFunctionEnumeration.GETPRESCRIPTIONSTATUS.getValue());
		
		// check request conditions
		boolean result = institution.getInstitutionId() != null && isBelowRequestLimit(profileSuccessFailureCounts[0], profileSuccessFailureCounts[1]) 
				&& isBelowRequestLimit(activeSuccessFailureCounts[0], activeSuccessFailureCounts[1]);

		// check if user had successful full extract or active extract
		result = isBelowRequestLimit(profileSuccessFailureCounts[0], profileSuccessFailureCounts[1]) 
				&& isBelowRequestLimit(activeSuccessFailureCounts[0], activeSuccessFailureCounts[1]);

		return result;
	}

	private void verifyPatientAndInstitution(Patient patient, Institution institution) {
		if ((patient == null) || (institution == null)) {
			throw new IllegalArgumentException("patient and institution must BOTH be not null! patient=" +
				patient + ", institution=" + institution);
		}
	}

	public boolean checkSendProfileRequest(Patient patient,	Institution institution) {
		verifyPatientAndInstitution(patient, institution);
		
		Long[] profileSuccessFailureCounts = getSuccessFailureCount(patient.getUserProfileId(), institution.getId(), RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue());
		
		Long countProfileAttemptFailtures = 0L;
		
		try {
			Date now = new Date();
			countProfileAttemptFailtures = this.requestRepository.countFailedRequestAttempts(patient.getUserProfileId(), 
					RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue(), institution.getId(), 
					getFromTimestamp(now), getToTimestamp(now));
		} catch( Exception ex) {
			String message = String.format("Error getting request attempt failure counts. User Profile Id: %d, Insitution ID: %d, Function: %d", 
					patient.getUserProfileId(), institution.getId(), RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue());
			log.error(message);
			throw new MHVRuntimeException(message, ex);
		}
		
		// check request conditions
		return isBelowRequestLimit(profileSuccessFailureCounts[0], profileSuccessFailureCounts[1]) && (countProfileAttemptFailtures < getMaxRxProfileFailures());
	}

	public boolean checkSendRefillRequest(Patient patient, Institution institution) {
		return true;
	}

	public Date checkRxExtractCutOffDate(Patient patient, Institution institution)
	{
		List<Request> extractRequests = null;
		Date latest = null;
		try {
			extractRequests = requestRepository.getLatestSuccessfulExtractRequests(patient.getUserProfile().getId(), institution.getId());
			if(extractRequests != null && extractRequests.size() > 0) {
				for (Request request : extractRequests) {
					for (RequestAttempt attempt : request.getRequestAttempts()) {
						
						if(latest == null || attempt.getCreatedDate().after(latest)) {
							latest = attempt.getCreatedDate();
						}
					}
				}
			}
		} catch(Exception e) {
			String message = String.format("Error getting last successful extract request for User Profile Id: %d, Institution Id: %d. ", 
					patient.getUserProfileId(), institution.getId());
			log.error(message , e);
			throw new MHVRuntimeException(message + e.getMessage(), e);
		}
		
		return latest;
	}
}
