package gov.va.med.mhv.rxrefill.service.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.common.data.model.Facility;
import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.common.data.model.PatientRegistryChange;
import gov.va.med.mhv.common.data.model.UserProfile;
import gov.va.med.mhv.rxrefill.converter.PharmacyPatientConverter;
import gov.va.med.mhv.rxrefill.converter.PrescriptionConverter;
import gov.va.med.mhv.rxrefill.converter.RxTrackingConverter;
import gov.va.med.mhv.rxrefill.data.model.Institution;
import gov.va.med.mhv.rxrefill.data.model.PharmPatientChange;
import gov.va.med.mhv.rxrefill.data.model.PharmacyPatient;
import gov.va.med.mhv.rxrefill.data.model.Prescription;
import gov.va.med.mhv.rxrefill.data.model.Request;
import gov.va.med.mhv.rxrefill.data.model.RxTracking;
import gov.va.med.mhv.rxrefill.data.repository.PatientRepository;
import gov.va.med.mhv.rxrefill.data.repository.PharmacyPatientRepository;
import gov.va.med.mhv.rxrefill.data.repository.PrescriptionRepository;
import gov.va.med.mhv.rxrefill.data.repository.RequestRepository;
import gov.va.med.mhv.rxrefill.data.repository.RxTrackingRepository;
import gov.va.med.mhv.rxrefill.data.sort.PrescriptionChainedComparator;
import gov.va.med.mhv.rxrefill.data.sort.PrescriptionFillDateComparator;
import gov.va.med.mhv.rxrefill.data.sort.PrescriptionNumberComparator;
import gov.va.med.mhv.rxrefill.dto.PrescriptionDTO;
import gov.va.med.mhv.rxrefill.dto.RxTrackingDTO;
import gov.va.med.mhv.rxrefill.enums.MhvStatusEnumeration;
import gov.va.med.mhv.rxrefill.enums.PrescriptionStatusEnumeration;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;
import gov.va.med.mhv.rxrefill.service.PatientChangeProcessor;
import gov.va.med.mhv.rxrefill.service.PrescriptionService;
import gov.va.med.mhv.rxrefill.service.impl.util.PatientChanges;
import gov.va.med.mhv.rxrefill.service.impl.util.RxRequestDispatcher;
import gov.va.med.mhv.rxrefill.util.PrescriptionUtils;
import gov.va.med.mhv.usermgmt.common.enums.ActivityActionTypeEnumeration;
import gov.va.med.mhv.usermgmt.common.enums.ActivityTypeEnumeration;
import gov.va.med.mhv.usermgmt.service.AccountActivityCreatorService;
import gov.va.med.mhv.usermgmt.util.activity.ActivityHelper;


@Component
public class PrescriptionServiceImpl implements PrescriptionService {

	private static Logger log = LogManager.getLogger(PrescriptionServiceImpl.class);

	private static final String UNKNOWN = "<?>";

	//prescription mhv statuses
	// pending mhv statuses
	private static final int[] MHV_STATUS_ENUM_NAMES_PENDING = {
		MhvStatusEnumeration.PENDING.getValue(),
		MhvStatusEnumeration.RESULTSBACKORNOTPICKEDUP.getValue()
	};

	// invalid mhv statuses
	private static final int[] MHV_STATUS_ENUM_NAMES_INVALID = {
		MhvStatusEnumeration.UNABLETORESOLVEPATIENT.getValue(),
		MhvStatusEnumeration.RESOLVEPRESCRIPTION.getValue(),
		MhvStatusEnumeration.DFNNOTMATCHRX.getValue()
	};

	@Autowired
	private PharmacyPatientRepository pharmacyPatientRepository;

	@Autowired
	private PrescriptionRepository prescriptionRepository;

	@Autowired
	private PharmacyPatientConverter pharmacyPatientConverter;
	
	@Autowired
	private PatientRepository patientRepository;

	@Autowired
	private PrescriptionConverter prescriptionConverter;

	@Autowired
	private RxTrackingConverter rxTrackingConverter;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private RxTrackingRepository rxTrackingRepository;

	@Autowired
	private PrescriptionUtils prescriptionUtils;

	@Autowired
	private RxRequestDispatcher dispatcher;

	@Resource(name = "activityProxy")
	private AccountActivityCreatorService accountActivityCreatorService;
	
	@Autowired
	private PatientChangeProcessor patientChangeProcessor;
	
	@Override
	@Transactional
	public PharmacyPatient getPharmacyPatient(Long userProfileId) {
		
		PharmacyPatient pharmacyPatient = null;
		
		if (userProfileId == null) {
			log.error("UserProfile is null");
			throw new MHVRuntimeException("UserProfile is null");
		}

		try {
			pharmacyPatient = pharmacyPatientRepository.findOne(userProfileId);
			
			if (pharmacyPatient == null) {
				
				pharmacyPatient = createPharmacyPatient(userProfileId);
				
			}
			
		} catch (Exception e) {
			log.error("Error in fetching pharmacyPatient ", e);
			throw new MHVRuntimeException("Error in fetching pharmacyPatient " + e.getMessage(), e);
		}

		return pharmacyPatient;
		
	}
	
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public PharmacyPatient createPharmacyPatient(Long userProfileId) {
		PharmacyPatient pharmacyPatient = new PharmacyPatient();
		pharmacyPatient.setId(userProfileId);
		
		Patient patient = this.patientRepository.getPatientByUserProfileId(userProfileId);
		
		if (null != patient) {
			pharmacyPatient.setPatientId(patient.getId());
			pharmacyPatient.setPatient(patient);
		} else {
			throw new IllegalArgumentException(String.format("User with user profile ID: %s is not a patient.", userProfileId));
		}
		
		return pharmacyPatientRepository.save(pharmacyPatient);
	}
	
	@Override
	public PharmacyPatient getPharmacyPatient(PharmacyPatient pharmPatient) {
		PharmacyPatient pharmacyPatient = null;
		Long userProfileId = null;

		if (null != pharmPatient && null != pharmPatient.getPatient() && null != pharmPatient.getPatient().getUserProfileId()) {
			userProfileId = pharmPatient.getPatient().getUserProfileId();
		} else {
			log.error("pharmPatient is null");
			throw new MHVRuntimeException("pharmPatient is null");
		}

		try {
			pharmacyPatient = pharmacyPatientRepository.findOne(userProfileId);
		} catch (Exception e) {
			log.error("Error in fetching pharmacyPatient. userProfileid:: <<<" + userProfileId + ">>> ", e);
			throw new MHVRuntimeException("Error in fetching pharmacyPatient. userProfileid:: <<<" + userProfileId + ">>> "  + e.getMessage(), e);
		}

		return pharmacyPatient;
	}


	@Override
	public ResponseUtil refillPrescriptions(
		PharmacyPatient pharmacyPatient, List<PrescriptionDTO> prescriptions)
	{

		if (log.isDebugEnabled()) {
			log.debug("007 >>>>>>>>>>> inside PrescriptionServiceImpl:refillPrescriptions. - pharmacyPatient::"+pharmacyPatient);
		}

		final ResponseUtil response = new ResponseUtil();
		final List<Prescription> refills = new ArrayList<Prescription>();
		for (PrescriptionDTO r: prescriptions) {
			refills.add((Prescription) prescriptionConverter.convertDTOPrescription(r));
		}
		if (log.isDebugEnabled()) {
			log.debug("008 >>>>>>>>>>> inside PrescriptionServiceImpl:refillPrescriptions. - refills::"+refills.size());
		}

		new MonitoredRequest("Refill Prescriptions", pharmacyPatient) {
			@Override
			protected void doRun() {
				int size = refills.size();
				if (size == 0) {
					return;
				}
				Patient patient = getPharmacyPatient().getPatient();
				if (patient == null) {
					return;
				}

				if (log.isDebugEnabled()) {
					log.debug("009 >>>>>>>>>>> inside PrescriptionServiceImpl:refillPrescriptions. - patient::"+patient);
				}

				if (log.isDebugEnabled()) {
					log.debug("Received request to refill " + size);
				}
				Prescription[] refillArray = toArray(refills);
				assert refillArray.length > 0;
				////RxRequestDispatcher dispatcher = new RxRequestDispatcher();
				if (log.isDebugEnabled()) {
					log.debug("009 >>>>>>>>>>> inside PrescriptionServiceImpl:refillPrescriptions. - Calling dispatchRefillRequests");
				}
				dispatcher.dispatchRefillRequests(patient, refills,response);
				//public int dispatchRefillRequests(Patient patient, List<Prescription> prescriptions, ResponseUtil response) {
				//auditRefillrequest(patient, refillArray,	!response.getMessages().hasErrorMessages());

				// Create AAL entry here..

				try {
					StringBuffer prescriptionNames = new StringBuffer(100);
					for (int i = 0; i < refillArray.length; i++) {
						Prescription p = refillArray[i];
						prescriptionNames.append(p.getDrugName());
						if (i + 1 < refillArray.length) {
							prescriptionNames.append(",");
						}
					}
					accountActivityCreatorService.createAccountActivityLog(
												ActivityHelper.createActivityDTOForSelf(
												patient.getUserProfile().getId(),
												response.getFailureMessage()!= null ? false:true,
												ActivityTypeEnumeration.RXREFILL,
												ActivityActionTypeEnumeration.REQUEST_REFILL,
												prescriptionNames.toString())
												);
				} catch (Exception e) {
					if (patient!=null) {
						log.error("Exception occured in Activity Track. userProfileid:: <<<" + patient.getUserProfile().getId() + ">>> ", e);
					}
					throw new MHVRuntimeException("Exception occured in Activity Track " + e.getMessage(), e);
				}
			}
			@Override
			protected String describeOnError() {
				//return describePrescriptions(prescriptions) + " " + super.describeOnError();
				return null;
			}

			private Prescription[] toArray(List<Prescription> prescriptions) {
				return refills.toArray(new Prescription[0]);
			}
		}.run();

		return response;
	}

	private static String describePrescriptions(Collection prescriptions) {
		StringBuilder builder = new StringBuilder("prescriptions: {");
		if (prescriptions != null) {
			for (Iterator i = prescriptions.iterator(); i.hasNext(); ) {
				builder.append("\n\tPrescription(");
				Prescription prescription = (Prescription) i.next();
				if (prescription != null) {
					builder.append("ID=");
					////builder.append(prescription.getPrescriptionId());
					builder.append(";Number=");
					builder.append(prescription.getPrescriptionNumber());
				} else {
					builder.append("NULL");
				}
				builder.append(")");
			}
		}
		builder.append("}");
		return builder.toString();
	}

	@Override
	@Transactional
	public ResponseUtil<List<PrescriptionDTO>> getActivePrescriptions(PharmacyPatient pharmacyPatient) {
		final ResponseUtil<List<PrescriptionDTO>> response = new ResponseUtil();

		new MonitoredRequest("Retrieve Active Prescriptions", pharmacyPatient) {
			@Override
			protected void doRun() {
				PatientChanges patientChanges = patientChangeProcessor.processPatientChanges(getPharmacyPatient());
				refreshActivePrescriptions(getPharmacyPatient(), response);
				
				if (log.isDebugEnabled())
					log.debug("PrescriptionServiceImpl - response.getFailureMessage()" + response.getFailureMessage());

				List<Prescription> successfulActivePrescriptions = null;
				List<Prescription> unsuccessfulActivePrescriptions = null;
				List<Prescription> allPrescriptions = null;
				
				List<PrescriptionDTO> activePrescriptions = null;
				

				try {
					if (log.isDebugEnabled()) {
						log.debug("calling getActivePrescriptionsWithLastRequestSuccessful query. - getUserId:: " + getUserId());
					}
					successfulActivePrescriptions = prescriptionRepository.getActivePrescriptionsWithLastRequestSuccessful(getUserId());
					if (log.isDebugEnabled()) {
						log.debug("called getActivePrescriptionsWithLastRequestSuccessful - Count of successfulActivePrescriptions:: " + successfulActivePrescriptions.size());
					}
					
					if (null != successfulActivePrescriptions && successfulActivePrescriptions.size() > 0) {
						setRxStatuses(successfulActivePrescriptions, true, true);
					}

					unsuccessfulActivePrescriptions = prescriptionRepository.getActivePrescriptionsWithLastRequestUnsuccessful(getUserId());
					if (log.isDebugEnabled()) {
						log.debug("called getActivePrescriptionsWithLastRequestUnsuccessful - Count of unsuccessfulActivePrescriptions:: " + unsuccessfulActivePrescriptions.size());
					}

					if (null != unsuccessfulActivePrescriptions && unsuccessfulActivePrescriptions.size() > 0) {
						setRxStatuses(unsuccessfulActivePrescriptions, true, false);
					}

					allPrescriptions = joinListsWithLatestPrescriptions(successfulActivePrescriptions, unsuccessfulActivePrescriptions);
					patientChangeProcessor.postProcessPatientChanges(getUserId(), patientChanges, allPrescriptions);
					
					activePrescriptions = convertToDtos(allPrescriptions);
					setRxTrackingNumbers(activePrescriptions);
					
				} catch(Exception e) {
					response.setFailure(true);
					log.error("Error in fetching active prescriptions. userProfileid:: <<<" + getUserId() + ">>> ", e);
					throw new MHVRuntimeException("Error in fetching active prescriptions " + e.getMessage(), e);
				}
				
				response.setFailure(false);
				response.setSuccess(true);
				response.setPojoObject(activePrescriptions);
				
			}
		}.run();

		if (log.isDebugEnabled())
			log.debug("PrescriptionServiceImpl - response.getFailureMessage()" + response.getFailureMessage());

		Map<String, String> infoMessages = response.getInfoMessages();
		if (null != infoMessages && infoMessages.size() > 0) {
			if (log.isDebugEnabled())
				log.debug("infoSize : " + infoMessages.size());
			Set set = infoMessages.entrySet();
			Iterator iterator = set.iterator();
	        while(iterator.hasNext()) {
	             Map.Entry me = (Map.Entry)iterator.next();
	             log.debug("Key is: "+ me.getKey() + "& Value is: "+me.getValue());
	        }
			/*
			for (String key: infoMessages.keySet()) {
				log.debug("key : " + key);
				if (RxRequestMessages.isMessagingError(key)) {
					log.info("inside for : FAILED");
				} else if (RxRequestMessages.isStatusError(key)) {
					log.info("inside for : REJECTED");
					return response;
				}
			}
			*/
		}

		return response;
	}


	@Override
	public ResponseUtil getHistoricalPrescriptionsWithRequestStatuses(PharmacyPatient pharmacyPatient) {
		final ResponseUtil response = new ResponseUtil();

		log.info("Retrieve Historical Prescriptions");

		new MonitoredRequest("Retrieve Historical Prescriptions", pharmacyPatient) {
			@Override
			protected void doRun() {
				PatientChanges patientChanges = new PatientChanges();

				if(getPharmacyPatient()!=null) {
					patientChanges = patientChangeProcessor.processPatientChanges(getPharmacyPatient());
					////if(patientChanges!=null) {
					refreshHistoricalPrescriptions(getPharmacyPatient(), response);
					////}
				}

				List<Prescription> successfulHistoricalPrescriptions = null;
				List<Prescription> unsuccessfulHistoricalPrescriptions = null;
				List<Prescription> historicalPrescriptions = null;
				
				List<PrescriptionDTO> historicalPrescriptionDtos = null;
				
				
				try {
					successfulHistoricalPrescriptions = prescriptionRepository.getHistoricalPrescriptionsWithLastRequestSuccessful(getUserId());
					if (successfulHistoricalPrescriptions==null) {
						successfulHistoricalPrescriptions = new ArrayList<Prescription>();
					}
					

					if (null != successfulHistoricalPrescriptions && successfulHistoricalPrescriptions.size() > 0) {

						if (log.isDebugEnabled()) {
							log.debug("  1000 >>>>>>>>>>>> in side getHistoricalPrescriptionsWithRequestStatuses - calling setRxStatuses successfulHistoricalPrescriptionDtos.size() : " + successfulHistoricalPrescriptions.size());
						}

						setRxStatuses(successfulHistoricalPrescriptions, true, true);
					}

					unsuccessfulHistoricalPrescriptions = prescriptionRepository.getHistoricalPrescriptionsWithLastRequestUnsuccessful(getUserId());
					if (unsuccessfulHistoricalPrescriptions==null) {
						unsuccessfulHistoricalPrescriptions = new ArrayList<Prescription>();
					}

					
					if (null != unsuccessfulHistoricalPrescriptions && unsuccessfulHistoricalPrescriptions.size() > 0) {

						if (log.isDebugEnabled()) {
							log.debug("  1001 >>>>>>>>>>>> in side getHistoricalPrescriptionsWithRequestStatuses - calling setRxStatuses unsuccessfulHistoricalPrescriptionDtos.size() : " + unsuccessfulHistoricalPrescriptions.size());
						}

						setRxStatuses(unsuccessfulHistoricalPrescriptions, true, false);
					}

					historicalPrescriptions = joinListsWithLatestPrescriptions(successfulHistoricalPrescriptions, unsuccessfulHistoricalPrescriptions);
					patientChangeProcessor.postProcessPatientChanges(getUserId(),patientChanges, historicalPrescriptions);
					
					historicalPrescriptionDtos = convertToDtos(historicalPrescriptions);
					setRxTrackingNumbers(historicalPrescriptionDtos);

				} catch(Exception e) {
					response.setFailure(true);
					log.error("Error in fetching historical prescriptions. userProfileid::  <<<" + getUserId() + ">>> ", e);
					throw new MHVRuntimeException("Error in fetching historical prescriptions " + e.getMessage(), e);
				}
				
				response.setFailure(false);
				response.setSuccess(true);
				response.setPojoObject(historicalPrescriptionDtos);

				if (log.isDebugEnabled()) {
					log.debug("=========inside getHistoricalPrescriptionsWithRequestStatuses - before calling postProcessPatientChanges ======");
				}
				
			}
		}.run();

		return response;
	}


	@Override
	public List<Request> getLastSuccessfulExtractRequestForUser(PharmacyPatient pharmacyPatient) {
		//ResponseUtil response = new ResponseUtil();
		Long userId = null;
		List<Request> requestList = null;

		if (null != pharmacyPatient && null != pharmacyPatient.getPatient()
				&& null != pharmacyPatient.getPatient().getUserProfile()
				&& null != pharmacyPatient.getPatient().getUserProfile().getId()) {
			
			userId = pharmacyPatient.getPatient().getUserProfile().getId();
		
		} else {
			log.error("pharmacyPatient is having null values");
			throw new MHVRuntimeException("pharmacyPatient is having null values");
		}

		try {
		
			requestList = requestRepository.getLatestSuccessfulExtractRequests(userId);
		
		} catch(Exception e) {
			log.error("Error in getLatestSuccessfulExtractRequests. userProfileid::  <<<" + userId + ">>> ", e);
			throw new MHVRuntimeException("Error in getLatestSuccessfulExtractRequests " + e.getMessage(), e);
		}
		
		return requestList;
	}
	
	
	@Override
	public List<Request> getRequestByFunctionAndInstitution(Long userId, Long institutionId) {
		//ResponseUtil response = new ResponseUtil();
		List<Request> requestList = null;

		try {
			requestList = requestRepository.getRequestsForPatientAndInstitution(userId, institutionId);
		} catch(Exception e) {
			log.error("Error in getRequestByFunctionAndInstitution. userProfileid::  <<<" + userId + ">>> ", e);
			throw new MHVRuntimeException("Error in getRequestByFunctionAndInstitution " + e.getMessage(), e);
		}
		
		return requestList;
	}


	/**
	 * Handles timing the doRun() execution and captures runtime exception
	 * in log.
	 * Additionally the pharmacy Patient is updated to the latests values
	 * from the business object.
	 */
	private abstract class MonitoredRequest {

		private final String name;
		private final PharmacyPatient pharmacyPatient;
		protected MonitoredRequest(String name, PharmacyPatient pharmacyPatient)
		{
			this.name = name;
			this.pharmacyPatient = getLatest(pharmacyPatient);
		}

		protected String getName() {
			return name;
		}

		protected PharmacyPatient getPharmacyPatient() {
			return pharmacyPatient;
		}

		protected long getUserId() {
			return toUserId(pharmacyPatient);
		}

		public void run() {
			StopWatch stopWatch = null;
			if (log.isDebugEnabled()) {
				stopWatch = new StopWatch();
				stopWatch.start();
			}
			try {
				doRun();
			} catch (RuntimeException e) {
				onRuntimeException(e);
				throw e;
			}
			if (log.isDebugEnabled() && (stopWatch != null)) {
				stopWatch.stop();
				log.debug("PERFORMANCE - " + getName() + " in " + stopWatch.getTime() + "ms");
			}
		}

		protected abstract void doRun();

		protected void onRuntimeException(RuntimeException e) {
			// Exception (stacktrace) will be logged in the ExceptionHandler
			// Do not log it here
			log.error("Unexpected exception occurred while " + getName() + " " + describeOnError());
		}

		protected String describeOnError() {
			return "for " + describe(getPharmacyPatient());
		}
	}

	
	@Override
	public void deletePrescriptionsForUserInstitutions(Long userId, Long institutionId) {
		List<Request> requestsToDelete = null; //new ArrayList(RequestBO.getRequestsForPatientAndInstitution(userId, institutionId));

		try {
			requestsToDelete = requestRepository.getRequestsForPatientAndInstitution(userId, institutionId);
		} catch(Exception e) {
			log.error("Error in fetching requestsToDelete. userProfileid::  <<<" + userId + ">>> ", e);
			throw new MHVRuntimeException("Error in fetching requestsToDelete " + e.getMessage(), e);
		}

		//TODO: is updated request stored in DB? - Prasad
		for (Request request : requestsToDelete) {
			request.setIsInactive(Boolean.TRUE);
		}

		List<Prescription> prescriptionsToInactivate = null; //PrescriptionBO.getPrescriptionsForInactivation(userId,institutionId);
		try {
			prescriptionsToInactivate = prescriptionRepository.getPrescriptionsForInactivation(userId, institutionId);
		} catch(Exception e) {
			log.error("Error in fetching prescriptionsToInactivate. userProfileid::  <<<" + userId + ">>> ", e);
			throw new MHVRuntimeException("Error in fetching prescriptionsToInactivate " + e.getMessage(), e);
		}

		//TODO: is updated prescription stored in DB? - Prasad
		for (Prescription prescription: prescriptionsToInactivate) {
			prescription.setIsInactive(Boolean.TRUE);
		}
	}

	/**
	 * get latest PharmacyPatient for userProfileId
	 *
	 * @param pharmacyPatient
	 * @return
	 */
	@Transactional
	private PharmacyPatient getLatest(PharmacyPatient pharmacyPatient) {
		PharmacyPatient latestPharmacyPatient = null;

		if (null == pharmacyPatient) {
			log.error("A pharamcyPatient is required");
			throw new MHVRuntimeException("A pharamcyPatient is required");
		}

		Patient patient = pharmacyPatient.getPatient();
		if (null == patient) {
			log.error("patient is required");
			throw new MHVRuntimeException("patient is required");
		}

		UserProfile userProfile = patient.getUserProfile();
		if (null == userProfile) {
			log.error("userProfile is required");
			throw new MHVRuntimeException("userProfile is required");
		}

		Long userProfileId = userProfile.getId();
		if (null == userProfileId) {
			log.error("userProfileId is null");
			throw new MHVRuntimeException("userProfileId is null");
		}

		try {
			latestPharmacyPatient = pharmacyPatientRepository.findOne(userProfileId);
			latestPharmacyPatient.setPatient(patient);
		} catch (Exception e) {
			log.error("Error in fetching pharmacyPatient ", e);
			throw new MHVRuntimeException("Error in fetching pharmacyPatient " + e.getMessage(), e);
		}

		log.debug("*****************pharmacyPatient.getPatient().getUserProfile().getId() : getLatest : " + latestPharmacyPatient.getPatient().getUserProfile().getId());
		if (null != latestPharmacyPatient.getPatient()) {
			log.debug("latestPharmacyPatient.getPatient() is not null");
			if (null != latestPharmacyPatient.getPatient().getUserProfile()) {
				log.debug("latestPharmacyPatient.getPatient().getUserProfile() is not null");
				if (null != latestPharmacyPatient.getPatient().getUserProfile().getAccessRoles()) {
					log.debug("latestPharmacyPatient.getPatient().getUserProfile().getAccessRoles() is not null");

					log.debug("*****************latestPharmacyPatient.getPatient().getUserProfile().getAccessRoles().size() " + latestPharmacyPatient.getPatient().getUserProfile().getAccessRoles().size());
				} else {
					log.debug("latestPharmacyPatient.getPatient().getUserProfile().getAccessRoles() is null");
				}
			} else {
				log.debug("latestPharmacyPatient.getPatient().getUserProfile() is null");
			}
		} else {
			log.debug("latestPharmacyPatient.getPatient() is null");
		}

		return latestPharmacyPatient;
	}

	private static long toUserId(PharmacyPatient pharmacyPatient) {
		return pharmacyPatient.getPatient().getUserProfile().getId().longValue();
	}

	public static String describe(PharmacyPatient pharmacyPatient) {
		Patient patient = null;
		if (pharmacyPatient != null) {
			patient = pharmacyPatient.getPatient();
		}
		return describe(patient);
	}

	public static String describe(Patient patient) {
		StringBuilder builder = new StringBuilder("Patient[");
		if (patient != null) {
			UserProfile userProfile = patient.getUserProfile();
			if (userProfile != null) {
				builder.append("UserName=").append(userProfile.getUserName());
			} else {
				builder.append("PatientID=").append(patient.getId());
			}
			builder.append(";ICN=").append(patient.getIcn());
		} else {
			builder.append(UNKNOWN);
		}
		builder.append("]");
		return builder.toString();
	}

	public static String describe(PharmPatientChange pharmacyChange) {
		StringBuilder builder = new StringBuilder("PharmacyPatientChange[");
		if (pharmacyChange != null) {
			PatientRegistryChange change = pharmacyChange.getPatientRegistryChange();
			if (change != null) {
				builder.append("RecordedAt=");
				builder.append(change.getRecordedOnDate());
				builder.append(";OldIcn=");
				builder.append(change.getOldIcn());
				builder.append(";OldFacilityCount=");
				builder.append(change.getOldFacilityCount());
			} else {
				builder.append("RegistrationChange=").append(UNKNOWN);
			}
		}
		builder.append("]");
		return builder.toString();
	}

	private List<PrescriptionDTO> convertToDtos(List<Prescription> prescriptions) {
		List<PrescriptionDTO> prescriptionDtos = null;

		if (null != prescriptions && prescriptions.size() > 0) {
			prescriptionDtos = new ArrayList<PrescriptionDTO>();
			for (Prescription activePrescription : prescriptions) {
				/*if (log.isDebugEnabled()) {
					log.debug("activePrescription.getUserProfile().getAccessRoles().size() : " + activePrescription.getUserProfile().getAccessRoles().size());
				}*/

				prescriptionDtos.add(prescriptionConverter.convertDomainPrescription(activePrescription));
			}

			if (log.isDebugEnabled()) {
				log.debug("activePrescriptionDtos count : " + prescriptionDtos.size());
			}
		}

		return prescriptionDtos;
	}

	/* TODO: move to presentation layer?
	 * isLastRequestSuccessful, isInvalidMhvStatus, isRefillable
	 * & status (submitted) are for UI and not to be persisted.
	 */
	private void setRxStatuses(List<Prescription> rxList, boolean activeRx, boolean lastRequestSuccessful) {


		if (log.isDebugEnabled()) {
			log.debug("  1002 >>>>>>>>>>>> in side setRxStatuses - rxList.size() : " + rxList.size());
		}

		for (Prescription rx : rxList) {

			if (log.isDebugEnabled()) {
				log.debug("  1003 >>>>>>>>>>>> in side setRxStatuses - rx record : " + rx);
			}

            // set default values
			rx.setIsInvalidMhvStatus(new Boolean(false));
            rx.setIsRefillable(new Boolean(false));
			rx.setIsLastRequestSuccessful(new Boolean(lastRequestSuccessful));

			if (log.isDebugEnabled()) {
				log.debug("  1003a >>>>>>>>>>>> in side setRxStatuses - rx record status: " + rx.getStatus()+"  -- getMhvStatus: " + rx.getMhvStatus()+"  -- getNumberOfRefills: " + rx.getNumberOfRefills());
			}

			// rename statuses
            if (PrescriptionStatusEnumeration.SUSPENDED.getValue().equals(rx.getStatus()))
                rx.setStatus(PrescriptionStatusEnumeration.REFILLINPROCESS.getValue());

            if (PrescriptionStatusEnumeration.DISCONTINUEDEDIT.getValue().equals(rx.getStatus()))
                rx.setStatus(PrescriptionStatusEnumeration.DISCONTINUED.getValue());

			if (log.isDebugEnabled()) {
				log.debug("  1003a >>>>>>>>>>>> in side setRxStatuses - rx record getMhvStatus : " + rx.getMhvStatus());
			}

            if (checkMhvStatusesPending(rx.getMhvStatus())) {
				/* pending mhv status: checking submitted status
				 * when mhvStatus is 'pending' or 'processed',
				 * rx status is 'active', and last request is successful
				 * change rx status to 'submitted'
				 */
				if (PrescriptionStatusEnumeration.ACTIVE.getValue().equals(rx.getStatus())
						&& rx.getIsLastRequestSuccessful().booleanValue()) {
					rx.setStatus(PrescriptionStatusEnumeration.SUBMITTED.getValue());
				}
			}

			if (activeRx) {
                //checking MHV status
				if (checkMhvStatusesInvalid(rx.getMhvStatus())) {
					//invalid mhv status: setting isInvalidMhvStatus
					rx.setIsInvalidMhvStatus(new Boolean(true));
				} else if (PrescriptionStatusEnumeration.ACTIVE.getValue().equals(rx.getStatus())
                        && rx.getNumberOfRefills().intValue() > 0
                        && !checkMhvStatusesPending(rx.getMhvStatus())) {

                    rx.setIsRefillable(new Boolean(true));
				}
			}
		}

		return;
	}

	private boolean checkMhvStatusesPending(int status) {
		return checkMhvStatuses(MHV_STATUS_ENUM_NAMES_PENDING, status);
	}

	private boolean checkMhvStatusesInvalid(int status) {
		return checkMhvStatuses(MHV_STATUS_ENUM_NAMES_INVALID, status);
	}

	private boolean checkMhvStatuses(int[] mhvStatuses, int status) {
		for (int mhvStatus : mhvStatuses) {
			if (mhvStatus == status)
				return true;
		}
		return false;
	}


	private List<Prescription> joinListsWithLatestPrescriptions(List<Prescription> rxList1, List<Prescription> rxList2) {
		if (log.isDebugEnabled()) {
			log.debug("========inside joinListsWithLatestPrescriptions===== ");
		}
		List<Prescription> mergedRxList = new ArrayList<Prescription>();

        if (rxList1 == null || rxList1.isEmpty()) return (rxList2 == null || rxList2.isEmpty()) ? null : sortPrescriptions(rxList2);
        if (rxList2 == null || rxList2.isEmpty()) return (rxList1 == null || rxList1.isEmpty()) ? null : sortPrescriptions(rxList1);

        if (log.isDebugEnabled()) {
			log.debug("mergedRxList.size() " + mergedRxList.size());
		}
        // add list1 items first
        for (Prescription rx1 : rxList1) {

            // loop through list2 to find the same id
            for (Prescription rx2 : rxList2) {

                if (rx1.getId().equals(rx2.getId())) {

                    // check which modified date is later
                    if (rx1.getModifiedDate().compareTo(rx2.getModifiedDate()) < 0)
                        rx1 = rx2; //rx2 replaces rx1

                    rxList2.remove(rx2); // remove the dup record
                    break; // done with this list1 item
                }
            }
            mergedRxList.add(rx1);
        }

        // add all unmatched list2 items
        mergedRxList.addAll(rxList2);

        if (log.isDebugEnabled()) {
			log.debug("mergedRxList.size() " + mergedRxList.size());
		}

        return sortPrescriptions(mergedRxList);
    }


	/**
	 * sort prescriptions by lastFilldate descending and rx# ascending
	 *
	 * @param rxList
	 * @return
	 */
	private List<Prescription> sortPrescriptions(List<Prescription> rxList) {
		if (log.isDebugEnabled()) {
			log.debug("inside sortPrescriptions rxList.size() " + ((null != rxList) ? rxList.size() : 0));
		}

		// sort prescriptions by lastFilldate descending and rx# ascending
		Collections.sort(rxList, new PrescriptionChainedComparator(
						new PrescriptionFillDateComparator(),
						new PrescriptionNumberComparator()));

		return rxList;
	}

	/**
	 * Execute the RefreshActivePrescriptions service
	 */
	public void refreshActivePrescriptions(PharmacyPatient pharmacyPatient, ResponseUtil response) {
		Patient patient = pharmacyPatient.getPatient();
		if (null != patient) {
			dispatcher.dipatchStatusRequests(patient, response);
		}
	}

	/**
	 * Execute the refreshHistoricalPrescriptions service
	 */
	public void refreshHistoricalPrescriptions(PharmacyPatient pharmacyPatient, ResponseUtil response) {
		Patient patient = pharmacyPatient.getPatient();
		if (null != patient) {
			dispatcher.dipatchExtractRequests(patient, response);
		}
	}

	private void setRxTrackingNumbers(List<PrescriptionDTO> rxList) {
		HashMap <String, Integer> trackCount = new HashMap<String, Integer>();
		if(rxList!=null && rxList.size() > 0) {
			for (PrescriptionDTO rx : rxList) {
				List<RxTracking> trackingDetailsList = new ArrayList<RxTracking>();
				if(rx.getInstitution()!=null && rx.getInstitution().getStationNumber()!=null && rx.getPrescriptionNumber()!=null) {
					trackingDetailsList = rxTrackingRepository.getPrescriptionTrackingBySiteAndRxNumber(rx.getInstitution().getStationNumber().trim(), rx.getPrescriptionNumber().trim());
				}
				
				List<RxTrackingDTO> trackingList = new ArrayList<RxTrackingDTO>();
				boolean isPrescriptionCriteriaMet = trackingDetailsList != null && trackingDetailsList.size() > 0 && rx.getReleaseDateTime()!=null;
				boolean tracking=false;
				if (isPrescriptionCriteriaMet) {
					long diffDays = calculateDayDifference(rx);

					if (diffDays <= 30) {
						tracking=true;
					}

					for (RxTracking rxTracking : trackingDetailsList) {
						RxTrackingDTO newRxDto = rxTrackingConverter.convertDomainRxTracking(rxTracking);

						boolean isViewImageDisplayed=false;
						if(rxTracking.getNdc()!=null) {
							String ndc=rxTracking.getNdc().trim();

							if(ndc!=null) {
								int ndcLength=ndc.length();
								if(ndcLength > 10) {
									isViewImageDisplayed=true;
								}
							}
						}
						newRxDto.setViewImageDisplayed(isViewImageDisplayed);
						trackingList.add(newRxDto);

					}
				}

				rx.setTracking(tracking);
				rx.setRxTrackingList(trackingList);
				buildTrackingCountMap(trackCount,trackingList);

			}
		}

		if(rxList!=null && rxList.size() > 0) {
			for (PrescriptionDTO rx : rxList) {

				for (RxTrackingDTO rxTracking : rx.getRxTrackingList()) {
					String trackingNumber=rxTracking.getTrackingNumber();
					Integer countForTracking = trackCount.get(trackingNumber);
					if (countForTracking > 1) {
						rxTracking.setOthersInSamePackage(true);
					}
					else {
						rxTracking.setOthersInSamePackage(false);
					}

				}
			}
		}


	}


	private void buildTrackingCountMap(HashMap<String, Integer> trackCount, List<RxTrackingDTO> rxList) {
		for (RxTrackingDTO rxTracking : rxList) {
			String trackingNumber = rxTracking.getTrackingNumber().trim();
			Integer trackingCount = trackCount.get(trackingNumber);

			if (trackingCount != null) {
				trackingCount++;
			} else {
				trackingCount = 1;
			}
			trackCount.put(trackingNumber, trackingCount);
		}
	}


	private long calculateDayDifference(PrescriptionDTO rx) {
		Date releaseDate = rx.getReleaseDateTime();
		Date today = new Date();
		long diff = today.getTime() - releaseDate.getTime();
		long diffDays = diff / (24 * 60 * 60 * 1000);
		return diffDays;
	}


	@Override
	@Transactional
	public ResponseUtil getAllPrescriptions(PharmacyPatient pharmacyPatient) {
		final ResponseUtil response = new ResponseUtil();

		new MonitoredRequest("Retrieve Active Prescriptions", pharmacyPatient) {
			@Override
			protected void doRun() {
				PatientChanges patientChanges = patientChangeProcessor.processPatientChanges(getPharmacyPatient());
				List<Prescription> allPrescriptions = null;
				List<PrescriptionDTO> allPrescriptionDtos = null;
				
				try {

					allPrescriptions = prescriptionRepository.getAllPrescriptionsForPatient(getUserId());
					if (log.isDebugEnabled()) {
						log.debug("called getAllPrescriptionsForPatient - Count of getAllPrescriptionsForPatient:: " + allPrescriptions.size());
					}

					if (null != allPrescriptions && allPrescriptions.size() > 0) {
						setRxStatuses(allPrescriptions, true, false);
					}
					
					patientChangeProcessor.postProcessPatientChanges(getUserId(),patientChanges, allPrescriptions);

					allPrescriptionDtos = convertToDtos(allPrescriptions);
					setRxTrackingNumbers(allPrescriptionDtos);
				} catch(Exception e) {
					response.setFailure(true);
					log.error("Error in fetching active prescriptions. userProfileid::  <<<" + getUserId() + ">>> ", e);
					throw new MHVRuntimeException("Error in fetching active prescriptions " + e.getMessage(), e);
				}
				
				response.setFailure(false);
				response.setSuccess(true);
				response.setPojoObject(allPrescriptionDtos);
			}
		}.run();

		return response;
	}


}

