/**
 * 
 */
package gov.va.med.mhv.rxrefill.service.impl;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.mhv.rxrefill.data.model.Institution;
import gov.va.med.mhv.rxrefill.data.model.Prescription;
import gov.va.med.mhv.rxrefill.data.model.PrescriptionRequest;
import gov.va.med.mhv.rxrefill.data.model.Request;
import gov.va.med.mhv.rxrefill.data.model.RequestAttempt;
import gov.va.med.mhv.rxrefill.data.repository.PrescriptionRepository;
import gov.va.med.mhv.rxrefill.data.repository.PrescriptionRequestRepository;
import gov.va.med.mhv.rxrefill.data.repository.RequestRepository;
import gov.va.med.mhv.rxrefill.enums.PrescriptionStatusEnumeration;
import gov.va.med.mhv.rxrefill.enums.RequestStatusEnumeration;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;
import gov.va.med.mhv.rxrefill.service.PrescriptionRefreshResponseProcessor;
import gov.va.med.mhv.rxrefill.service.impl.util.UtilityMethods;
import gov.va.med.mhv.rxrefill.util.PrescriptionUtils;
import gov.va.med.mhv.rxrefill.util.RequestUtils;

/**
 * @author DNS   egberb
 *
 */
@Component
public class PrescriptionRefreshResponseProcessorImpl implements PrescriptionRefreshResponseProcessor {
	
	private static Logger LOG = LogManager.getLogger(PrescriptionRefreshResponseProcessorImpl.class);
	
	@Autowired
	private RequestRepository requestRepository;
	
	@Autowired
	private PrescriptionRepository prescriptionRepository;
	
	@Autowired
	private PrescriptionRequestRepository prescriptionRequestRepository;
	
	@Autowired
	RequestUtils requestUtils;

	@Autowired
	private PrescriptionUtils prescriptionUtils;
	
	/* (non-Javadoc)
	 * @see gov.va.med.mhv.rxrefill.service.PrescriptionRefreshResponseProcessor#processResponse(gov.va.med.mhv.rxrefill.data.model.Request, gov.va.med.mhv.rxrefill.data.model.RequestAttempt, java.util.List)
	 */
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public List<Prescription> processResponse(Request request, RequestAttempt attempt, List<Prescription> inPrescriptions)
	{
		return (requestUtils.isRefillRequest(request)) ? processRefillResults(request, attempt, inPrescriptions)
				: processExtractResults(request, attempt, inPrescriptions);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.mhv.rxrefill.service.PrescriptionRefreshResponseProcessor#processBadResponse(gov.va.med.mhv.rxrefill.data.model.Request, gov.va.med.mhv.rxrefill.data.model.RequestAttempt, java.lang.String)
	 */
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public void processBadResponse(Request request, RequestAttempt attempt, String newStatus)
	{
		if (LOG.isDebugEnabled()) {
			LOG.debug(">>>>>>>>>>>> in side processBadResponse.  newStatus::"+newStatus);
		}

		final Timestamp now = UtilityMethods.dateToTimestamp(new Date());
		if (attempt != null) {
			attempt.setStatus(newStatus);
			attempt.setModifiedDate(now);
		}
		request.setStatus(newStatus);
		request.setModifiedDate(now);
		//ServiceFactory.createEntityMaintenanceService().save(request);
		try{
			if (LOG.isDebugEnabled()) {
				LOG.debug(">>>>>>>>>>>> saving request.  request::"+request);
			}
			request = requestRepository.save(request);
		} catch(Exception e) {
			LOG.error("Error in storing request ", e);
			throw new MHVRuntimeException("Error in storing request " + e.getMessage(), e);
		}

		if (requestUtils.isRefillRequest(request)) {

			if (LOG.isDebugEnabled()) {
				LOG.debug(">>>>>>>>>>>> inside isRefillRequest. request::"+request);
			}

			// update prescription requests
			for (PrescriptionRequest prescriptionRequest : request.getPrescriptionRequests()) {

				if (LOG.isDebugEnabled()) {
					LOG.debug(">>>>>>>>>>>> inside for loop. prescriptionRequest::"+prescriptionRequest);
				}
				prescriptionRequest.setStatus(newStatus);
				//ServiceFactory.createEntityMaintenanceService().save(prescriptionRequest);
				try{

					if (LOG.isDebugEnabled()) {
						LOG.debug(">>>>>>>>>>>> before saving  prescriptionRequest::"+prescriptionRequest);
					}

					prescriptionRequest = prescriptionRequestRepository.save(prescriptionRequest);
				} catch(Exception e) {
					LOG.error("Error in storing prescriptionRequest ", e);
					throw new MHVRuntimeException("Error in storing prescriptionRequest " + e.getMessage(), e);
				}
			}
		}
	}
	
	private List<Prescription> processRefillResults(Request request, RequestAttempt attempt, List<Prescription> inPrescriptions)
	{
		final Timestamp now  = UtilityMethods.dateToTimestamp(new Date());
		List<Prescription> outPrescriptions = new ArrayList<Prescription>();
		if (LOG.isDebugEnabled()) {
			LOG.debug("Process " + (inPrescriptions != null ? inPrescriptions.size() : 0) + " refill results");
		}

		if (null != inPrescriptions && inPrescriptions.size() > 0) {
			for (Prescription inPrescription : inPrescriptions) {
				for (PrescriptionRequest prescriptionRequest : request.getPrescriptionRequests()) {
					Prescription storedPrescription = prescriptionRequest.getPrescription();
					if (inPrescription.getPrescriptionNumber().equals(storedPrescription.getPrescriptionNumber()))
					{
						// match found
						cloneRefillPrescriptionProperties(inPrescription, storedPrescription);
						storedPrescription.setModifiedDate(now);

						//storedPrescription = ServiceFactory.createEntityMaintenanceService().save(storedPrescription).getPrescription();
						try{
							storedPrescription = prescriptionRepository.save(storedPrescription);
						} catch(Exception e) {
							LOG.error("Error in storing storedPrescription in DB ", e);
							throw new MHVRuntimeException("Error in storing storedPrescription in DB " + e.getMessage(), e);
						}

						postProcessRefill(inPrescription, storedPrescription, prescriptionRequest);
						outPrescriptions.add(storedPrescription);
					}
				}
			}

			attempt.setModifiedDate(now);
			attempt.setStatus(RequestStatusEnumeration.SUCCESSFUL.getDescription());
			request.setModifiedDate(now);
			request.setStatus(RequestStatusEnumeration.SUCCESSFUL.getDescription());

			//ServiceFactory.createEntityMaintenanceService().save(request);
			try{
				request = requestRepository.save(request);
			} catch(Exception e) {
				LOG.error("Error in storing storedPrescription in DB ", e);
				throw new MHVRuntimeException("Error in storing storedPrescription in DB " + e.getMessage(), e);
			}
		} else {
			LOG.error("inPrescriptions is null");
			throw new MHVRuntimeException("inPrescriptions is null");
		}

		return outPrescriptions;
	}
	
	private List<Prescription> processExtractResults(Request request, RequestAttempt attempt, List<Prescription> inPrescriptions)
	{
		LOG.debug(">>>>>>>>>>>>> in side processExtractResults of RxRequestHelper - begining");
		Timestamp now = UtilityMethods.dateToTimestamp(new Date());
		List<Prescription> outPrescriptions = null;

		if (null != inPrescriptions && inPrescriptions.size() > 0) {
			int inPrescriptionCount = inPrescriptions.size();
			outPrescriptions = new ArrayList<Prescription>(inPrescriptionCount);
			if (LOG.isDebugEnabled()) {
				LOG.debug(String.format("ExtractResults contains %s records", inPrescriptionCount));
			}
			Long userId = request.getUserId();
			Institution institution = request.getInstitution();
			checkForAbsentActive(request, inPrescriptions);
			List<Prescription> currentPrescriptions = null;
			for (Prescription inPrescription: inPrescriptions) {
				if (LOG.isDebugEnabled()) {
					LOG.debug(String.format("userId: %s, institutionId: %s, RxNumber: %s", userId, institution.getId(), inPrescription.getPrescriptionNumber()));
					LOG.debug("inPrescription " + inPrescription);
				}
				try {
					if (LOG.isDebugEnabled()) {
						LOG.debug("Getting Current prescriptions for - "+inPrescription.getPrescriptionNumber()+" - "+institution.getId()+" - "+userId);
					}
					currentPrescriptions = prescriptionRepository.getPrescriptionByPatientSiteNumber(userId, institution.getId(), inPrescription.getPrescriptionNumber());
				} catch(Exception e) {
					LOG.error(String.format("Error in fetching currentPrescriptions userId: %s, institutionId: %s, RxNumber: %s", userId, institution.getId(), inPrescription.getPrescriptionNumber()), e);
					throw new MHVRuntimeException("Error in fetching currentPrescriptions " + e.getMessage(), e);
				}

				if (LOG.isDebugEnabled()) {
					LOG.debug("=========after search ======");
					LOG.debug("inPrescription " + inPrescription);
				}

				if (!currentPrescriptions.isEmpty()) {
					if (LOG.isDebugEnabled()) {
						LOG.debug("currentPrescriptions is not empty");
					}
					Prescription currentPrescription = currentPrescriptions.get(0);
					cloneExtractPrescriptionProperties(inPrescription, currentPrescription);
					currentPrescription.setModifiedDate(now);

					try {
						if (LOG.isDebugEnabled()) {
							LOG.debug("=========before saving currentPrescription======");
							LOG.debug("currentPrescription " + currentPrescription);
						}
						currentPrescription = prescriptionRepository.save(currentPrescription);
					} catch(Exception e) {
						LOG.error("Error in storing currentPrescription ", e);
						throw new MHVRuntimeException("Error in storing currentPrescription " + e.getMessage(), e);
					}

					outPrescriptions.add(currentPrescription);
				} else {
					if (LOG.isDebugEnabled()) {
						LOG.debug("currentPrescriptions is empty");
					}

					inPrescription.setCreatedDate(now);
					inPrescription.setModifiedDate(now);
					inPrescription.setUserId(userId);
					inPrescription.setInstitution(institution);
					//outPrescriptions.add(ServiceFactory.createEntityMaintenanceService().save(inPrescription).getPrescription());
					try {
						if (LOG.isDebugEnabled()) {
							LOG.debug("=========before saving inPrescription======");
							LOG.debug("inPrescription " + inPrescription);
						}
						inPrescription = prescriptionRepository.save(inPrescription);
					} catch(Exception e) {
						LOG.error("Error in storing inPrescription ", e);
						throw new MHVRuntimeException("Error in storing inPrescription " + e.getMessage(), e);
					}

					outPrescriptions.add(inPrescription);
				}

				if (LOG.isDebugEnabled()) {
					LOG.debug("outPrescriptions.size() " + outPrescriptions.size());
				}
			}
			attempt.setModifiedDate(now);
			attempt.setStatus(RequestStatusEnumeration.SUCCESSFUL.getDescription());
			request.setModifiedDate(now);
			request.setStatus(RequestStatusEnumeration.SUCCESSFUL.getDescription());

			//request = ServiceFactory.createEntityMaintenanceService().save(request).getRequest();
			try{
				request = requestRepository.save(request);
			} catch(Exception e) {
				LOG.error("Error in storing request ", e);
				throw new MHVRuntimeException("Error in storing request " + e.getMessage(), e);
			}
		} else {
			LOG.error("inPrescriptions is null");
			throw new MHVRuntimeException("inPrescriptions is null");
		}

		LOG.debug(">>>>>>>>>>>>> ending processExtractResults of RxRequestHelper");

		return outPrescriptions;
	}
	
	private void cloneRefillPrescriptionProperties(Prescription fromPrescription, Prescription toPrescription) {
		toPrescription.setMhvStatus(fromPrescription.getMhvStatus());
		if (fromPrescription.getStatus() != null) {
			toPrescription.setStatus(fromPrescription.getStatus());
			if (prescriptionUtils.isSubmitted(fromPrescription)) {
				toPrescription.setLastRefillSubmittedDate(new Date());
			}
		}
	}
	
	private void cloneExtractPrescriptionProperties(Prescription fromPrescription, Prescription toPrescription)
	{
		if (LOG.isDebugEnabled()) {
			LOG.debug(String.format("inside cloneExtractPrescriptionProperties RxId %s Rx# %s", fromPrescription.getId(), fromPrescription.getPrescriptionNumber()));
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("========= before clone - fromPrescription ======");
			LOG.debug("fromPrescription:: " + fromPrescription);
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("=========before clone - toPrescription ======");
			LOG.debug("toPrescription:: " + toPrescription);
		}
		toPrescription.setArchivedDate(fromPrescription.getArchivedDate());
		toPrescription.setDaysSupply(fromPrescription.getDaysSupply());
		toPrescription.setDivision(fromPrescription.getDivision());
		toPrescription.setDivisionName(fromPrescription.getDivisionName());
		toPrescription.setDrugName(fromPrescription.getDrugName());
		toPrescription.setExpirationCancelDate(fromPrescription.getExpirationCancelDate());
		toPrescription.setIen(fromPrescription.getIen());
		toPrescription.setIssueDateTime(fromPrescription.getIssueDateTime());
		toPrescription.setLastFillDate(fromPrescription.getLastFillDate());
		toPrescription.setMailWindow(fromPrescription.getMailWindow());
		toPrescription.setMhvStatus(fromPrescription.getMhvStatus());

		if (fromPrescription.getMhvStatusDate() != null) {
			int mhvStatus = (fromPrescription.getMhvStatus() != null ? fromPrescription.getMhvStatus() : 0);
			if (mhvStatus == 1 || mhvStatus == 2) {
				toPrescription.setMhvStatusDate(fromPrescription.getMhvStatusDate());
			}
		}

		toPrescription.setNumberOfRefills(fromPrescription.getNumberOfRefills());
		toPrescription.setPlacerOrderNumber(fromPrescription.getPlacerOrderNumber());
		//toPrescription.setId(fromPrescription.getId());
		toPrescription.setProviderFirstName(fromPrescription.getProviderFirstName());
		toPrescription.setProviderId(fromPrescription.getProviderId());
		toPrescription.setProviderLastName(fromPrescription.getProviderLastName());
		toPrescription.setQuantity(fromPrescription.getQuantity());
		toPrescription.setReleaseDateTime(fromPrescription.getReleaseDateTime());
		toPrescription.setRemarks(fromPrescription.getRemarks());
		toPrescription.setSig(fromPrescription.getSig());
		if (prescriptionUtils.isMHVPending(fromPrescription) && prescriptionUtils.isActive(fromPrescription)) {
			// prescription is waiting in mhv->vista queue but status shows as
			// active - then it is submitted
			toPrescription.setStatus(PrescriptionStatusEnumeration.SUBMITTED.getValue());
		} else {
			toPrescription.setStatus(fromPrescription.getStatus());
		}

		if (LOG.isDebugEnabled()) {
			LOG.debug("========= after clone - fromPrescription ======");
			LOG.debug("fromPrescription:: " + fromPrescription);
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("=========after clone - toPrescription ======");
			LOG.debug("toPrescription:: " + toPrescription);
		}

		if (LOG.isDebugEnabled()) {
			LOG.debug(String.format("exiting cloneExtractPrescriptionProperties RxId %s Rx# %s", fromPrescription.getId(), fromPrescription.getPrescriptionNumber()));
		}
	}
	
	private void postProcessRefill(Prescription inPrescription, Prescription storedPrescription, PrescriptionRequest prescriptionRequest) {
		final int MHV_STATUS_FILLED = 1;
		final int MHV_STATUS_PROCESSED = -1;

		int mhvStatus = inPrescription.getMhvStatus().intValue();

		if (mhvStatus == MHV_STATUS_FILLED || mhvStatus == MHV_STATUS_PROCESSED) {
			prescriptionRequest.setStatus(RequestStatusEnumeration.SUCCESSFUL.getDescription());
		} else {
			prescriptionRequest.setStatus(RequestStatusEnumeration.REJECTED.getDescription());
			if (LOG.isDebugEnabled()) {
				LOG.debug(String.format("Prescription refill request for prescription %s at facility %s was rejected with a status of %s and mhv status code of %s",
						storedPrescription.getPrescriptionNumber(), storedPrescription.getInstitution().getInstitutionId(), inPrescription.getIen(), mhvStatus));
			}
		}
	}

	private void checkForAbsentActive(Request request, List<Prescription> inPrescriptions)
	{
		Map<String, Prescription> incomingPrescriptions = new HashMap<String, Prescription>();
		for (Prescription prescription : inPrescriptions) {
			incomingPrescriptions.put(prescription.getPrescriptionNumber(), prescription);
		}

		List<Prescription> userActivePrescriptions = null;
		//PrescriptionBO.getActivePatientPrescriptions(request.getUserId());
		try{
			userActivePrescriptions = prescriptionRepository.getActivePatientPrescriptions(request.getUserId());
		} catch(Exception e) {
			LOG.error("Error in fetching userActivePrescriptions ", e);
			throw new MHVRuntimeException("Error in fetching userActivePrescriptions " + e.getMessage(), e);
		}

		for (Prescription prescription : userActivePrescriptions) {
			if (hasSameInstitution(request, prescription) && !incomingPrescriptions.containsKey(prescription.getId()))
			{
				prescription.setStatus(PrescriptionStatusEnumeration.UNKNOWN.getValue());
			}
		}
	}
	
	private boolean hasSameInstitution(Request request, Prescription prescription)
	{
		return (prescription != null) && (request != null)
		    && (prescription.getInstitution() != null)
		    && (prescription.getInstitution().getInstitutionId() != null)
		    && (request.getInstitution() != null)
		    && prescription.getInstitution().getInstitutionId().equals(request.getInstitution().getInstitutionId());
	}
}
