package gov.va.med.mhv.rxrefill.service.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.PropertySource;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.rxrefill.converter.PharmacyPatientConverter;
import gov.va.med.mhv.rxrefill.converter.RequestConverter;
import gov.va.med.mhv.rxrefill.data.model.PharmacyPatient;
import gov.va.med.mhv.rxrefill.data.model.Request;
import gov.va.med.mhv.rxrefill.data.repository.PatientRepository;
import gov.va.med.mhv.rxrefill.data.repository.PharmacyPatientRepository;
import gov.va.med.mhv.rxrefill.dto.PharmacyPatientDTO;
import gov.va.med.mhv.rxrefill.dto.PrescriptionDTO;
import gov.va.med.mhv.rxrefill.dto.RequestDTO;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;
import gov.va.med.mhv.rxrefill.service.PharmacyRefillWebService;
import gov.va.med.mhv.rxrefill.service.PrescriptionService;

@Component(value="prService")
@PropertySource("classpath:/${MHV_ENV_PROPERTY}.rxrefill.application.properties")
public class PharmacyRefillServiceImpl implements PharmacyRefillWebService{

    private static Logger log = LogManager.getLogger(PharmacyRefillServiceImpl.class);

	@Autowired
	private PrescriptionService prescriptionService;

	@Autowired
	private PharmacyPatientRepository pharmacyPatientRepository;
	
	@Autowired
	private PharmacyPatientConverter pharmacyPatientConverter;

	@Autowired
	private RequestConverter requestConverter;

	@Override
	public PharmacyPatientDTO getPharmacyPatient(Long userProfileId) throws MHVException {
		PharmacyPatient pharmacyPatient = null;
		PharmacyPatientDTO pharmacyPatientDto = null;

		try {
			pharmacyPatient = loadPharmacyPatient(userProfileId);

			if (null != pharmacyPatient) {
				pharmacyPatientDto = pharmacyPatientConverter.convertDomainPharmacyPatient(pharmacyPatient);
			}
		} catch (Exception e) {
			log.error("Exception Caught Getting Pharmacy Patient <<<" + userProfileId + ">>> ", e);
			throw new MHVException("Exception Caught Getting Pharmacy Patient", e);
		}

		return pharmacyPatientDto;
	}
	

	/**
	 * Method load the pharmacy patient. If the pharmacy patient does 
	 * not exist, a new pharmacy patient is created based on the Patient.
	 * @param userProfileId
	 * @return
	 */
	private PharmacyPatient loadPharmacyPatient(Long userProfileId) {
		PharmacyPatient pharmacyPatient = null;
		
		if (userProfileId == null) {
			log.error("UserProfile is null");
			throw new MHVRuntimeException("UserProfile is null");
		}

		try {
			pharmacyPatient = pharmacyPatientRepository.getPharmacyPatient(userProfileId);
			
			if (pharmacyPatient == null) {
				pharmacyPatient = this.prescriptionService.createPharmacyPatient(userProfileId);
			}
			
		} catch (Exception e) {
			log.error("Error in fetching pharmacyPatient ", e);
			throw new MHVRuntimeException("Error in fetching pharmacyPatient " + e.getMessage(), e);
		}

		return pharmacyPatient;
	}
	
	@Override
	public ResponseUtil<List<PrescriptionDTO>> getActivePrescriptions(Long userProfileId) throws MHVException {
		PharmacyPatient pharmacyPatient = null;
		ResponseUtil<List<PrescriptionDTO>> rxResponse = new ResponseUtil();
		List<PrescriptionDTO> activePrescriptionDtos = null;

		try {
			pharmacyPatient = loadPharmacyPatient(userProfileId);

			rxResponse = prescriptionService.getActivePrescriptions(pharmacyPatient);
			if(log.isDebugEnabled()){
				log.debug("PharmacyRefillServiceImpl - response.getFailureMessage()" + rxResponse.getFailureMessage());
			}

	        if (!rxResponse.isFailure()) {
	        	activePrescriptionDtos = (List<PrescriptionDTO>) rxResponse.getPojoObject();
	        }

		} catch (Exception e) {
			log.error("Exception Caught in getActivePrescriptions<<<" + userProfileId + ">>> ", e);
			throw new MHVException("Exception Caught in getActivePrescriptions", e);
		}

		//return activePrescriptionDtos;
		return rxResponse;
	}

	@Override
	public List<RequestDTO> refillPrescriptions(Long userProfileId, List<PrescriptionDTO> refillList) throws MHVException {
		PharmacyPatient pharmacyPatient = null;
		ResponseUtil rxResponse = new ResponseUtil();
		List<RequestDTO> refillRequests = null;

		if (log.isDebugEnabled()) {
			log.debug("005 >>>>>>>>>>> inside PharmacyRefillServiceImpl:refillPrescriptions. calling getPharmPatient - userName::"+userProfileId+" --- refillList Count::"+refillList.size());
		}

		try {
			pharmacyPatient = loadPharmacyPatient(userProfileId);

			if (log.isDebugEnabled()) {
				log.debug("006 >>>>>>>>>>> inside PharmacyRefillServiceImpl:refillPrescriptions. calling refillPrescriptions - pharmacyPatient::"+pharmacyPatient);
			}

			rxResponse = prescriptionService.refillPrescriptions(pharmacyPatient, refillList);

			if (log.isDebugEnabled()) {
				log.debug("017 >>>>>>>>>>> inside PharmacyRefillServiceImpl:refillPrescriptions. After calling refillPrescriptions - rxResponse::"+rxResponse.isFailure());
			}

	        if (!rxResponse.isFailure()) {

				if (log.isDebugEnabled()) {
					log.debug("018 >>>>>>>>>>> inside PharmacyRefillServiceImpl:refillPrescriptions. inside rxResponse - true");
				}

	        	refillRequests = (List<RequestDTO>) rxResponse.getPojoObject();
	        	if (refillRequests != null) {
					if (log.isDebugEnabled()) {
						log.debug("018a >>>>>>>>>>> inside PharmacyRefillServiceImpl:refillPrescriptions. refillRequests - count::"+refillRequests.size());
					}
				} else {
					if (log.isDebugEnabled()) {
						log.debug("018b >>>>>>>>>>> inside PharmacyRefillServiceImpl:refillPrescriptions. refillRequests - is null");
					}
				}
	        }
		} catch (Exception e) {
			log.error("Exception Caught in getActivePrescriptions <<<" + userProfileId + ">>> ", e);
			throw new MHVException("Exception Caught in getActivePrescriptions", e);
		}

		return refillRequests;
	}

	@Override
	public RequestDTO getLastSuccessfulExtractRequest(Long userProfileId) throws MHVException {
		PharmacyPatient pharmacyPatient = null;
		List<Request> requestList = null;
		RequestDTO requestDto = null;

		try {
			pharmacyPatient = loadPharmacyPatient(userProfileId);

			requestList = prescriptionService.getLastSuccessfulExtractRequestForUser(pharmacyPatient);

			if (null != requestList && requestList.size() > 0) {
				requestDto = requestConverter.convertDomainRequest(requestList.get(0));
			} else {
				log.error("requestList is null. <<<" + userProfileId + ">>> ");
			}

		} catch (Exception e) {
			log.error("Exception Caught in getLastSuccessfulExtractRequest <<<" + userProfileId + ">>> ", e);
			throw new MHVException("Exception Caught in getLastSuccessfulExtractRequest", e);
		}

		return requestDto;
	}
	
	@Override
	public List<RequestDTO> getRequestByFunctionAndInstitution(Long userId, Long institutionId) throws MHVException {
		List<Request> refillRequests = null;
		List<RequestDTO> refillRequestDtos = null;

		if (log.isDebugEnabled()) {
			log.debug("005 >>>>>>>>>>> inside PharmacyRefillServiceImpl:getRequestByFunctionAndInstitution. calling getRequestByFunctionAndInstitution - userName::"+userId+" --- refillList Count::");
		}

		try {
			refillRequests = prescriptionService.getRequestByFunctionAndInstitution(userId, institutionId);
			if (log.isDebugEnabled()) {
				log.debug("017 >>>>>>>>>>> inside PharmacyRefillServiceImpl:getRequestByFunctionAndInstitution. After calling getRequestByFunctionAndInstitution - rxResponse::");
			}
			if(refillRequests == null && refillRequests.size() == 0) {
				refillRequests = new ArrayList<Request>();
			}
			
			refillRequestDtos = (null != refillRequests) ? new RequestConverter().convertDomainRequestList(refillRequests) : null;

		} catch (Exception e) {
			log.error("Exception Caught in getActivePrescriptions <<<" + userId + ">>> ", e);
			throw new MHVException("Exception Caught in getRequestByFunctionAndInstitution", e);
		}

		return refillRequestDtos;
	}
	

	@Override
	public List<PrescriptionDTO> getPrescriptionsHistory(Long userProfileId) throws MHVException {
		PharmacyPatient pharmacyPatient = null;
		ResponseUtil rxResponse = new ResponseUtil();
		List<PrescriptionDTO> historyPrescriptionDtos = null;

		try {
			pharmacyPatient = loadPharmacyPatient(userProfileId);

			rxResponse = prescriptionService.getHistoricalPrescriptionsWithRequestStatuses(pharmacyPatient);

	        if (!rxResponse.isFailure()) {
	        	historyPrescriptionDtos = (List<PrescriptionDTO>) rxResponse.getPojoObject();
	        }

		} catch (Exception e) {
			log.error("Exception Caught in getPrescriptionsHistory <<<" + userProfileId + ">>> ", e);
			throw new MHVException("Exception Caught in getPrescriptionsHistory", e);
		}

		return historyPrescriptionDtos;
	}

	@Override
	public List<PrescriptionDTO> getAllPrescriptions(Long userProfileId)
			throws MHVException {
		PharmacyPatient pharmacyPatient = null;
		ResponseUtil rxResponse = new ResponseUtil();
		List<PrescriptionDTO> allPrescriptionDtos = null;

		try {
			pharmacyPatient = loadPharmacyPatient(userProfileId);

			rxResponse = prescriptionService.getAllPrescriptions(pharmacyPatient);

	        if (!rxResponse.isFailure()) {
	        	allPrescriptionDtos = (List<PrescriptionDTO>) rxResponse.getPojoObject();
	        }

		} catch (Exception e) {
			log.error("Exception Caught in getActivePrescriptions <<<" + userProfileId + ">>> ", e);
			throw new MHVException("Exception Caught in getActivePrescriptions", e);
		}

		return allPrescriptionDtos;
	}

}
