/**
 * 
 */
package gov.va.med.mhv.rxrefill.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.mhv.common.data.model.Facility;
import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.common.data.model.PatientRegistryChange;
import gov.va.med.mhv.rxrefill.data.model.Institution;
import gov.va.med.mhv.rxrefill.data.model.PharmPatientChange;
import gov.va.med.mhv.rxrefill.data.model.PharmacyPatient;
import gov.va.med.mhv.rxrefill.data.model.Prescription;
import gov.va.med.mhv.rxrefill.data.model.Request;
import gov.va.med.mhv.rxrefill.data.repository.PharmacyPatientRepository;
import gov.va.med.mhv.rxrefill.data.repository.PrescriptionRepository;
import gov.va.med.mhv.rxrefill.data.repository.RequestRepository;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;
import gov.va.med.mhv.rxrefill.service.PatientChangeProcessor;
import gov.va.med.mhv.rxrefill.service.PrescriptionService;
import gov.va.med.mhv.rxrefill.service.impl.util.PatientChanges;
import gov.va.med.mhv.rxrefill.util.PrescriptionUtils;

/**
 * @author DNS   egberb
 *
 */
@Component
public class PatientChangeProcessorImpl implements PatientChangeProcessor {
	
	private static Logger log = LogManager.getLogger(PatientChangeProcessorImpl.class);
	
	@Autowired
	private PrescriptionUtils prescriptionUtils;
	
	@Autowired
	private PrescriptionRepository prescriptionRepository;
	
	@Autowired
	private PharmacyPatientRepository pharmacyPatientRepository;
	
	@Autowired
	private RequestRepository requestRepository;
	
	@Autowired
	private PrescriptionService prescriptionService;

	/*
	 * (non-Javadoc)
	 * @see gov.va.med.mhv.rxrefill.service.PatientChangeProcessor#processPatientChanges(gov.va.med.mhv.rxrefill.data.model.PharmacyPatient)
	 */
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public PatientChanges processPatientChanges(PharmacyPatient pharmacyPatient) {
		PharmPatientChange mostRecentChange = addNewPharmacyChanges(pharmacyPatient);
		if (mostRecentChange == null) {
			log.debug("All patient registration changes has been accounted for in pharmacy data");
			return null;
		}

		Patient patient = pharmacyPatient.getPatient();
		PatientChanges patientChanges = new PatientChanges();
		PatientRegistryChange patientChange = mostRecentChange.getPatientRegistryChange();
		if (StringUtils.isBlank(patientChange.getOldIcn())) {
			markRemovedFacilityRecordsAsInactive(patient, patientChange);
		} else {
			markAllFacilityRecordsAsInactive(patient, patientChanges);
		}
		return patientChanges;
	}

	/*
	 * (non-Javadoc)
	 * @see gov.va.med.mhv.rxrefill.service.PatientChangeProcessor#postProcessPatientChanges(long, gov.va.med.mhv.rxrefill.service.impl.util.PatientChanges, java.util.List)
	 */
	@Override
	@Transactional (propagation = Propagation.REQUIRES_NEW)
	public void postProcessPatientChanges(long userId, PatientChanges patientChanges, List<Prescription> prescriptions) {

		if (null != prescriptions && prescriptions.size() > 0) {
			
			for (Prescription prescription: prescriptions) {
				if (patientChanges != null) {
					Prescription deletedPrescription = patientChanges.findInactivePrescription(prescription.getPrescriptionNumber());
					if (prescriptionUtils.isSubmitted(deletedPrescription)) {
						// This is necessary, because otherwise the last submitted
						// date is lost, as it is MHV only data
						prescription.setLastRefillSubmittedDate(deletedPrescription.getLastRefillSubmittedDate());
						prescriptionRepository.save(prescription);
					}
				}
			}
		} 		
	}
	
	private void markRemovedFacilityRecordsAsInactive(Patient patient, PatientRegistryChange patientRegChange) {
		if (log.isDebugEnabled()) {
			log.debug("Patient's (" + PrescriptionServiceImpl.describe(patient) + ") registration has been updated since the last request: marking only records from removed facilities as inactive");
		}
		Long userId = patient.getUserProfile().getId();
		//TODO: is map size should be 10?? - Prasad
		Map<String, Institution> oldInstitutionMap = new HashMap<String, Institution>(10);
		List<Request> latestRequestsList = new ArrayList<Request>();

		try {
			latestRequestsList = requestRepository.getLatestRequestsByFacility(userId);
		} catch(Exception e) {
			log.error("Error in fetching latestRequestsList. userProfileid::  <<<" + patient.getUserProfile().getId() + ">>> ", e);
			throw new MHVRuntimeException("Error in fetching latestRequestsList " + e.getMessage(), e);
		}

		if (null != latestRequestsList && latestRequestsList.size() > 0) {
			for (Request request : latestRequestsList) {
				Institution institution = request.getInstitution();
				if (null != institution) {
					oldInstitutionMap.put(institution.getStationNumber(), institution);
				}
			}

			// compare new institutions to the old institutions
			Set<Institution> validStationNumberSet = new HashSet<Institution>();
			for (Facility facility : patient.getFacilities()) {
				Institution institution = oldInstitutionMap.get(facility.getName());
				if (institution != null) {
					validStationNumberSet.add(institution);
				}
			}

			// next, remove valid institutions from the old set. This set will then
			// be used to mark old facilities as deleted.
			for (Institution institution : validStationNumberSet) {
				Institution removed = oldInstitutionMap.remove(institution.getStationNumber());
				assert (removed != null) : "Failed to remove " + institution.getStationNumber() + " from oldInstitutionMap";
				if (log.isDebugEnabled()) {
					log.debug("Keep requests and prescriptions for user '" + patient.getUserProfile().getUserName()
							+ "' and current facility " + institution.getStationNumber() + "'");
				}
			}

			// finally, mark invalid institutions as deleted:
			for (Institution institution : oldInstitutionMap.values()) {
				if (log.isDebugEnabled()) {
					log.debug("Delete requests and prescriptions for user '" + patient.getUserProfile().getUserName()
							+ "' and old facility " + institution.getStationNumber() + "'");
				}
				this.prescriptionService.deletePrescriptionsForUserInstitutions(userId, institution.getId());
			}
		} else {
			log.error("latestRequestsList is null");
		}
	}
	
	/**
	 * Sets all prescriptions and requests to a status of "deleted" for the
	 * given patient. At no time are these records actually deleted from the
	 * database.
	 * @param patient The <tt>Patient</tt> for which to mark existing records
	 *        as "deleted"
	 */
	@Transactional
	private void markAllFacilityRecordsAsInactive(Patient patient, PatientChanges patientChanges)
	{
		if (log.isDebugEnabled()) {
			log.debug("Patient's (" + PrescriptionServiceImpl.describe(patient) + ") registration"
				+ " has been updated since the last request:"
				+ " marking records from all facilities as inactive");
		}
		Long userId = patient.getUserProfile().getId();

		List<Request> requests = null;
		try {
			requests = requestRepository.getRequestsForPatient(userId);
		} catch(Exception e) {
			log.error("Error in fetching requests for patient. userProfileid::  <<<" + userId + ">>> ", e);
			throw new MHVRuntimeException("Error in fetching requests for patient " + e.getMessage(), e);
		}

		//TODO: is updated request stored in DB? - Prasad
		for (Request request: requests) {
			request.setIsInactive(true);
		}
		
		List<Prescription> prescriptions = null;
		try {
			prescriptions = prescriptionRepository.getPrescriptionsForPatient(userId);
		} catch(Exception e) {
			log.error("Error in fetching prescriptions for patient. userProfileid::  <<<" + userId + ">>> ", e);
			throw new MHVRuntimeException("Error in fetching prescriptions for patient " + e.getMessage(), e);
		}

		//TODO: is updated prescription stored in DB? - Prasad
		for (Prescription prescription : prescriptions) {
			prescription.setIsInactive(true);
			patientChanges.addInactivePrescription(prescription);
		}
	}
	
	/**
	 *
	 * @param pharmacyPatient
	 * @param changeSet
	 * @param pharmacyChangeSet
	 * @return The most recent change
	 */
	private PharmPatientChange addNewPharmacyChanges(PharmacyPatient pharmacyPatient) {
		PharmPatientChange mostRecentChange = null;
		Date mostRecentPharmacyChange = null;
		Map<Long, PharmPatientChange> pharmacyChangesByKey = new HashMap<Long, PharmPatientChange>();

		for (PharmPatientChange change : pharmacyPatient.getPharmacyPatientChanges()) {
			pharmacyChangesByKey.put(change.getPatientRegistryChangeId(), change);
			Date changeDate = change.getPatientRegistryChange().getRecordedOnDate();
			if ((mostRecentPharmacyChange == null) || changeDate.before(mostRecentPharmacyChange)) {
				mostRecentPharmacyChange = changeDate;
			}
		}

		for (PatientRegistryChange patientChange : pharmacyPatient.getPatient().getPatientRegistryChanges()) {
			
			if (pharmacyChangesByKey.containsKey(patientChange.getId())) {
				// Already exists in the pharmacy changes
				continue;
			}
			
			Date changeDate = patientChange.getRecordedOnDate();
			
			// Create a new instance
			PharmPatientChange change = new PharmPatientChange();
			change.setPatientRegistryChange(patientChange);
			change.setParentPharmacyPatient(pharmacyPatient);
			
			pharmacyPatient.addPharmacyPatientChange(change);
			
			if (((mostRecentChange == null) || changeDate.after(mostRecentChange.getPatientRegistryChange().getRecordedOnDate()))
					// a new change must be the most recent patient registry
					// change
					&& ((mostRecentPharmacyChange == null) || mostRecentPharmacyChange.before(changeDate)))
			// a new change must be newer than the most recent pharmacy
			// change that has already applied
			{
				mostRecentChange = change;
				if (log.isDebugEnabled()) {
					log.debug("Found a new/more recent registration change " + PrescriptionServiceImpl.describe(mostRecentChange));
				}
			}
		}
		
		return mostRecentChange;
	}
	
}
