/**
 * 
 */
package gov.va.med.mhv.rxrefill.service.impl;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.common.data.model.Facility;
import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.rxrefill.data.model.Institution;
import gov.va.med.mhv.rxrefill.data.model.Prescription;
import gov.va.med.mhv.rxrefill.data.model.PrescriptionRequest;
import gov.va.med.mhv.rxrefill.data.model.Request;
import gov.va.med.mhv.rxrefill.data.model.RequestAttempt;
import gov.va.med.mhv.rxrefill.data.repository.RequestRepository;
import gov.va.med.mhv.rxrefill.enums.RequestFunctionEnumeration;
import gov.va.med.mhv.rxrefill.enums.RequestStatusEnumeration;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;
import gov.va.med.mhv.rxrefill.service.ExtractRequestCreator;
import gov.va.med.mhv.rxrefill.service.impl.util.RxRequestMessages;
import gov.va.med.mhv.rxrefill.service.impl.util.RxRequestStrategy;
import gov.va.med.mhv.rxrefill.service.impl.util.UtilityMethods;
import gov.va.med.mhv.rxrefill.util.RxMessagesUtil;

/**
 * @author DNS   egberb
 *
 */
@Component
public class ExtractRequestCreatorImpl implements ExtractRequestCreator {
	
	private static Logger LOG = LogManager.getLogger(ExtractRequestCreatorImpl.class);
	
	@Autowired
	private RequestRepository requestRepository;
	
	@Autowired
	private UtilityMethods utilityMethods;

	@Autowired
	private RxRequestStrategy strategy;
	
	@Autowired
	private RxMessagesUtil rxMessagesUtil;
	
	/**
	 * Create the extract requests and save them to the database so that subsequent requests 
	 * coming into the system are not duplicated.
	 * @param patient The Patient for whom we are creating the extractions requests for
	 * @param function The type of extract (Historical, Active, Refill, All)
	 * @param response The response object that the results where information relevant to the 
	 * request can be stored.
	 * @return The list of requests for the patient.
	 */
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public List<Request> multipleExtractRequests(Patient patient, int function, ResponseUtil response) {

		if (LOG.isDebugEnabled()) {
					LOG.debug(" 100 >>>>>>>>>>>> in side multiplexExtractRequests - begining");
		}
		List<Request> requestList = new ArrayList<Request>();
		//List<Request> result = new ArrayList<Request>();
		for (Facility facility : patient.getFacilities()) {
			if (LOG.isDebugEnabled()) {
						LOG.debug(" 100a >>>>>>>>>>>> Facility Name:: "+facility.getName());
			}
			Institution institution = utilityMethods.getInstitution(facility.getName());
			if (LOG.isDebugEnabled()) {
						LOG.debug(" 100b >>>>>>>>>>>> Institution Name:: "+institution.getName());
			}
			if (validateInstitution(institution)) {
				if (LOG.isDebugEnabled()) {
							LOG.debug(" 100d >>>>>>>>>>>> Inside if block of testInstitution...  Function::"+function);

							LOG.debug("RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE::"+RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue());
							LOG.debug("strategy.checkSendHistoricalUpdateRequest(patient, institution)::"+strategy.checkSendHistoricalUpdateRequest(patient, institution));

							LOG.debug("RequestFunctionEnumeration.GETPRESCRIPTIONSTATUS::"+RequestFunctionEnumeration.GETPRESCRIPTIONSTATUS.getValue());
							LOG.debug("strategy.checkSendActiveUpdateRequest(patient, institution)::"+strategy.checkSendActiveUpdateRequest(patient, institution));
				}
				boolean exhaustedTries = true;
				
				Request request = null;
				final Date now = new Date();
				if ((RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue() == function
						&& strategy.checkSendHistoricalUpdateRequest(patient, institution))
						|| (RequestFunctionEnumeration.GETPRESCRIPTIONSTATUS.getValue() == function
						&& strategy.checkSendActiveUpdateRequest(patient, institution))) {
					// if there is prior profile or active extract and
					// corresponding function requested - create update request
					Date cutoffDate = strategy.checkRxExtractCutOffDate(patient, institution);
					if (LOG.isDebugEnabled()) {
								LOG.debug(" 101 >>>>>>>>>>>> in side multiplexExtractRequests - calling createExtractRequest -1");
					}
					request = createExtractRequest(patient, institution, now, cutoffDate, RequestFunctionEnumeration.GETPRESCRIPTIONUPDATE.getValue());
				} else if (RequestFunctionEnumeration.GETPRESCRIPTIONSTATUS.getValue() == function
						&& strategy.checkSendStatusRequest(patient, institution)) {
					// if request for active and there is no prior profile or
					// active extracts
					if (LOG.isDebugEnabled()) {
								LOG.debug(" 102 >>>>>>>>>>>> in side multiplexExtractRequests - calling createExtractRequest -2");
					}
					request = createExtractRequest(patient, institution, now, null, function);
				}

				if (request != null) {
					requestList.add(request);
					exhaustedTries = false;
				}
				if (strategy.checkSendProfileRequest(patient, institution)) {
					// if there is no profile extract - create profile request
					// for any function
					if (LOG.isDebugEnabled()) {
								LOG.debug(" 103 >>>>>>>>>>>> in side multiplexExtractRequests - calling createExtractRequest -3");
					}
					request = createExtractRequest(patient, institution, now, null, RequestFunctionEnumeration.GETPRESCRIPTIONPROFILE.getValue());
					if (request != null) {
						requestList.add(request);
						exhaustedTries = false;
					}
				}
				if (exhaustedTries) {
					if (LOG.isDebugEnabled()) {
								LOG.debug(" 104 >>>>>>>>>>>> in side multiplexExtractRequests - exhaustedTryes is true.. Request is null..");
					}
					addExhaustedTriesMessage(response, patient, now, institution, function);
				}
			} else if (institution == null) {
				LOG.error(String.format("Building request information for facility '%s', but not registered as an institution. Please check INSTITUTION(S) table.", facility.getName()));
				response.setFailure(true);
				response.setFailureMessage(rxMessagesUtil.getErrorVistaSystemfailure());
			} else {
				if (LOG.isDebugEnabled()) {
							LOG.debug(" 100d >>>>>>>>>>>> Inside else block of testInstitution...");
				}
				// User cannot change situation, so not alerting user,
				// just ignoring.
				LOG.info(String.format("Request information for patient %s from institution (Name: %s; SN: %s) which is not a treatment nor an active facility",
						patient.getId(), institution.getName(), institution.getStationNumber()));
			}
		}
		
		return requestList;
	}
	
	/**
	 * @param patient
	 * @param institution
	 * @param createdDate
	 * @return
	 */
	@Override
	@Transactional
	public Request createExtractRequest(Patient patient, Institution institution, Date createdDate, Date cutOffDate, int function) {
		if (LOG.isDebugEnabled()) {
					LOG.debug(" 110 >>>>>>>>>>>> in side createExtractRequest - begining");
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug(String.format("Adding %s request for site %s ...", function, institution.getStationNumber()));
		}

		final Timestamp createdAt = UtilityMethods.dateToTimestamp(createdDate);
		final Timestamp cutoffAt = UtilityMethods.dateToTimestamp(cutOffDate);
		Long userId = (patient.getUserProfile().getId());
		final Timestamp fromDate = UtilityMethods.dateToTimestamp(UtilityMethods.getDateWithoutTime(createdDate));
		final Timestamp toDate = UtilityMethods.dateToTimestamp(new Date(fromDate.getTime() + 24 * 60 * 60 * 1000));
		//List requests = RequestBO.getRequests(userId, function, institution.getInstitutionId(), fromDate, toDate);
		List<Request> requests = null;
		try {
			requests = requestRepository.getRequests(userId, function, institution.getInstitutionId(), fromDate, toDate);
		} catch(Exception e) {
			String message = String.format("Error retrieving requests during request creation. User Profile ID: %d, Institution Station: %s, Function: %d  ", 
					patient.getUserProfileId(), institution.getStationNumber(), function);
			LOG.error(message, e);
			throw new MHVRuntimeException(message + e.getMessage(), e);
		}
		Request request = null;
		
		if (requests.isEmpty()) {
			// create request
			request = new Request();
			request.setUserId(userId);
			request.setCreatedDate(createdAt);
			request.setInstitution(institution);
			request.setRequestFunction(function);
		} else {
			// use existing
			request = (Request) requests.get(0);
		}

		request.setStatus(RequestStatusEnumeration.INPROGRESS.getDescription());
		if (LOG.isDebugEnabled()) {
					LOG.debug(" 111 >>>>>>>>>>>> in side createExtractRequest - calling setModifiedDate :: "+createdAt);
		}
		request.setModifiedDate(createdAt);
		if (LOG.isDebugEnabled()) {
					LOG.debug(" 112 >>>>>>>>>>>> in side createExtractRequest - calling setStartDate :: "+cutoffAt);
		}
		request.setStartDate(cutoffAt);
		RequestAttempt attempt = new RequestAttempt();
		attempt.setParentRequest(request);
		attempt.setStatus(RequestStatusEnumeration.INPROGRESS.getDescription());
		attempt.setCreatedDate(createdAt);
		attempt.setModifiedDate(createdAt);
		request.addRequestAttempt(attempt);

		try {
			request = requestRepository.save(request);
		} catch(Exception e) {
			String message = String.format("Error in storing request in DB. User Profile ID: %d, Institution Station: %s, Function: %d  ", 
					patient.getUserProfileId(), institution.getStationNumber(), function);
			LOG.error(message, e);
			throw new MHVRuntimeException(message + e.getMessage(), e);
		}

		if (LOG.isDebugEnabled()) {
			LOG.debug(String.format("created request for site %s ...", institution.getStationNumber()));
			LOG.debug(String.format("Request Id %s ...", request.getId()));
		}

		return request;
	}
	
	/**
	 * @param createdDate
	 * @param prescriptions
	 * @return
	 */
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public Request createRefillRequest(Timestamp now, List<Prescription> prescriptions) {
		Prescription prescription = prescriptions.get(0);

		Institution institution = prescription.getInstitution();
		Long userId = prescription.getUserId();

		Timestamp fromDate = UtilityMethods.dateToTimestamp(UtilityMethods.getDateWithoutTime(now));
		Timestamp toDate = UtilityMethods.dateToTimestamp(new Date(fromDate.getTime() + 24 * 60 * 60 * 1000));

		List<Request> requests = null;
		try {
			requests = requestRepository.getRequests(userId, RequestFunctionEnumeration.REFILLPRESCRIPTION.getValue(), institution.getInstitutionId(), fromDate, toDate);
		} catch(Exception e) {
			String message = String.format("Error getting requests prior to refill request creation. User Profile ID: %d, Institution Station: %s ", 
					userId, institution.getStationNumber());
			LOG.error(message, e);
			throw new MHVRuntimeException(message + e.getMessage(), e);
		}

		Request request = (requests.isEmpty())
			? createRefillRequest(userId, now, institution)  // create new
			: (Request) requests.get(0); // use existing

		request.setStatus(RequestStatusEnumeration.INPROGRESS.getDescription());
		request.setModifiedDate(now);
		addAttempt(request, now, RequestStatusEnumeration.INPROGRESS.getDescription());
		addPrescriptions(request, prescriptions, now, RequestStatusEnumeration.INPROGRESS.getDescription());

		//RequestServiceResponse response = ServiceFactory.createEntityMaintenanceService().save(request);
		try{
			
			request = requestRepository.save(request);
			if (LOG.isDebugEnabled()) {
						LOG.debug(" 70i >>>>>>>>>>>> request after saving requestRepository.::"+request);
			}

		} catch(Exception e) {
			String message = String.format("Error getting requests prior to refill request creation. User Profile ID: %d, Institution Station: %s ", 
					userId, institution.getStationNumber());
			LOG.error(message, e);
			throw new MHVRuntimeException(message + e.getMessage(), e);
		}

		return request;
	}
	
	private Request createRefillRequest(Long userId, Timestamp now, Institution institution) {
		// create request
		Request request = new Request();
		request.setUserId(userId);
		request.setCreatedDate(now);
		request.setInstitution(institution);
		request.setRequestFunction(RequestFunctionEnumeration.REFILLPRESCRIPTION.getValue());
		return request;
	}

	private void addAttempt(Request request, Timestamp now, String status) {
		// create prescription request attempt
		RequestAttempt attempt = new RequestAttempt();
		attempt.setParentRequest(request);
		attempt.setStatus(status);
		attempt.setCreatedDate(now);
		attempt.setModifiedDate(now);
		request.addRequestAttempt(attempt);
	}

	private void addPrescriptions(Request request, List<Prescription> prescriptions, Timestamp now, String status) {
		if (LOG.isDebugEnabled()) {
					LOG.debug(" 70c >>>>>>>>>>>> in addPrescriptions. prescriptions size "+prescriptions.size()+" -- status value::"+status);
		}

		for (Prescription prescription : prescriptions) {

			if (LOG.isDebugEnabled()) {
						LOG.debug(" 70d >>>>>>>>>>>> in side for loop of addPrescriptions.");
			}

			// create prescription request
			PrescriptionRequest prescriptionRequest = new PrescriptionRequest();
			prescriptionRequest.setParentRequest(request);
			prescriptionRequest.setCreatedDate(now);
			prescriptionRequest.setModifiedDate(now);
			prescriptionRequest.setPrescription(prescription);
			prescriptionRequest.setStatus(status);

			if (LOG.isDebugEnabled()) {
						LOG.debug(" 70e >>>>>>>>>>>> prescriptionRequest - ."+ prescriptionRequest);
			}

			request.addPrescriptionRequest(prescriptionRequest);
		}

			if (LOG.isDebugEnabled()) {
						LOG.debug(" 70h >>>>>>>>>>>> request after adding addPrescriptionRequest.::"+request);
			}
	}
	
	private boolean validateInstitution(Institution institution) {
		boolean result = ((institution != null)
				&& institution.getIsTreatment().equalsIgnoreCase("Y")
				&& institution.getIsActive().equalsIgnoreCase("Y"));
		if (LOG.isDebugEnabled()) {
					LOG.debug(" 100c >>>>>>>>>>>> testInstitution result:: "+result);
		}
		return result;
	}
	
	private void addExhaustedTriesMessage(ResponseUtil response, Patient patient,
			Date createdDate, Institution institution, int function)
	{
		//non-persistent request for uniformity
		Request nonPersistedRequest = new Request();
		nonPersistedRequest.setUserId(patient.getUserProfile().getId());
		nonPersistedRequest.setCreatedDate(UtilityMethods.dateToTimestamp(createdDate));
		nonPersistedRequest.setInstitution(institution);
		nonPersistedRequest.setRequestFunction(function);
		nonPersistedRequest.setExceededMaxFailedAttempts(Boolean.TRUE);
        // addProperty so new messages are allowed to the HashSet
		// of InformationalMessages
		/*response.getMessages().addMessage(MessageUtils.createInformationalMessage(RxRequestMessages.REQUEST_STATUS,
				new String[] {nonPersistedRequest.getInstitution().getName()},
				new Object[] {nonPersistedRequest}));*/
		//TODO: prescriptionservice.request - Prasad
		response.getInfoMessages().put(RxRequestMessages.REQUEST_STATUS, RxRequestMessages.createRequestStatusMessage(nonPersistedRequest));
		response.setPojoObject(nonPersistedRequest);
	}


}
