package gov.va.med.domain.service.messaging.transceiver;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.Request;
import gov.va.med.domain.service.messaging.Response;
import gov.va.med.domain.service.messaging.decode.MessageDecoder;
import gov.va.med.domain.service.messaging.encode.MessageEncoder;
import gov.va.med.domain.service.messaging.environment.EndPoint;
import gov.va.med.domain.service.messaging.parse.HL7MessageMetadata;

import gov.va.med.domain.service.messaging.MessagingLogHelper;

/**
 * <pre> Wraps a low level Transceiver using a Decorator-like Pattern to add endocde
 * and decode behavior that is applied based on the encoder and decoder supplied.
 * <P>
 * This class also will log the outbound message and the inbound response
 * in the message log table if enabled in ConfigService parameter
 * "messaging:HL7_MESSAGE_LOGGING" which is read as a Boolean
 * <P>
 * This class can be used for synchrounous or asynchronous messaging.
 * The decoder used is the synchronous decoder NOT the asynchronous decoder.
 * <P>
 * For asynchronous,  the transport may be truly one way such as JMS where no
 * response is generated, or logically one way but truly synchronous such
 * as an HTTP request where a real response is generated but only treated as
 * an acknowlegement.
 * <P>
 * The decoder specified must be sensitive to what response is expected. <BR>
 * If a JMS transceiver returns null, the IdentityDecoder may be appropriate.<BR>
 * If an HTTP Transceiver is used and only expecting an HTTP 200,  a Decoder that
 * wraps this in an OK Response may be appropriate.
 * <P>
 * Requires an EndPoint, Transceiver, MessageEncoder and MessageDecoder to be set
 * prior to invoking Transcieve.<br>
 * <P>
 * If encoding is to be supressed, use the IdentityEncoder <br>
 * If decoding is to be supressed, use the IdentityDecoder <br>
 * If both are to be suppressed, consider not using this class as
 * its sole purpose is to decorate the transceive action with encoding and decoding.
 * </pre>
 */
@Component
public class TransceiverFacade {

	EndPoint endPoint;
	Transceiver transceiver;
	MessageEncoder encoder;
	MessageDecoder decoder;
	Request request;

	private static Logger logger = LogManager.getLogger(TransceiverFacade.class);

	private MessagingLogHelper messagingLogHelper;

	public TransceiverFacade() {
	}

	public TransceiverFacade(Transceiver aTransceiver,
	                         MessageEncoder anEncoder,
	                         MessageDecoder aDecoder,
	                         Request aRequest,
	                         MessagingLogHelper messagingLogHelper) {
		setTransceiver(aTransceiver);
		setDecoder(aDecoder);
		setEncoder(anEncoder);
		setRequest(aRequest);
		
		this.messagingLogHelper = messagingLogHelper;
	}

	/**
	 * Transceive with outbound encoding and inbound decoding.<br>
	 * Sends the payload held by the Request.
	 * Uses encoding parameters and protocol parameters from the EndPoint as needed.
	 * <P>
	 * Encoder is required:  If no Encoding is to be done, specify the IdentityEncoder
	 * Decoder is required: if no Decoding is to be done, specify IdentiyDecoder.
	 * Another option is to subclass to override selective steps in the transceive
	 * process to supress or alter the way the payload is handled.
	 * <P>
	 * This method is a implemented as a Template Method with callouts for each step
	 * to allow subclasses to override selective steps in the process.
	 * @return either a valid (isOk()) or invalid response (!isOk())
	 * 		   If invalid, it will contain either an exception (cause) or a status code and
	 * 		   description and possibly the payload if the problem occurred after
	 *         a successful transcieve.
	 * <P>
	 * @throws MessagingException
	 */
	public Response transceive() throws MessagingException  {
		//Step ZERO - Check pre-conditions.
	    IPayload decodedPayload = null;
		validate();
		//Step ONE - encode an IPayload into an encoded Object
		Object encodedRequestPayload
			= prepareOutboundPayload(getRequest().getPayload());

		String userId = extractUserId(request);

		//NOTE:  If sent via Deliver Services, the MessageControlId is
		// overlaid by the Delievery Service's Envelope. So when using this
		// log entry for troubleshooting,  be aware its id is not acctual
		logger.debug("About to log outbound message for " + request.getFunctionName() +" --- userId::"+userId);
		String functionName = request.getFunctionName();
		String msg = encodedRequestPayload.toString();

		messagingLogHelper.logOutboundMessage(userId, functionName, msg);

		//Step TWO - Transcieve
		logger.debug("About to invoke transceiver for" + request.getFunctionName());
		Object rawResponse = invokeTransceiver(encodedRequestPayload);

		logger.debug("About to log inbound message " + request.getFunctionName());
		messagingLogHelper.logInboundMessage(userId, functionName, String.valueOf(rawResponse));

		//Step THREE - decode from raw response into IPayload
		decodedPayload = prepareInboundPayload(rawResponse);

		postProcessInboundPayload(decodedPayload);

		//Step FOUR - Convert the IPayload into a Response
		return prepareResponse(decodedPayload);
	}

	protected void validate() throws MessagingException {
		if(getEncoder() == null) {
			throw new MessagingException("Missing Encoder in TransceiverFacade for function [" +
			                             getRequest().getFunctionName() + "]");
		}
		if(getDecoder() == null) {
			throw new MessagingException("Missing Decoder in TransceiverFacade for function [" +
			                             getRequest().getFunctionName() + "]");
		}
		if(getTransceiver() == null) {
			throw new MessagingException("Missing Transciever in TransceiverFacade for function [" +
			                             getRequest().getFunctionName() + "]");
		}
	}
	/**
	 * Callout to prepare the payload for transcieve.
	 * By default, invokes the encoder to prepare the outbound request
	 * @param payload
	 * @return a payload ready to send via the transceiver
	 * @throws MessagingException
	 */
	protected Object prepareOutboundPayload(IPayload payload) throws MessagingException {
		return getEncoder().encode(payload, getEndPoint());
	}
	/**
	 * Callout to invoke transciever that can be overridden if some alternate behavior
	 * is needed such as pre and post processing or special transceiver set up.
	 * @param encodedRequestPayload
	 * @return the raw payload from the transceive action - may be null.
	 * @throws MessagingException
	 */
	protected Object invokeTransceiver(Object encodedRequestPayload) throws MessagingException {

		return getTransceiver().transceive(encodedRequestPayload);
	}
	/**
	 * Callout to prepare the response payload for the reicpeint.
	 * By default, invokes the decoder to prepare the inbound response
	 * @param payload raw response payload
	 * @return a payload ready to hand back to the business tier.
	 * @throws MessagingException
	 */
	protected IPayload prepareInboundPayload(Object rawResponse) throws MessagingException {
		return getDecoder().decode(rawResponse);
	}
	/**
	 * Callout to do any necessary massaging of the response payload that was not the purvue
	 * of the decoder.   Default is to do nothing.
	 * @param decodedPayload
	 */
	protected void postProcessInboundPayload(IPayload decodedPayload) {

		//TODO populate the payload with any missing data (i.e. keys)
	}
	/*
	 * Callout to create an appropriate response for the Payload for normal
	 * processing.
	 */
	protected Response prepareResponse(IPayload decodedPayload) {

		return new Response(getRequest().getMessageType(),
		                    decodedPayload,
		                    true);
	}
	protected void setTransceiver(Transceiver aTransceiver) {
		transceiver = aTransceiver;
		setEndPoint(transceiver.getEndPoint());
	}

    /**
     * Digs out the user id from request in a fault-tolerant way in case
     * the request does not have a user id.
     */
    protected String extractUserId(Request request) {
		if (request.isApplicationAckRequest())
		{
		    return ((HL7MessageMetadata)request.getPayload()).getIcn();
		}

        try {
            return String.valueOf(request.getUserId());
        }
        catch (Throwable e) {
            return null;
        }
    }

	protected MessageEncoder getEncoder() 	{return encoder;}
	protected MessageDecoder getDecoder() 	{return decoder;}
	protected Transceiver getTransceiver() 	{return transceiver;}
	protected EndPoint getEndPoint() 		{return endPoint;}
	protected Request getRequest() 			{return request;}
	protected void setEncoder(MessageEncoder anEncoder) {encoder = anEncoder;}
	protected void setDecoder(MessageDecoder aDecoder) 	{decoder = aDecoder;}
	protected void setEndPoint(EndPoint anEndPoint) 	{endPoint = anEndPoint;}
	protected void setRequest(Request aRequest) 		{request = aRequest;}
}

