package gov.va.med.domain.service.messaging.transceiver;

//import gov.va.med.domain.persistence.jdbc.dao.MessagingLoggingDao;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.environment.EndPoint;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * Standard superclass for transceivers.  Provides some convenience methods
 * especially to dig out endpoint protocol parameters.  Transceivers are not
 * required to subclass Transciever however,  they only need to implement the 
 * ITransceiver interface.  
 *
 * @author Slava Uchitel
 * @version $Id: Transceiver.java,v 1.11 2005/08/05 10:59:39 slava.uchitel Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public abstract class Transceiver implements ITransceiver {
	//private MessagingLoggingDao messagingLoggingDao = new MessagingLoggingDao();
	private EndPoint endPoint;
	
	private static Logger logger = LogManager.getLogger(Transceiver.class);
	
	protected Logger getLogger()
	{
	    return logger;
	}

	public Transceiver() {
	}

	public Transceiver(EndPoint anEndPoint) {
		setEndPoint(anEndPoint);
	}
	/**
	 * Subclasses must implement this to send message and return a response.
	 * @payload the object to send as appropriate for the protocol of the
	 *          transceiver.  The subclass is permitted to msassage the payload
	 *          as needed but this is normally done in the encoder.  So
	 * 			massaging generally should be at a low level that is unaware of 
	 *          specific message content.  
	 * @return  The object received as a response from the send.  This may be null.
	 *          Also, the transceiver is permitted to massage the value as needed
	 *          prior to returning it, but this is normally done in the decoder.  So
	 * 			massaging generally should be at a low level that is unaware of 
	 *          specific message content.
	 */
	public abstract Object transceive(Object payload) throws MessagingException;

	/**
	 * Returns the named parameter from the EndPoint's protocol parameters
	 *
	 * @param key key the parameter name requested
	 * @return the found value or null if not found.
	 */
	protected String getParameter(String key) {
		return getParameter(key, null);
	}

	/**
	 * Mimics Properties getProperty(key, defaultValue) by allowing
	 * the designation of a default value if the key is not found.
	 *
	 * @param key the parameter name requested
	 * @param defaultValue the value to return if the key is not in endpoint's paramaters
	 * @return either the value found or the default.
	 */
	protected String getParameter(String key, String defaultValue) {
		String result = getEndPoint().getProtocolParameter(key);
		if(result == null) {
			return defaultValue;
		}
		return result;
	}
	/**
	 * Type safe getter for int value paremeters
	 * @param key of the protocol parameter
	 * @return
	 * @throws NumberFormatException 
	 */
	protected int getIntParameter(String key) throws NumberFormatException {
	    return Integer.parseInt(getParameter(key));
	}
	
	/**
	 * Type safe getter for int value paremeters with default capability
	 * for missing parameters.  
	 * @param key of the protocol parameter
	 * @param defaultValue the substitute (String) value to use if key not found
	 * @return
	 * @throws NumberFormatException 
	 */
	protected int getIntParameter(String key, String defaultValue) {
	    return Integer.parseInt(getParameter(key, defaultValue));
	}

	public EndPoint getEndPoint() { return endPoint;}
	public void setEndPoint(EndPoint anEndPoint) {endPoint = anEndPoint;}

}
