package gov.va.med.domain.service.messaging.transceiver;

import gov.va.med.domain.service.messaging.MessagingConstants;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.environment.EndPoint;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import ca.uhn.hl7v2.llp.LLPException;
/**
 * SocketTransceiver
 *
 * @author Slava Uchitel
 * @version $Id: SocketTransceiver.java,v 1.8 2005/09/07 12:16:18 slava.uchitel Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public abstract class SocketTransceiver extends Transceiver
{
    private static Logger logger = LogManager.getLogger(SocketTransceiver.class);
    private long endTime;
    private long startTime;
    private Socket socket;

    {
        if(logger.isDebugEnabled())
            logger.debug("SocketTransceiver Loading Latest Version");
    }

    public SocketTransceiver() {
        super();
    }

    public SocketTransceiver(EndPoint anEndPoint) {
        super(anEndPoint);
    }

    protected long getEndTime()
    {
        return endTime;
    }

    protected void setEndTime(long value)
    {
      endTime = value;
    }

    protected long getStartTime()
    {
        return startTime;
    }

    protected void setStartTime(long value)
    {
      startTime = value;
    }

    protected void send(String data, OutputStream outStream)
    throws LLPException
    {
        try
        {
            outStream.write(data.getBytes());
            outStream.flush();
        }
        catch(SocketTimeoutException ste){
        	logger.error("SocketTimeoutException: Send operation Socket timed out." + ste);
        	throw new LLPException("SocketTimeoutException:Send operation timed out.", ste);
        }
        catch(SocketException se){
        	logger.error("SocketException: Socket appears to have been closed during sending operation." + se);
        	throw new LLPException("SocketException:Send operation timed out.", se);
        }
        catch(IOException ioe){
            logger.error("Caught IOException while sending <<< " + ioe + "<<< ");
        	throw new LLPException("Send operation failed. <<< ", ioe);
        }
        if(System.currentTimeMillis() > getEndTime())
        {
            logger.warn("Send operation time is greater than getEndTime()");
        	//throw new LLPException("Send operation timed out.");
        }        
        
    }

    protected String receive(InputStream inStream, String endMarker, int endMarkerMinPosition)
    throws LLPException
    {
        StringBuffer buffer = new StringBuffer();
        InputStreamReader reader = new InputStreamReader(inStream);
        int i = 0 ;
        char[] arr = new char[512];
        while (i != -1 && buffer.indexOf(endMarker) < endMarkerMinPosition)
        {
            try
            {
                i = reader.read(arr, 0, arr.length);
                if(i > 0)
                {
                    buffer.append(arr, 0, i);
                }
            }catch(SocketTimeoutException ste){
            	logger.error("SocketTimeoutException: Receive Socket timed out, the following received so far: >>>\n\r" + buffer.toString() + "<<<");
            	throw new LLPException("SocketTimeoutException:Receive operation timed out.", ste);
            }
            catch(SocketException se){
            	logger.error("SocketException on read() attempt. Socket appears to have been closed: >>>\n\r" + buffer.toString() + "<<< " + se.getMessage() + "<<< ");
            	throw new LLPException("SocketException:Receive operation timed out.", se);
            }
            catch(IOException ioe){
                logger.error("Caught IOException while receiving, the following received so far: >>>\n\r" + buffer.toString() + "<<< " + ioe + "<<< ");
            	throw new LLPException("Receive operation failed. <<< ", ioe);
            }
            if(System.currentTimeMillis() > getEndTime())
            {
                logger.warn("Receive operation timed out is greater than getEndTime().");
            	//throw new LLPException("Receive operation timed out.");
            }

        }
        return buffer.toString();
    }

    protected void connect() throws MessagingException
    {
        if(logger.isDebugEnabled()){
        	logger.debug(">>>>>>>>> Inside SocketTransceiver Connect method.");
        }
		
        disconnect();
        String systemType = getEndPoint().getEncodingParameter(MessagingConstants.SYSTEM_TYPE);
        //NOTE:
        //Production systems use "HL7." prefix plus DN to connect.
        //Test systems use IP (because their DN is often unresolvable).
        String socketAddr = null;
        if (systemType.equals("P"))
        {
            socketAddr = "HL7."+getParameter(MessagingConstants.FULLY_QUALIFIED_HOSTNAME);
        }
        else
        {
            socketAddr = getParameter(MessagingConstants.SOCKET_ADDRESS);
        }
        int socketPort = getIntParameter(MessagingConstants.SOCKET_PORT);
        int timeOut = getIntParameter(MessagingConstants.CALL_TIMEOUT_MSEC);
        setStartTime(System.currentTimeMillis());
        setEndTime(getStartTime() + timeOut);

        if(logger.isDebugEnabled()){
        	logger.debug(">>>>>>>>> Inside SocketTransceiver Connect method. Values:- socketAddr:"+socketAddr+" - socketPort:"+socketPort+" - timeOut:"+timeOut);
        }

        try
        {
            socket = new Socket(socketAddr, socketPort);
            socket.setSoTimeout(timeOut);
        } catch (UnknownHostException e) {
        	String msg = "Unable to contact remote host at address " + socketAddr + " and port " + socketPort;
        	getLogger().error(msg, e);
        	throw new MessagingException(msg, e);
		} catch (IOException e) {
        	String msg = "Unable to contact remote host at address " + socketAddr + " and port " + socketPort;
        	getLogger().error(msg, e);
        	throw new MessagingException(msg, e);
        }
    }

    protected void disconnect()
    {
        if (socket == null)
            return;
        try
        {
            if (socket.isConnected())
                socket.close();
        }
        catch (IOException e)
        {
            ;//do nothing
        }
    }

    protected InputStream getInputStream() throws MessagingException
    {
        if (socket==null || socket.isClosed())
            throw new MessagingException("Trancseiver is not connected.");
        try
        {
            return socket.getInputStream();
        }
        catch (IOException e)
        {
            throw new MessagingException("Failed to obtain InputStream from Socket.", e);
        }
    }

    protected OutputStream getOutputStream() throws MessagingException
    {
        if (socket==null || socket.isClosed())
            throw new MessagingException("Trancseiver is not connected.");
        try
        {
            return socket.getOutputStream();
        }
        catch (IOException e)
        {
            throw new MessagingException("Failed to obtain OutputStream from Socket.", e);
        }
    }

    public abstract Object transceive(Object payload) throws MessagingException;
}
