package gov.va.med.domain.service.messaging.transceiver;

import gov.va.med.domain.service.messaging.MessagingException;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.stereotype.Component;

import ca.uhn.hl7v2.llp.LLPException;

/**
 * HL7DCLLPTransceiver
 * 
 * @author Slava Uchitel
 * @version $Id: HL7DCLLPTransceiver.java,v 1.6 2005/05/11 20:27:18
 *          odysseas.pentakalos Exp $
 * @since MHV 2.0 <br>
 *        Mar 1, 2005
 */
@Component("hl7DCLLPTransceiver")
public class HL7DCLLPTransceiver extends SocketTransceiver {
    private static Logger log = LogManager.getLogger(HL7DCLLPTransceiver.class);

    private static final String STR_CRLF = "\r\n";

    private static final String STR_CR = "\r";

    private static final char CR = '\r';

    private static final int PREFIX_LENGTH = 3;

    private static final String DATA_TERMINATOR = "\u001B\u001B\u001B";

    private static final String DATA_BEGIN_MARKER = "220 OK\r\nDATA PARAM=MPI\r\n";

    public Object transceive(Object payload) throws MessagingException {
        String result = "";
        connect();
        InputStream is=null;
        OutputStream os=null;		
        try {
            is=getInputStream();
            os=getOutputStream();		

        	result = sendMessage((String) payload, is, os);
        	return result;
        } catch (LLPException e) {
        	log.error("Unable to send message over LLP protocol. Error: ", e);
        	throw new MessagingException("Unable to send message over LLP protocol. Error: ", e);
        } finally {
        	if(is != null){
        		try {
					is.close();
				} catch (IOException e) {
					log.error(e);
				}
        	}
        	if(os != null){
        		try {
					os.close();
				} catch (IOException e) {
					log.error(e);
				}
        	}
        	disconnect();
        }
    }
    
    protected String sendMessage(String msg, InputStream inStream, OutputStream outStream)
    		throws LLPException, MessagingException {
    	String responseString = null;
		String data = null;
		sendCommand("HELO" + STR_CRLF, outStream);
		responseString = receiveCommandResponse(inStream);
		//System.out.print("<<<" + ss);
		
		if (responseString.startsWith("2")) {
		    sendCommand("DATA PARAM=MPI" + STR_CRLF, outStream);
		    sendData(msg, outStream); 
		    responseString = receiveCommandResponse(inStream); 
		    
		    //MPI starts running query right after receiving DATAPARAM block.
		    //It reads TURN command after performing query when all results are gathered.
		    sendCommand("TURN" + STR_CRLF, outStream);
		      
		    //ss = receiveCommandResponse(ins);
		    data = receiveData(inStream);
		    sendCommand("200 OK" + STR_CRLF, outStream); 
		    sendCommand("QUIT" + STR_CRLF, outStream);
		} 
		else {
		    sendCommand("QUIT" + STR_CRLF, outStream);
		    throw new LLPException("Error code " + responseString + " received from MPI.");
		}
		//System.out.println(">>>QUIT");
		log.debug("Spent " + (System.currentTimeMillis() - getStartTime()) + " milliseconds in call.");
		return data;
	}
    
    public String formatIncomingMessage(String message) throws LLPException {
        int count = 0;
        int marker = 0;
        StringBuffer result = new StringBuffer();
        if (!message.startsWith(DATA_BEGIN_MARKER)) {
            throw new LLPException("MPI Data Begin marker is not found");
        }
        message = message.substring(DATA_BEGIN_MARKER.length());
        while (marker < message.length()) {
            count = Integer.parseInt(message.substring(marker, marker
                    + PREFIX_LENGTH));
            if (count > 0) {
                String subString = message.substring(marker + PREFIX_LENGTH, marker
                        + PREFIX_LENGTH + count);
                if (!subString.equals(DATA_TERMINATOR)) {
                    result.append(subString);
                }
            } 
            else {
                if (result.charAt(result.length() - 1) != CR) {
                    result.append(CR);
                }
            }
            marker = marker + PREFIX_LENGTH + count;
        }
        return result.toString();
    }

    private String formatOutgoingMessage(String msg) throws LLPException {
        String[] results = new String[4];
        String[] parts = msg.split("\r");
        if (parts.length != 3)
            throw new LLPException("Query has to have exactly 3 segments.");
        for (int i = 0; i < parts.length; i++)
            results[i] = parts[i];
        results[3] = DATA_TERMINATOR;
        StringBuffer message = new StringBuffer();
        for (int i = 0; i < results.length; i++) {
            String prefix = String.valueOf(results[i].length());
            while (prefix.length() < PREFIX_LENGTH) {
                prefix = "0" + prefix;
            }
            message.append(prefix).append(results[i]);
        }
        return message.toString();
    }

    protected void sendCommand(String command, OutputStream outStream)
            throws LLPException {
        log.debug("Sending: " + command);
        send(command, outStream);
    }

    protected void sendData(String data, OutputStream outStream)
            throws LLPException {
        String message = formatOutgoingMessage(data);
        log.debug("Sending: " + message);
        send(message, outStream);
    }

    protected String receiveCommandResponse(InputStream inStream)
            throws LLPException {
        String message = receive(inStream, STR_CRLF, 0);
        log.debug("Received: " + message);
        return message;
    }

    protected String receiveData(InputStream inStream) 
    		throws LLPException {
        String message = this.receive(inStream, DATA_TERMINATOR, 0);
        log.debug("Received: " + message);
        return (formatIncomingMessage(message));
    }

 }