package gov.va.med.domain.service.messaging.transceiver;

import gov.va.med.deliveryservice.delegate.DeliveryServiceDelegateFactory;
import gov.va.med.deliveryservice.delegate.IDeliveryService;
import gov.va.med.deliveryservice.delegate.envelopes.HL7Envelope;
import gov.va.med.deliveryservice.delegate.exceptions.HVMException;
import gov.va.med.deliveryservice.delegate.jms.JMSDeliveryServiceDelegate;
import gov.va.med.domain.service.config.MhvConfigService;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.environment.EndPoint;

import java.util.Properties;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.stereotype.Component;

/**
 * DeliveryServicesTransceiver - uses DeliveryServicesDelegate to transmit HL7 messages DSD does not
 * support Commit ACK so nothing comes back from transceiver. IdentityDecoder should be used for
 * decoding resulting object.
 * 
 * @author Slava Uchitel Created on Apr 19, 2005
 * @version $Id: DeliveryServicesTransceiver.java,v 1.19 2005/09/13 21:07:39 slava.uchitel Exp $
 * @since MHV 2.0 <br>
 *        creation-date
 */
@Component
public class DeliveryServicesTransceiver extends Transceiver {
	private static Logger logger = LogManager.getLogger(DeliveryServicesTransceiver.class);

	public DeliveryServicesTransceiver() {
	}

	public DeliveryServicesTransceiver(EndPoint anEndPoint) {
		super(anEndPoint);
	}

	private static IDeliveryService dsd = null;
	private static final String JMSDELIVERYSERVICEDELEGATE = "gov.va.med.deliveryservice.dsd.delegates.jms.JMSDeliveryServiceDelegate";

	protected IDeliveryService getDsd() throws HVMException {
		if (dsd == null) {

			Properties props = null;


			props = new Properties();

			props.setProperty("java.naming.provider.url", MhvConfigService.getInstance().getParamValueString("ds:JNDI_PROVIDER_URL"));
			props.setProperty("java.naming.security.principal", MhvConfigService.getInstance().getParamValueString("ds:JAVA_NAMING_SECURITY_PRINCIPAL"));
			props.setProperty("java.naming.security.credentials", MhvConfigService.getInstance().getParamValueString("ds:JAVA_NAMING_SECURITY_CREDENTIALS"));

			props.setProperty("java.naming.factory.initial", MhvConfigService.getInstance().getParamValueString("ds:JNDI_INITIAL_FACTORY"));
			props.setProperty("gov.va.med.dsd.jms.dsinbound", MhvConfigService.getInstance().getParamValueString("ds:INBOUND_QUEUE"));
			props.setProperty("gov.va.med.dsd.jms.queuefactory", MhvConfigService.getInstance().getParamValueString("ds:QUEUE_FACTORY"));

			/*
      	 props.setProperty("java.naming.provider.url", MhvConfigService.getInstance().getParamValueString("ds:JNDI_PROVIDER_URL"));
         props.setProperty("java.naming.security.principal", MhvConfigService.getInstance().getParamValueString("ds:JAVA_NAMING_SECURITY_PRINCIPAL"));
         props.setProperty("java.naming.security.credentials", MhvConfigService.getInstance().getParamValueString("ds:JAVA_NAMING_SECURITY_CREDENTIALS"));

         props.setProperty("java.naming.factory.initial", MhvConfigService.getInstance().getParamValueString("ds:JNDI_INITIAL_FACTORY"));
         props.setProperty("gov.va.med.dsd.jms.dsinbound", MhvConfigService.getInstance().getParamValueString("ds:INBOUND_QUEUE"));
         props.setProperty("gov.va.med.dsd.jms.queuefactory", MhvConfigService.getInstance().getParamValueString("ds:QUEUE_FACTORY"));
			 */

			/*
         props.setProperty("java.naming.provider.url", "t3://DNS   viea.vha.med.DNS   :7001");
         props.setProperty("java.naming.security.principal", "weblogic");
         props.setProperty("java.naming.security.credentials", "wl_admin");

         props.setProperty("java.naming.provider.url", "t3://DNS   sch1.vha.med.DNS   :8531");
         props.setProperty("java.naming.security.principal", "weblogic");
         props.setProperty("java.naming.security.credentials", "easydoesIT");
			 */

			try {
				logger.debug("Creating delivery services delegate with the following properties: \n" + props);
				dsd = DeliveryServiceDelegateFactory.getDeliveryService(props);

				if (dsd != null)
					logger.debug("Created delivery services delegate with the following properties: \n" + ((JMSDeliveryServiceDelegate)dsd).getTestProperties());
				else
					logger.debug("Failed to create delivery services delegate.");

			}
			finally {
				logger.info("Properties used to create the delivery services delegate were: \n" + props);
			}
		}
		return dsd;
	}

	/**
	 * This method parses the given ER7, copies known MSH fields to the envelope's properties and
	 * sets the message as the envelope's content.
	 * 
	 * @param er7Message
	 * @return new hl7 envelope
	 * @throws HVMException
	 */
	protected HL7Envelope prepareEnvelope(String er7Message) throws HVMException {
		HL7Envelope envelope = HL7Envelope.createFromER7Message(er7Message);
		return envelope;
	}

	/**
	 * ER7 encoded HL7 message should be used as a payload. <p/> (non-Javadoc)
	 * 
	 * @see gov.va.med.domain.messaging.transceiver.Transceiver#transceive(java.lang.Object)
	 */
	public Object transceive(Object payload) throws MessagingException {

		String er7Message = (String) payload;
		HL7Envelope envelope = null;
		try {
			envelope = prepareEnvelope(er7Message);
			if (getDsd() != null) {
				getDsd().sendMessagePTP(envelope);
			}
			else {
				throw new MessagingException("Error connecting to Delivery Services " + er7Message);
			}
			return envelope.getMessageID();
		}
		catch (Throwable e) {
			getLogger().error(e);
			throw new MessagingException("Error transmitting message with Delivery Services " + er7Message, e);
		}
	}

}
