package gov.va.med.domain.service.messaging.transceiver;

import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.framework.exception.ErrorCodeCatalog;

import java.io.IOException;
import java.security.Security;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.stereotype.Component;

import com.fortify.annotations.FortifyNotPassword;

/**
 * <pre>
 * Concrete Superclass for Transceivers that send HTTP or HTTPS requests.
 * It uses parameters obtained from the EndPoint
 * to prepare and send the GET or POST.
 * <p/>
 * It should normally be created using the Tramsceiver factory.
 * <p/>
 * This class is not thread-safe because it saves the EndPoint.
 *  So an instance is required for each different EndPoint
 * but it may be cached and reused for the same EndPoint.
 * <p/>
 * In most cases this class can be used without extension.
 * <p/>
 * Suports proxy, proxy authentication and SSL
 * <p/>
 *  Example parameters for no proxy and no SSL (https) and timeout of 30 seconds:
 * <p/>
 *  SEND_METHOD=http
 *  HTTP_URL=https://www.verisign.com/
 *  HTTP_TIMEOUT_MILLIS=30000
 * <p/>
 * Example properties for proxy and SSL (https):
 * <p/>
 *  SEND_METHOD=http
 *  HTTP_URL=https://www.verisign.com/
 *  HTTP_TIMEOUT_MILLIS=30000
 * <p/>
 *  #Required to define the Proxy server
 *  HTTP_USING_PROXY=true
 *  HTTP_PROXY_HOST=internetabh.eds.com
 *  HTTP_PROXY_PORT    
 * <p/>
 *  #Needed to set appropriate Secure Server handlers and such for HTTPS
 *  HTTP_USING_SSL=true
 * <p/>
 *  #Needed if the proxy server requires ID and password
 *  HTTP_USING_PROXY_AUTHENTICATION=true
 *  HTTP_PROXY_USER_ID=myProxyUserID
 *  HTTP_PROXY_REALM=Cisco Content Engine
 *  HTTP_PROXY_PASSWORD=myProxyPassword
 *  </pre>
 *
 * @author Joel Goldberg, EDS, VHA
 * @version $Id: BaseHTTPTransceiver.java,v 1.5 2005/06/07 12:36:47 joel.goldberg Exp $
 */
@Component
public class BaseHTTPTransceiver extends Transceiver {
 
    private static Logger logger = LogManager.getLogger(BaseHTTPTransceiver.class);
    
	// Keys for properties used to send an HTTP or HTTPS request
	public static final String HTTP_PROXY_HOST_KEY = "HTTP_PROXY_HOST";
	public static final String HTTP_URL_KEY = "HTTP_URL";
	public static final String HTTP_PROXY_PORT_KEY = "HTTP_PROXY_PORT";
	public static final String HTTP_PROXY_USER_ID_KEY = "HTTP_PROXY_USER_ID";
	@FortifyNotPassword
	public static final String HTTP_PROXY_PASSWORD_KEY = "HTTP_PROXY_PASSWORD";
	public static final String HTTP_PROXY_REALM_KEY = "HTTP_PROXY_REALM";
	public static final String HTTP_TIMEOUT_MILLIS_KEY = "HTTP_TIMEOUT_MILLIS";
	public static final String USING_HTTP_PROXY_KEY = "HTTP_USING_PROXY";
	public static final String USING_SSL_KEY = "HTTP_USING_SSL";
	public static final String HTTP_GET_OR_POST_KEY = "HTTP_GET_OR_POST";
	public static final String HTTP_CONTENT_TYPE_KEY = "HTTP_CONTENT_TYPE";
	public static final String POST = "POST";
	public static final String USING_PROXY_AUTHENTICATION_KEY = "HTTP_USING_PROXY_AUTHENTICATION";

	// values used to set up HttpClient for send request
	public static final String JSSE_PROVIDER_CLASS = "com.sun.net.ssl.internal.ssl.Provider";
	public static final String HANDLER_PACKAGE_KEY = "java.protocol.handler.pkgs";
	public static final String HTTPS_HANDLER_PACKAGE = "com.sun.net.ssl.internal.www.protocol";
	public static final String HANDLER_DELIMITER = "|";

	public BaseHTTPTransceiver() {
	}

	/**
	 * Superclass/default behaviour does not do any special processing of
	 * send String or return String.  Subclasses that need to format one or both of these
	 * should override this method or use the TransformingTransceiver decorator to wrap
	 * the instance of BaseHTTPTransceiver.
	 *
	 * @return The raw String representation of the HTTP Response.
	 *         sub classes may post-process the response to return a first class object.
	 */
	public Object transceive(Object payload) throws MessagingException {
		long startMillis = System.currentTimeMillis();

		String result = sendMessage(payload.toString());

		 
		logger.info("Elapsed millis = [" +
		           (System.currentTimeMillis() - startMillis) + "] invoking " + getURL());

		return result;

	}

	/**
	 * Send the message using HttpClient
	 */
	protected String sendMessage(String stringToSend)
	    throws MessagingException {

		HttpMethodBase method = null;
		try {
			HttpClient httpClient = createHTTPClient();
			ensureSSL();
			setProxy(httpClient);
			httpClient.setConnectionTimeout(getTimeoutMillis());
			method = createMethod(stringToSend);
			setContentType(method);
			executeMethod(httpClient, method);
			return obtainResultFromMethod(method);
		}
		catch(IOException e) {
			throw new MessagingException(ErrorCodeCatalog.Communication.TRANSCEIVER_SEND_ERROR, e);
		}
		finally {
			if(method != null){
			  method.releaseConnection();
			}
		}
	}

	/**
	 * Template method in case subclasses prefer to override.
	 */
	protected HttpClient createHTTPClient() {
		return new HttpClient();
	}

	/**
	 * Template method in case subclasses prefer to override.
	 */
	protected HttpMethodBase createMethod(String stringToSend) {

		if(isPOST()) {
			return createPostMethod(stringToSend);
		}

		return createGetMethod();

	}

	/**
	 * Template method for subclasses to override if needed.
	 */
	protected PostMethod createPostMethod(String stringToSend) {
		PostMethod method = new PostMethod(getURL());
		method.setRequestBody(stringToSend);
		return method;
	}

	/**
	 * Template method for subclasses to override if needed.
	 */
	protected GetMethod createGetMethod() {
		return new GetMethod(getURL());
	}

	/**
	 * Template method for subclasses to override if needed.
	 */
	protected void executeMethod(HttpClient httpClient, HttpMethodBase method)
	    throws IOException {
		httpClient.executeMethod(method);
	}

	/**
	 * Returns the meaningful portion of the response i.e. the body.
	 * May be overridden by subclasses if a different result is neeeded.
	 * @throws IOException 
	 */
	protected String obtainResultFromMethod(HttpMethodBase method) throws IOException {
		return method.getResponseBodyAsString();
	}

	/**
	 * Checks first to see if Security provider is set.  If not, set SSL
	 * provider and security protocol handler in System.properties.
	 * The setting manages concatenation to existing handlers
	 */
	protected void ensureSSL() throws MessagingException {
		if(!isUsingSecureSocketLayer()) { return;}

		try {
			if(Security.getProvider(com.sun.net.ssl.internal.ssl.Provider.class.getName()) != null) {
				return;	// SSL already established
			}

			Security.addProvider(new com.sun.net.ssl.internal.ssl.Provider());

			// append our value to the property if not already there.

			if(System.getProperty("java.protocol.handler.pkgs", "").indexOf(
			    "com.sun.net.ssl.internal.www.protocol" + HANDLER_DELIMITER) < 0) {

				System.setProperty("java.protocol.handler.pkgs",
				                   System.getProperty("java.protocol.handler.pkgs") +
				                   HANDLER_DELIMITER + "com.sun.net.ssl.internal.www.protocol" +
				                   HANDLER_DELIMITER);
			}
		}
		catch(Exception e) {
			throw new MessagingException(ErrorCodeCatalog.Communication.TRANSCEIVER_SSL_ERROR, e);
		}
	}

	protected void setProxy(HttpClient httpClient) {
		if(!isUsingProxy()) {return;}

		httpClient.getHostConfiguration().setProxy(getProxyHost(), getProxyPort());

		if(!isUsingProxyAuthentication()) {return;}

		httpClient.getState().setProxyCredentials(getProxyRealm(), getProxyHost(),
		                                          new UsernamePasswordCredentials(getProxyUserID(), getProxyPassword()));

	}

	protected void setContentType(HttpMethodBase method) {
		String contentType = getContentType();
		if(contentType != null && contentType.length() > 0) {
			method.setRequestHeader("Content-Type", contentType);
		}
	}

	protected boolean isUsingSecureSocketLayer() {
		String value = getParameter(USING_SSL_KEY, "false");
		return Boolean.valueOf(value).booleanValue();
	}

	protected boolean isUsingProxy() {
		String value = getParameter(USING_HTTP_PROXY_KEY, "false");
		return Boolean.valueOf(value).booleanValue();
	}

	protected boolean isUsingProxyAuthentication() {
		String value = getParameter(USING_PROXY_AUTHENTICATION_KEY, "false");
		return Boolean.valueOf(value).booleanValue();
	}

	protected String getProxyHost() {
		return getParameter(HTTP_PROXY_HOST_KEY);

	}

	public String getURL() {
		return getParameter(HTTP_URL_KEY);
	}

	protected int getTimeoutMillis() {
		String port = getParameter(HTTP_TIMEOUT_MILLIS_KEY,
		                           "30000");  // default to 30 secoinds for now
		return Integer.parseInt(port);
	}

	protected int getProxyPort() {
		String port = getParameter(HTTP_PROXY_PORT_KEY);
		return Integer.parseInt(port);

	}

	protected String getProxyRealm() {
		return getParameter(HTTP_PROXY_REALM_KEY,
		                    "Cisco Content Engine");  // default for now

	}

	protected String getProxyUserID() {
		return getParameter(HTTP_PROXY_USER_ID_KEY);

	}

	protected String getProxyPassword() {
		return getParameter(HTTP_PROXY_PASSWORD_KEY);

	}

	protected String getContentType() {
		return getParameter(HTTP_CONTENT_TYPE_KEY, "text/html");
	}

	protected boolean isPOST() {
		String value = getParameter(HTTP_GET_OR_POST_KEY);
		return POST.equalsIgnoreCase(value);
	}
} 
