package gov.va.med.domain.service.messaging.parse;

import gov.va.med.domain.service.messaging.encode.hl7.util.HL7Helper;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7v24Helper;
import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.GenericComposite;
import ca.uhn.hl7v2.model.GenericPrimitive;
import ca.uhn.hl7v2.model.Segment;
import ca.uhn.hl7v2.model.Type;
import ca.uhn.hl7v2.model.Varies;
import ca.uhn.hl7v2.model.v24.datatype.CE;
import ca.uhn.hl7v2.model.v24.datatype.CX;
import ca.uhn.hl7v2.model.v24.datatype.XCN;
import ca.uhn.hl7v2.model.v24.segment.ERR;
import ca.uhn.hl7v2.model.v24.segment.MFE;
import ca.uhn.hl7v2.model.v24.segment.MFI;
import ca.uhn.hl7v2.model.v24.segment.MSA;
import ca.uhn.hl7v2.model.v24.segment.MSH;
import ca.uhn.hl7v2.model.v24.segment.PID;
import ca.uhn.hl7v2.model.v24.segment.QAK;
import ca.uhn.hl7v2.model.v24.segment.QPD;
import ca.uhn.hl7v2.model.v24.segment.QRD;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;

/**
 * HL7PreParser
 * Preparses HL7 messages for metatdata. Only works with V2.4 ER encoded messages and
 * MPI messages that are V2.3
 *
 * @author Slava Uchitel
 * @version $Id: HL7PreParser.java,v 1.25 2005/08/08 09:52:35 slava.uchitel Exp $
 * @since MHV 2.0 <br>03/16/2005
 */

public class HL7PreParser extends DefaultHL7ParseEventListener {
	private HL7MessageMetadata metaData = null;
	// TODO consider a quick lookup of message type and keying the 
	// segmentsOfInterest in a HashMap 
	// by this value.  Or using a hidden factory method to instantiate
	// a subclass of HL7PreParser that is designed for the message type.  
	private static String[] segmentsOfInterest = {"MSH", "MSA", "ERR", "QPD", "QRD", "QAK", "MFI", "PID", "MFE"};
	
	
	private void setIcn(String icn, HL7MessageMetadata metaData)
	{	
	    if (metaData.getIcn() == null)
	        metaData.setIcn(icn);
	}
	
	private void setDfn(String dfn, HL7MessageMetadata metaData)
	{	
	    if (metaData.getDfn() == null) 
	        metaData.setDfn(dfn);
	}
	
	private void setSsn(String ssn, HL7MessageMetadata metaData)
	{	
	    if (metaData.getSsn() == null)
	        metaData.setSsn(ssn);
	}
	
	/**
	 * return true if the segment is needed for building metadata
	 */
	public boolean onSegmentStart(String segName) {
		for(int i = 0; i < segmentsOfInterest.length; i++)
			if(segmentsOfInterest[i].equalsIgnoreCase(segName))
				return true;
		return false;
	}
	
	/**
	 * populate the metadata from the value in this segment 
	 */
	public void onSegmentEnd(String segName, Segment seg) throws HL7Exception {
		if(segName.equalsIgnoreCase("MSH")) { 
			populateFrom((MSH)seg);
		}
		else if(segName.equalsIgnoreCase("MSA")) { 
			populateFrom((MSA)seg);
		}
		else if(segName.equalsIgnoreCase("ERR")) { 
			populateFrom((ERR)seg);
		}
		else if(segName.equalsIgnoreCase("QPD")) {
			populateFrom((QPD)seg);
		}
		else if(segName.equalsIgnoreCase("QRD")) {
			populateFrom((QRD)seg);
		}
		else if(segName.equalsIgnoreCase("QAK")) { 
			populateFrom((QAK)seg);
		}
		else if(segName.equalsIgnoreCase("MFI")) { 
			populateFrom((MFI)seg);
		}
		else if(segName.equalsIgnoreCase("MFE")){
		    populateFrom((MFE)seg);
		}
		else if(segName.equalsIgnoreCase("PID")) {
			populateFrom((PID)seg);
		}
	}

	private void populateFrom(MFE mfe) throws HL7Exception {
	    if (getMetaData().getMasterFileId().equalsIgnoreCase("TFL"))
	    {    
	        Varies temp = mfe.getPrimaryKeyValueMFE(0);
	        Type data = temp.getData();
	        Type data1 = ((GenericComposite)data).getComponent(3);
	        String icn = ((GenericPrimitive)(((Varies)data1)).getData()).getValue();
	        setIcn(icn, getMetaData());
	    }
    }
	
	private void populateFrom(MFI mfi) {
        getMetaData().setMasterFileId(mfi.getMasterFileIdentifier().getIdentifier().getValue());
    }

    private void populateFrom(QAK qak) {
        getMetaData().setQueryAcknowledgementCode(qak.getQueryResponseStatus().getValue());
       
        String recordCount = qak.getThisPayload().getValue();
        getMetaData().setRecordsInThisMessage(toInteger(recordCount));
        
        recordCount = qak.getHitCountTotal().getValue();
        getMetaData().setRecordsTotalNumber(toInteger(recordCount));
    }

    private void populateFrom(QPD qpd) throws HL7Exception {
        getMetaData().setRequestId(HL7Helper.getStringField(qpd, 3, 0));
        getMetaData().setCategory(HL7Helper.getStringField(qpd, 4, 0));
        setIcn(HL7Helper.getStringField(qpd, 7, 0), getMetaData());
        setDfn(HL7Helper.getStringField(qpd, 8, 0), getMetaData());
        getMetaData().setExtractDate(HL7Helper.getStringField(qpd, 9, 0));
    }

    private void populateFrom(ERR err) throws HL7Exception {
        getMetaData().setAckError(err.getErrorCodeAndLocation(0).getCodeIdentifyingError().getIdentifier().getValue());
    }

    private void populateFrom(MSA msa) {
        getMetaData().setAckCode(msa.getAcknowledgementCode().getValue());
        getMetaData().setAckText(msa.getTextMessage().getValue());
        getMetaData().setCorrelationMessageControlId(msa.getMessageControlID().getValue());
    }

    private void populateFrom(MSH msh) {
        getMetaData().setMessageType(msh.getMessageType().getMessageType().getValue());
        getMetaData().setTriggerEvent(msh.getMessageType().getTriggerEvent().getValue());
        getMetaData().setMessageStructure(msh.getMessageType().getMessageStructure().getValue());
        getMetaData().setMessageContolId(msh.getMessageControlID().getValue());
        getMetaData().setApplicationAckInd(msh.getApplicationAcknowledgmentType().getValue());
        getMetaData().setCommitAckInd(msh.getAcceptAcknowledgmentType().getValue());
        getMetaData().setSendingSystemId(msh.getSendingFacility().getNamespaceID().getValue());
        getMetaData().setSendingSystemDomain(msh.getSendingFacility().getUniversalID().getValue());
        getMetaData().setSendingApplication(msh.getSendingApplication().getNamespaceID().getValue());
        getMetaData().setReceivingSystemId(msh.getReceivingFacility().getNamespaceID().getValue());
        getMetaData().setReceivingSystemDomain(msh.getReceivingFacility().getUniversalID().getValue());
        getMetaData().setReceivingApplication(msh.getReceivingApplication().getNamespaceID().getValue());
        getMetaData().setSystemType(msh.getProcessingID().getProcessingID().getValue());
    }
    
    private String getCategoryFromQrdWhatFilter(String where, String what)
    {
        String result = where + "-" + what;
        return result;
    }

    private void populateFrom(QRD qrd) {
        getMetaData().setExtractDate(qrd.getQueryDateTime().getValue());
        getMetaData().setRequestId(qrd.getQueryID().getValue());
        XCN[] idList = qrd.getWhoSubjectFilter();
        for (int i=0; i<idList.length;i++)
        {
            XCN idEntry = idList[i];
            String id = idEntry.getIDNumber().getValue();
            String idType = idEntry.getIdentifierTypeCode().getValue();
            if (idType.equals(HL7Helper.pidType[HL7Helper.ID_SSN]))
            {
                setSsn(id, getMetaData());
            }
            else if (idType.equals(HL7Helper.pidType[HL7Helper.ID_ICN]))
            {
                setIcn(id, getMetaData());
            }
            else if (idType.equals(HL7Helper.pidType[HL7Helper.ID_DFN]))
            {
                setDfn(id, getMetaData());
            }
        }
        CE whatEntry = qrd.getWhatSubjectFilter()[0];
        String what = whatEntry.getIdentifier().getValue();
        CE whereEntry = qrd.getWhatDepartmentDataCode()[0];
        String where = whereEntry.getIdentifier().getValue();
        getMetaData().setCategory(getCategoryFromQrdWhatFilter(where, what));
        
    }
    
    private void populateFrom(PID pid) {
        CX[] ides = pid.getPatientIdentifierList();
        for (int i=0; i<ides.length; i++)
        {
            CX ide =  ides[i];
            String id = ide.getID().getValue();
            String idType = ide.getIdentifierTypeCode().getValue();
            if (idType.equals(HL7v24Helper.pidType[HL7v24Helper.ID_SSN]))
            {	
                setSsn(id, getMetaData());
            }
            else if (idType.equals(HL7v24Helper.pidType[HL7v24Helper.ID_DFN]))
            {	
                setDfn(id, getMetaData());
            }
            else if (idType.equals(HL7v24Helper.pidType[HL7v24Helper.ID_ICN])) 
            {         
                 setIcn(id, getMetaData());
            }
        } 
    }
    
    public synchronized HL7MessageMetadata parse(String msg) 
    			throws HL7Exception, EncodingNotSupportedException {
		setMetaData(new HL7MessageMetadata());
		getMetaData().setMessageSize(msg.length());
		
		HL7FilteringParser parser = new HL7FilteringParser("2.4");
		parser.setEventListener(this);
		parser.setErrorListener(new DefaultHL7ParseErrorListener()); // no special error processing
		parser.setClassOverrideListener(
		        new IHL7MessageClassOverrideListener()
		        {

                    public Class onFindMessageClass(Class messageClass, String structure, String version)
                    { 
                        //NOTE: any class we override must be included in here.
                        if (messageClass.equals(ca.uhn.hl7v2.model.v24.message.VXR_V03.class))
                        {
                            return gov.va.med.domain.service.messaging.decode.hl7.structure.VXR_V03.class;
                        }
                        return null;
                    }
		            
		        }
		);
		parser.parse(msg);
		return getMetaData();
	}

    private HL7MessageMetadata getMetaData() {
        return metaData;
    }
    private void setMetaData(HL7MessageMetadata metaData) {
        this.metaData = metaData;
    }
    private Integer toInteger(String value) {
        if(value != null && value.trim().length() > 0) {
        	return new Integer(value);
        }
        return null;
    }

}
