package gov.va.med.domain.service.messaging.parse;

import gov.va.med.domain.service.messaging.encode.hl7.util.HL7Helper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ca.uhn.hl7v2.model.Message;
import ca.uhn.hl7v2.model.Structure;

/**
 * Listens for field or segment errors from HL7 parsing and decoding.   This is the 
 * Base class that generally work for all cases.  
 * <P>
 * To properly orgainze Errors, it must  set with the name of the root segment for the 
 * record in error.  The error will be added to a list of errors keyed by the named root segment 
 * along with related errors. 
 * <P>
 * If the rooSegementName is not provided, all errors will be grouped under the Message Structure
 * (i.e. the root of the enitre message).
 * @author Joel Goldberg
 * @version $Id: BaseProcessingParseErrorListener.java,v 1.5 2005/08/03 20:18:08 joel.goldberg Exp $
 * @since MHV 2.0 <br>Jun 15, 2005
 */
public class BaseProcessingParseErrorListener 
    		extends DefaultHL7ParseEventListener
    		implements IHL7ProcessingParseErrorListener
{
    private Map errorsByRootSegment; 
    private String rootName;
    
    public BaseProcessingParseErrorListener() {
        super(); 
    }
    public BaseProcessingParseErrorListener(String aRootName) {
        super(); 
        setRootName(aRootName);
    }
 
    public void onParseFieldException(HL7ParseFieldError error)  { 
        addKeyedError(error);
    }
    
    public void onParseSegmentException(HL7ParseSegmentError error) { 
        addKeyedError(error);
    }
    
    public void onDecoderDataError(HL7DecoderDataError error) {
        addKeyedError(error);
        
    }
    public Message getMessage(HL7ParseError error) {
        return error.getStructure().getMessage(); 
    }
    
    /**
     * Adds the error to the Error Map after determining the root segment
     * to use as the key. 
     * <P>
     * This method expects the message structure to be built sufficiently to navigate to the 
     * root segment to enable the root segment to be extracted.  
     */
    public void addKeyedError(HL7ParseError error) {
        addKeyedError(getRootStructure(error.getStructure()), error);
    }
    public Structure getRootStructure(Structure structure) {
        return HL7Helper.climbTreeToStructure(structure, getRootName());
    }
    /**
     * Adds the error under the key provided. 
     */
    public void addKeyedError(Structure key, HL7ParseError error)	{
    
        List errors = (List)getErrorsByRootSegment().get(key);
        if (errors == null) {
            errors = new ArrayList();
            getErrorsByRootSegment().put(key, errors);
        }
        errors.add(error);
    }

    public Map getErrorsByRootSegment() {
        if (errorsByRootSegment == null) {
            errorsByRootSegment = new HashMap();
        }
        return errorsByRootSegment;
    }
    
    protected void setErrorsByRootSegment(Map errorsByRootSegment) {
        this.errorsByRootSegment = errorsByRootSegment;
    }
    /**
     * Returns the name of the root segment for the logical record currently being 
     * processed/built.
     * <P>
     * The segment is the key for the list of errors occurring within it an its
     * child segments.  That is: all the segments pertaining to the logical represented
     * by this limb of the message tree beneath the root.   
     * <P>
     * This variable should be set or derived (usually by the Decoder who initializes this class)
     * to enable the listener to navigate as the common link between the rrors and the 
     * resulting DTO.  If null, all errors will be grouped under the Message Structure itself which
     * is the root of the entire message.
     */
    public String getRootName(){
        return rootName;
    }

    protected void setRootName(String rootName) {
        this.rootName = rootName;
    }

}
