package gov.va.med.domain.service.messaging.encode.hl7;

import java.util.Date;

import gov.va.med.domain.model.Patient;
import gov.va.med.domain.model.Prescription;
import gov.va.med.domain.model.RxRefillPayload;
import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.service.messaging.MessagingConstants;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.encode.hl7.structure.QBP_Q13_RXP;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7HeaderParameters;
import gov.va.med.domain.service.messaging.encode.hl7.util.RdfField;
import gov.va.med.domain.service.messaging.environment.EndPoint;
import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.v24.segment.RXE;

/**
 * RxProfileEncoder
 * This will encode and fill QBP_Q13 Prescription Profile query by parameter message structure segments.
 * This query will generate a tabular response.
 *
 * @author Slava Uchitel
 * @version $Id: RxProfileEncoder.java,v 1.8 2005/08/10 15:38:28 ye.jin Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public abstract class RxProfileEncoder extends HL7TabularEncoder implements MessagingConstants {
	private static final String[][] struct = {
		{"Prescription Number", "NM", "20"},
		{"IEN", "NM", "30"},
		{"Drug Name", "ST", "40"},
		{"Issue Date/Time", "TS", "26"},
		{"Last Fill Date", "TS", "26"},
		{"Release Date/Time", "TS", "26"},
		{"Expiration or Cancel Date", "TS", "26"},
		{"Status", "ST", "25"},
		{"Quantity", "NM", "11"},
		{"Days Supply", "NM", "3"},
		{"Number of Refills", "NM", "3"},
		{"Provider", "XPN", "150"},
		{"Place Order Number", "ST", "30"},
		{"Mail/Window", "ST", "1"},
		{"Division", "NM", "3"},
		{"Division Name", "ST", "20"},
		{"MHV Request Status", "NM", "3"},
		{"MHV Status Date", "TS", "26"},
		{"Remarks", "ST", "75"},
		{"SIG", "TX", "1024"}
	};

	public Object encode(IPayload payload, EndPoint endPoint)
	    throws MessagingException {
		RxRefillPayload rxRefillPayload = (RxRefillPayload)payload;
		Prescription[] prescriptions = rxRefillPayload.getPrescriptionList();
		String[] rxNumbers = new String[0];
		if(prescriptions != null) {
			if (getLogger().isDebugEnabled())
				getLogger().debug("In side encode - prescriptions is not null");
			rxNumbers = new String[prescriptions.length];
			for(int i = 0; i < prescriptions.length; i++) {
				rxNumbers[i] = String.valueOf(prescriptions[i].getPrescriptionNum());
				if (getLogger().isDebugEnabled())
					getLogger().debug("rxNumbers value::"+rxNumbers[i]);
			}
		}
		try {
			if (getLogger().isDebugEnabled())
				getLogger().debug("In side encode - in side try block");
			HL7HeaderParameters params = buildHL7HeaderParameters(endPoint, rxRefillPayload);

			if (getLogger().isDebugEnabled())	{
				getLogger().debug("In side encode - calling createQueryMessage.");			
				getLogger().debug("In side encode - getPatient::"+rxRefillPayload.getPatient().getSocialSecurityNumber());
				getLogger().debug("In side encode - params::"+params.getMessageControlId());
				getLogger().debug("In side encode - getFromDate::"+rxRefillPayload.getFromDate());
				getLogger().debug("In side encode - getToDate::"+rxRefillPayload.getToDate());
			}
			QBP_Q13_RXP qbp_q13 =
			    createQueryMessage(rxRefillPayload.getPatient(),
			                       rxNumbers,
			                       params,
			                       rxRefillPayload.getFromDate(),
			                       rxRefillPayload.getToDate());
			//Parser parser = (Parser) new PipeParser();
			//return parser.encode(qbp_q13);
			if (getLogger().isDebugEnabled())
				getLogger().debug("In side encode - createQueryMessage is good.");
			return qbp_q13.er7Encode();
		}
		catch(DataTypeException e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
		catch(HL7Exception e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
	}

	/*
	  message = "MSH|^~\&|MHV EVAULT|200MH^ISA-UCHITELSLAVA.VHA.MED.DNS   ^DNS|MHV VISTA|658^TEST.HEVET.MED.DNS   ^DNS|20050202152556-0500||QBP^Q13^QBP_Q13|1107375956189151861|T|2.4\r"+
	        "QPD|Q13^RX^HL70471|0|0|RX||20050214\r"+
	        "PID|||111223456^^^USASSA^SS\r"+
	        "RXE|||||||||||||||2378697\r"+
	        "RXE|||||||||||||||2378696\r"+
	        "RCP|I\r"+
	        "RDF|18|Prescription Number^NM^32~IEN^NM^30~Drug Name^ST^40~Issue Date/Time^TS^26~Last Fill Date^TS^26~Release Date/Time^TS^26~Expiration or Cancel Date^TS^26~Status^ST^20~Quantity^NM^3~Days Supply^NM^3~Number of Refills^NM^3~Provider^XPN^60~Place Order Number^ST^3~Mail/Window^ST^1~Division^NM^3~MHV Request Status^NM^3~Remarks^ST^75~SIG^TX^1024\r";
	*/

	protected QBP_Q13_RXP createQueryMessage(Patient patient,
	                                         String[] rxNumbers,
	                                         HL7HeaderParameters params,
	                                         Date fromDateLimit, Date toDateLimit)
	    throws DataTypeException, HL7Exception {

		if (getLogger().isDebugEnabled())
			getLogger().debug("Calling initializeFields");

		initializeFields();

		QBP_Q13_RXP qbp_q13 = new QBP_Q13_RXP();

		//MSH
		params.setFieldSeparator(DEFAULT_FIELD_SEPARATOR);
		params.setDelimiters(DEFAULT_DELIMITERS);
		params.setAcceptAcknowledgmentType(EMPTY_STRING);
		params.setApplicationAcknowledgmentType(EMPTY_STRING);
		params.setEvent(EVENT);
		params.setTrigger(TRIGGER);
		params.setMessageStructure(STRUCTURE);
		fillMSHSegment(qbp_q13.getMSH(), params);

		if (getLogger().isDebugEnabled())
			getLogger().debug("Setting params done.");


		//QPD segment
		fillQPDSegment(qbp_q13.getQPD(),
		               params.getMessageControlId(),
		               fromDateLimit,
		               toDateLimit);

		if (getLogger().isDebugEnabled())
			getLogger().debug("Setting QPD segment done.");

		//PID
		fillPIDSegment(qbp_q13.getPID(),
		               patient.getSocialSecurityNumber(),
		               patient.getIcn(), null);

		if (getLogger().isDebugEnabled())
			getLogger().debug("Setting PID segment done.");

		//getLogger().error("rxNumbers value::"+rxNumbers[0]);

		//RXEs - Rx refill specific
		fillRxeSegments(rxNumbers, qbp_q13);

		if (getLogger().isDebugEnabled())
			getLogger().debug("RDF value::"+qbp_q13.getRDF());

		//RDF
		fillRDFSegment(qbp_q13.getRDF());

		if (getLogger().isDebugEnabled())
			getLogger().debug("RCP value::"+qbp_q13.getRCP()+" Query mode::"+getQueryMode());

		//RCP
		fillRCPSegment(qbp_q13.getRCP(), getQueryMode());

		return qbp_q13;
	}

	private void initializeFields() throws HL7Exception {
		RdfField[] fields = new RdfField[struct.length];
		for(int i = 0; i < fields.length; i++)
			fields[i] = new RdfField(struct[i][0], struct[i][1], struct[i][2]);
		setFields(fields);
	}

	private void fillRxeSegments(String[] rxNumbers, QBP_Q13_RXP qbp_q13)
	    throws HL7Exception, DataTypeException {
		for(int i = 0; i < rxNumbers.length; i++) {
			RXE rxe = qbp_q13.getRXE(i);
			rxe.getPrescriptionNumber().setValue(rxNumbers[i]);
		}
	}

	private HL7HeaderParameters buildHL7HeaderParameters(EndPoint endPoint, RxRefillPayload rxRefillPayload) {
		HL7HeaderParameters params = new HL7HeaderParameters();
		params.setSrcStationNumber(endPoint.getEncodingParameter(MessagingConstants.SENDING_FACILITY));
		params.setSrcDomain(endPoint.getEncodingParameter(MessagingConstants.SENDING_FACILITY_DOMAIN));
		params.setSrcApplication(endPoint.getEncodingParameter(MessagingConstants.SENDING_APPLICATION));
		params.setDstStationNumber(endPoint.getEncodingParameter(MessagingConstants.RECEIVING_FACILITY));
		params.setDstDomain(endPoint.getEncodingParameter(MessagingConstants.RECEIVING_FACILITY_DOMAIN));
		params.setDstApplication(endPoint.getEncodingParameter(MessagingConstants.RECEIVING_APPLICATION));
		params.setSystemType(endPoint.getEncodingParameter(MessagingConstants.SYSTEM_TYPE));
		params.setMessageControlId(rxRefillPayload.getRequestId().toString());
		params.setAcceptAcknowledgmentType(getQueryMode()==QUERY_MODE_SYNCHRONOUS?EMPTY_STRING:ALWAYS_ACKNOWLEDGE);
		params.setApplicationAcknowledgmentType(getQueryMode()==QUERY_MODE_SYNCHRONOUS?EMPTY_STRING:ALWAYS_ACKNOWLEDGE);
		return params;
	}

	public String[][] getColumnStructure() {
		return struct;
	}
}
