package gov.va.med.domain.service.messaging.encode.hl7;

import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.PHRRequestPayload;
import gov.va.med.domain.model.Patient;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.encode.hl7.structure.QBP_Q11;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7HeaderParameters;
import gov.va.med.domain.service.messaging.environment.EndPoint;

import java.util.Date;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.v24.segment.MSH;
import ca.uhn.hl7v2.model.v24.segment.PID;
import ca.uhn.hl7v2.model.v24.segment.QPD;
import ca.uhn.hl7v2.model.v24.segment.RCP;

/**
 * QBP_Q11_Encoder
 * This will encode and fill QBP_Q11 query by parameter message structure segments.
 * All sub-classes will utilize the QBP_Q11 message structure.
 * This query will generate a segment pattern response.
 *
 * @author Slava Uchitel
 * @version $Id: QBP_Q11_Encoder.java,v 1.1 2005/07/20 19:37:29 tom Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public abstract class QBP_Q11_Encoder extends HL7MessageEncoder {
	private static final String EVENT = "QBP";
	private static final String TRIGGER = "Q11";
	private static final String STRUCTURE = "QBP_Q11";

	protected QBP_Q11 createQueryMessage(Patient patient, EndPoint endPoint,
	                                     PHRRequestPayload payload)
	    throws DataTypeException, HL7Exception {
		return createQueryMessage(patient,
		                          buildHL7HeaderParameters(endPoint, payload),
		                          payload.getFromDate(),
		                          payload.getToDate());
	}

	protected QBP_Q11 createQueryMessage(Patient patient, HL7HeaderParameters params, Date fromDateLimit, Date toDateLimit)
	    throws DataTypeException, HL7Exception {
		QBP_Q11 qbp_q11 = null;
		qbp_q11 = new QBP_Q11();
		String acceptAcknowledgmentType = (getQueryMode() == QUERY_MODE_SYNCHRONOUS)?EMPTY_STRING:ALWAYS_ACKNOWLEDGE;
        String applicationAcknowledgmentType = (getQueryMode() == QUERY_MODE_SYNCHRONOUS)?EMPTY_STRING:ALWAYS_ACKNOWLEDGE;

		//MSH
		MSH msh = qbp_q11.getMSH();
		params.setFieldSeparator(DEFAULT_FIELD_SEPARATOR);
		params.setDelimiters(DEFAULT_DELIMITERS);
		params.setEvent(EVENT);
		params.setTrigger(TRIGGER);
		params.setMessageStructure(STRUCTURE);	
		params.setAcceptAcknowledgmentType(acceptAcknowledgmentType);
		params.setApplicationAcknowledgmentType(applicationAcknowledgmentType);
		fillMSHSegment(msh, params);

		//QPD segment
		QPD qpd = qbp_q11.getQPD();
		fillQPDSegment(qpd, params.getMessageControlId(), fromDateLimit, toDateLimit, patient.getIcn(), TRIGGER, getSubjectArea());

		//PID
		PID pid = qbp_q11.getPID();
		fillPIDSegment(pid,
		               patient.getSocialSecurityNumber(),
		               patient.getIcn(), null);

		//RCP
		RCP rcp = qbp_q11.getRCP();
		fillRCPSegment(rcp, getQueryMode());

		return qbp_q11;
	}

	public Object encode(IPayload payload, EndPoint ep)
	    throws MessagingException {

		PHRRequestPayload phrRequestPayload = (PHRRequestPayload)payload;
		Patient patient = phrRequestPayload.getPatient();
		try {
			HL7HeaderParameters params = buildHL7HeaderParameters(ep, phrRequestPayload);
			QBP_Q11 qbp_q11 =
			    this.createQueryMessage(patient, params,
			                            phrRequestPayload.getFromDate(),
			                            phrRequestPayload.getToDate());
			//Parser parser = (Parser) new PipeParser();
			//return parser.encode(qbp_q11);
			return qbp_q11.er7Encode();
		}
		catch(DataTypeException e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
		catch(HL7Exception e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
	}

	/**
	 * Subject area is required for all flavors of QBP_Q11 messages. It goes in the QPD segment
	 */
	public abstract String getSubjectArea();

	/**
	 * Query mode is required for all flavors of QBP_Q11 messages to fill MSH and RCP segments
	 */
	public abstract int getQueryMode();

}
