package gov.va.med.domain.service.messaging.encode.hl7;


import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.Patient;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7HeaderParameters;
import gov.va.med.domain.service.messaging.environment.EndPoint;
import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.v24.message.ADT_A37;
import ca.uhn.hl7v2.model.v24.segment.EVN;
import ca.uhn.hl7v2.model.v24.segment.MSH;
import ca.uhn.hl7v2.model.v24.segment.PID;
import ca.uhn.hl7v2.parser.Parser;
import ca.uhn.hl7v2.parser.PipeParser;

/**
 * MpiPatientUnlinkEncoder
 * Encodes MPI Patient Unlink Request using ADT_A37 message structure.
 *
 * @author Slava Uchitel
 * @version $Id: MpiPatientUnlinkEncoder.java,v 1.8 2005/07/12 12:13:41 slava.uchitel Exp $
 * @since MHV 2.1 <br>Apr 5, 2005
 */
public class MpiPatientUnlinkEncoder extends MPILinkUnlinkEncoder {

	//  override to specify trigger
	protected String getTrigger() {
		return "A37";
	}

	protected ADT_A37 createMessage(Patient patient, HL7HeaderParameters params)
	    throws  HL7Exception, DataTypeException {

		ADT_A37 adt_a37 = null;
		adt_a37 = new ADT_A37();

		//MSH
		params.setAcceptAcknowledgmentType(ALWAYS_ACKNOWLEDGE);
		params.setApplicationAcknowledgmentType(ALWAYS_ACKNOWLEDGE);
		MSH msh = adt_a37.getMSH();
		fillMSHSegment(msh, params);

		//EVN
		EVN evn = adt_a37.getEVN();
		fillEVNSegment(evn, params.getSrcStationNumber());

		//PIDs
		PID pid1 = adt_a37.getPID();
		fillPIDSegment(pid1, 1,
		               patient.getIcn(), String.valueOf(((Long)patient.getId()).longValue()), 
		               params.getSrcStationNumber(), 
		               patient.getFirstName(), patient.getLastName());
		PID pid2 = adt_a37.getPID2();
		fillPIDSegment(pid2, 2,
		               patient.getIcn(), String.valueOf(((Long)patient.getId()).longValue()), 
		               params.getSrcStationNumber(), 
		               patient.getFirstName(), patient.getLastName());

		return adt_a37;
	}

	//Expects Patient as payload with ICN set
	public Object encode(IPayload payload, EndPoint endPoint)
	    throws MessagingException {
		/*Should produce similar message
				MSH^~|\&^MPIF TRIGGER^200HV~ISA-UCHITELSLAVA.VHA.MED.DNS   ~DNS^MPIF TRIGGER^200M~MPI.DNS        .MED.DNS   ~DNS^20050406091536-0000^^ADT~A37^1112811336123~5540^T^2.4^^^AL^NE^^^^^^
		EVN^^^^^^200HV
				PID^1^123456787890^123456787890~~~USVHA~NI~VA FACILITY ID&200HV^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
				PID^2^123456787890^123456787890~~~USVHA~NI~VA FACILITY ID&200HV^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		*/
		Patient patient = (Patient)payload;
		try {
			HL7HeaderParameters params = new HL7HeaderParameters(endPoint);
			ADT_A37 adt_a37 =
			    createMessage(patient, params);
			Parser parser = new PipeParser();

			return parser.encode(adt_a37);
		}
		catch(DataTypeException e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
		catch(HL7Exception e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
	}

    protected String getStructure() {
        return "ADT_A37";
    }

}
