package gov.va.med.domain.service.messaging.encode.hl7;

import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.PHRRequestPayload;
import gov.va.med.domain.model.Patient;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.encode.hl7.structure.QBP_Q13;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7HeaderParameters;
import gov.va.med.domain.service.messaging.encode.hl7.util.RdfField;
import gov.va.med.domain.service.messaging.environment.EndPoint;

import java.util.Date;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.v24.datatype.RCD;
import ca.uhn.hl7v2.model.v24.segment.MSH;
import ca.uhn.hl7v2.model.v24.segment.PID;
import ca.uhn.hl7v2.model.v24.segment.QPD;
import ca.uhn.hl7v2.model.v24.segment.RCP;
import ca.uhn.hl7v2.model.v24.segment.RDF;

/**
 * HL7TabularEncoder
 * This will encode and fill QBP_Q13 query by parameter message structure segments.
 * All sub-classes will utilize the QBP_Q13 message structure.
 * This query will generate a tabular response.
 *
 * @author Slava Uchitel
 * @version $Id: HL7TabularEncoder.java,v 1.7 2005/06/27 20:42:34 tom Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public abstract class HL7TabularEncoder extends HL7MessageEncoder {
	protected static final String EVENT = "QBP";
	protected static final String TRIGGER = "Q13";
	protected static final String STRUCTURE = "QBP_Q13";
	private RdfField[] _fields = null;

	public void setFields(RdfField[] fields)
	    throws HL7Exception {
		if(fields.length == 0)
			throw new HL7Exception("No fields found during initialization");
		_fields = fields;
	}

	protected void fillQPDSegment(QPD qpd, String requestId,
			Date fromDateLimit, Date toDateLimit) throws DataTypeException,
			HL7Exception {
		fillQPDSegment(qpd, requestId, fromDateLimit, toDateLimit, EMPTY_STRING, TRIGGER, getSubjectArea());
	}
	
	protected void fillRDFSegment(RDF rdf)
	    throws DataTypeException, HL7Exception {
		rdf.getNumberOfColumnsPerRow().setValue(String.valueOf(_fields.length));
		for(int i = 0; i < _fields.length; i++) {
			RCD rcd = rdf.getColumnDescription(i);
			rcd.getSegmentFieldName().setValue(_fields[i].getName());
			rcd.getHL7DateType().setValue(_fields[i].getType());
			rcd.getMaximumColumnWidth().setValue(_fields[i].getLength());
		}
	}

	protected QBP_Q13 createQueryMessage(Patient patient, EndPoint endPoint,
	                                     PHRRequestPayload payload)
	    throws DataTypeException, HL7Exception {
		return createQueryMessage(patient,
		                          buildHL7HeaderParameters(endPoint, payload),
		                          payload.getFromDate(),
		                          payload.getToDate());
	}

	protected QBP_Q13 createQueryMessage(Patient patient, HL7HeaderParameters params, Date fromDateLimit, Date toDateLimit)
	    throws DataTypeException, HL7Exception {
		QBP_Q13 qbp_q13 = null;
		qbp_q13 = new QBP_Q13();
		String acceptAcknowledgmentType = (getQueryMode() == QUERY_MODE_SYNCHRONOUS)?EMPTY_STRING:ALWAYS_ACKNOWLEDGE;
        String applicationAcknowledgmentType = (getQueryMode() == QUERY_MODE_SYNCHRONOUS)?EMPTY_STRING:ALWAYS_ACKNOWLEDGE;

		//MSH
		MSH msh = qbp_q13.getMSH();
		params.setFieldSeparator(DEFAULT_FIELD_SEPARATOR);
		params.setDelimiters(DEFAULT_DELIMITERS);
		params.setEvent(EVENT);
		params.setTrigger(TRIGGER);
		params.setMessageStructure(STRUCTURE);
		params.setAcceptAcknowledgmentType(acceptAcknowledgmentType);
		params.setApplicationAcknowledgmentType(applicationAcknowledgmentType);
		fillMSHSegment(msh, params);

		//QPD segment
		QPD qpd = qbp_q13.getQPD();
		fillQPDSegment(qpd, params.getMessageControlId(), fromDateLimit, toDateLimit, patient.getIcn(), TRIGGER, getSubjectArea());

		//PID
		PID pid = qbp_q13.getPID();
		fillPIDSegment(pid,
		               patient.getSocialSecurityNumber(),
		               patient.getIcn(), null);

		//RDF
		RDF rdf = qbp_q13.getRDF();
		fillRDFSegment(rdf);

		//RCP
		RCP rcp = qbp_q13.getRCP();
		fillRCPSegment(rcp, getQueryMode());

		return qbp_q13;
	}

	public Object encode(IPayload payload, EndPoint ep)
	    throws MessagingException {
		initalizeFields();  // adds column metadata
		PHRRequestPayload phrRequestPayload = (PHRRequestPayload)payload;
		Patient patient = phrRequestPayload.getPatient();
		try {
			HL7HeaderParameters params = buildHL7HeaderParameters(ep, phrRequestPayload);
			QBP_Q13 qbp_q13 =
			    this.createQueryMessage(patient, params,
			                            phrRequestPayload.getFromDate(),
			                            phrRequestPayload.getToDate());
			//Parser parser = (Parser) new PipeParser();
			//return parser.encode(qbp_q13);
			return qbp_q13.er7Encode();
		}
		catch(DataTypeException e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
		catch(HL7Exception e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
	}

	private void initalizeFields() throws MessagingException {
		try {

			RdfField[] fields = new RdfField[getColumnStructure().length];
			for(int i = 0; i < fields.length; i++)
				fields[i] = new RdfField(getColumnStructure()[i][0],
				                         getColumnStructure()[i][1],
				                         getColumnStructure()[i][2]);
			setFields(fields);
		}
		catch(HL7Exception e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
	}

	/**
	 * Subject area is required for all flavors of QBP_Q13 messages. It goes in the QPD segment
	 */
	public abstract String getSubjectArea();

	/**
	 * Structure for expected columns is required for all flavors of QBP_Q13 messages to fill RDF segment
	 */
	public abstract String[][] getColumnStructure();

	/**
	 * Query mode is required for all flavors of QBP_Q13 messages to fill MSH and RCP segments
	 */
	public abstract int getQueryMode();

}
