package gov.va.med.domain.service.messaging.encode.hl7;

import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.PHRRequestPayload;
import gov.va.med.domain.model.Patient;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7HeaderParameters;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7Helper;
import gov.va.med.domain.service.messaging.environment.EndPoint;

import java.util.Date;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.Message;
import ca.uhn.hl7v2.model.v24.segment.MSH;
import ca.uhn.hl7v2.model.v24.segment.QRD;
import ca.uhn.hl7v2.model.v24.segment.QRF;
import ca.uhn.hl7v2.parser.Parser;
import ca.uhn.hl7v2.parser.PipeParser;

/**
 * HL7OriginalStyleEncoder
 * This will encode and fill VXQ_V01 query by parameter message structure segments.
 * All sub-classes will utilize the VXQ_V01 message structure.
 * This query will generate a tabular response.
 *
 * @author Slava Uchitel
 * @version $Id: HL7OriginalStyleEncoder.java,v 1.11 2005/07/27 19:28:05 slava.uchitel Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public abstract class HL7OriginalStyleEncoder extends HL7MessageEncoder {

	protected Message createQueryMessage(Message message, Patient patient,
			HL7HeaderParameters params, Date fromDateLimit, Date toDateLimit)
			throws DataTypeException, HL7Exception {
		String acceptAcknowledgmentType = (getQueryMode() == QUERY_MODE_SYNCHRONOUS) ? EMPTY_STRING
				: ALWAYS_ACKNOWLEDGE;
		String applicationAcknowledgmentType = (getQueryMode() == QUERY_MODE_SYNCHRONOUS) ? EMPTY_STRING
				: ALWAYS_ACKNOWLEDGE;

		//MSH
		params.setFieldSeparator(DEFAULT_FIELD_SEPARATOR);
		params.setDelimiters(DEFAULT_DELIMITERS);
		params.setEvent(getEvent());
		params.setTrigger(getTrigger());
		params.setMessageStructure(getMessageStructure());
		params.setAcceptAcknowledgmentType(acceptAcknowledgmentType);
		params.setApplicationAcknowledgmentType(applicationAcknowledgmentType);
		fillMSHSegment(getMsh(message), params);

		//QRD segment
		fillQRDSegment(getQrd(message), params.getMessageControlId(), patient);

		//QRF
		fillQRFSegment(getQrf(message), fromDateLimit, toDateLimit);

		return message;
	}
	
	protected void fillQRDSegment(QRD qrd, String requestId, Patient patient)
			throws DataTypeException, HL7Exception {
		String currentTime = dateToyyyyMMddhhmmss(new Date());
		qrd.getQueryDateTime().setValue(currentTime);		
		qrd.getQueryFormatCode().setValue("R");
		qrd.getQueryPriority().setValue((getQueryMode() == QUERY_MODE_SYNCHRONOUS) ? "I" : "D");
		qrd.getQueryID().setValue(requestId);

		qrd.getWhoSubjectFilter(0).getIDNumber().setValue(patient.getIcn());
		qrd.getWhoSubjectFilter(0).getFamilyName().getSurname().setValue(patient.getLastName());
		qrd.getWhoSubjectFilter(0).getGivenName().setValue(patient.getFirstName());
		qrd.getWhoSubjectFilter(0).getIdentifierTypeCode().setValue(HL7Helper.pidType[HL7Helper.ID_ICN]);
		
		
		qrd.getWhatSubjectFilter(0).getIdentifier().setValue(getSubjectFilter());
		qrd.getWhatDepartmentDataCode(0).getIdentifier().setValue(getSubjectArea());
	}
	
	protected void fillQRFSegment(QRF qrf, Date fromDateLimit, Date toDateLimit)
	    throws DataTypeException, HL7Exception {
		qrf.getWhereSubjectFilter(0).setValue(getSubjectFilter());
		qrf.getWhenDataStartDateTime().setValue(dateToyyyyMMdd(fromDateLimit));		
		qrf.getWhenDataEndDateTime().setValue(dateToyyyyMMdd(toDateLimit));
	}

	public Object encode(IPayload payload, EndPoint ep)
	    throws MessagingException {
		PHRRequestPayload phrRequestPayload = (PHRRequestPayload)payload;
		Patient patient = phrRequestPayload.getPatient();
		try {
			HL7HeaderParameters params = buildHL7HeaderParameters(ep, phrRequestPayload);
			
			Message message =
			    createQueryMessage(getQueryMessage(), patient, params, phrRequestPayload.getFromDate(),
			    					phrRequestPayload.getToDate());
			Parser parser = new PipeParser();
			return parser.encode(message);
			//return message.er7Encode();
		}
		catch(DataTypeException e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
		catch(HL7Exception e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
	}

	/**
	 * Event is required for all flavors of Original Style messages. It goes in the MSH segment
	 */
	public abstract String getEvent();
		
	/**
	 * Trigger is required for all flavors of Original Style messages. It goes in the MSH segment
	 */
	public abstract String getTrigger();
	
	/**
	 * Message structure is required for all flavors of Original Style messages. It goes in the MSH segment
	 */
	public abstract String getMessageStructure();
		
	/**
	 * Subject area is required for all flavors of Original Style messages. It goes in the QRD segment
	 */
	public abstract String getSubjectArea();

	/**
	 * Subject filter is required for all flavors of Original Style messages. It goes in the QRD segment
	 */
	public abstract String getSubjectFilter();
	
	/**
	 * Query mode is required for all flavors of Original Style messages to fill QRD segments
	 */
	public abstract int getQueryMode();
	
	/**
	 * This will return different message structure for each Original Style encoder subclass.
	 */
	public abstract Message getQueryMessage()
		throws DataTypeException, HL7Exception;	

	/**
	 * This will return an MSH segment from the message.
	 */
	public abstract MSH getMsh(Message message);	
	
	/**
	 * This will return an QRD segment from the message.
	 */
	public abstract QRD getQrd(Message message);		
	
	/**
	 * This will return an QRF segment from the message.
	 */
	public abstract QRF getQrf(Message message);			
}
