package gov.va.med.domain.service.messaging.decode.hl7;

import gov.va.med.domain.model.DtoListPayload;
import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.Prescription;
import gov.va.med.domain.model.StatusPayload;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.parse.BaseProcessingParseErrorListener;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.Structure;
import ca.uhn.hl7v2.model.v24.group.ORP_O10_ORCRXONTERXRRXCNTE;
import ca.uhn.hl7v2.model.v24.group.ORP_O10_RXONTERXRRXCNTE;
import ca.uhn.hl7v2.model.v24.message.ORP_O10;
import ca.uhn.hl7v2.model.v24.segment.RXE;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;
import org.springframework.stereotype.Component;

/**
 * RxRefillResultsDecoder
 * Decodes Rx Refill query result messages using the ORP_O10 message structure.
 *
 * @author Slava Uchitel
 * @version $Id: RxRefillResultsDecoder.java,v 1.8 2005/09/16 21:44:53 ye.jin Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
@Component
public class RxRefillResultsDecoder extends HL7MessageDecoder {
    public String getMessageName() {
            return "ORP-O10";
     }

     /*
        String msg = "MSH|^~\\&|EVET VISTA|658^TEST.HEVET.MED.DNS   ^DNS|EVET EVAULT|200MH^ISA-UCHITELSLAVA.VHA.MED.DNS   ^DNS|20050203120915-0400||ORP^O10|65810482601|D|2.4|||||US\r"+
                     "MSA|AA|0|\r"+
                     "PID|||111223456^^^USSSA&&HL70363^SS^VA FACILITY ID&200MH&L~9728^^^USVHA&&HL70363^PI^VA FACILITY ID&658&L||MYHVET ONE PATIENT^^^\r"+
                     "ORC|UA|2378697-20050203120728|2378697-20050203120728\r"+
                     "RXE|1^^^20050203120728|UA^-2^HL70119|1||1 refill unit||1||||||||2378697\r"+
                     "ORC|UA|2378696-20050203120728|2378696-20050203120728\r"+
                     "RXE|1^^^20050203120728|UA^-2^HL70119|1||1 refill unit||1||||||||2378696\r";
      */

     private Integer intFieldToInteger(String ss)
     {
        if (ss==null || ss.trim().length()==0 || ss.trim().equals("\"\"")) {
          return null;
        }
        return new Integer(ss);
     }

    private Date tsFieldToDate(String ss)
    	throws ParseException
    {
        if (ss==null || ss.trim().length()==0 || ss.trim().equals("\"\""))
          return null;
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
        return sdf.parse(ss);
    }

    public IPayload decode(Object encodedPayload)
        	throws MessagingException
        {
            DtoListPayload payload = new DtoListPayload();
            ORP_O10 orp_o10  = null;
            try
            {
                orp_o10  = (ORP_O10)parse(encodedPayload);
    			StatusPayload statusPayload = checkMSAStatus(orp_o10);
    			if(statusPayload != null) {
    				return statusPayload;
    			}

    			Structure[] tmp = orp_o10.getORP_O10_PIDNTEORCRXONTERXRRXCNTE().getAll("ORP_O10_ORCRXONTERXRRXCNTE");
    			for(int i = 0; i < tmp.length; i++) {
    				ORP_O10_RXONTERXRRXCNTE tmp0 = ((ORP_O10_ORCRXONTERXRRXCNTE)tmp[i]).getORP_O10_RXONTERXRRXCNTE();
    				Structure[] tmp1 = tmp0.getAll("RXE");
    				for(int j = 0; j < tmp1.length; j++) {
    					RXE rxeSegment = (RXE)tmp1[j];
    					payload.addDto(buildPrescription(rxeSegment));
    				}
    			}
    	        return payload;
            } catch (EncodingNotSupportedException e) {
                throw new MessagingException("Error casting incoming message to ORP_O10 : \r\n" + encodedPayload, e);
            } catch (HL7Exception e) {
                throw new MessagingException("Error casting incoming message to ORP_O10 : \r\n" + encodedPayload, e);
            }

        }

        private Prescription buildPrescription(RXE rxe) throws HL7Exception {
             String strRxNum = rxe.getPrescriptionNumber().getValue();
             Prescription prescription = new Prescription();
             prescription.setPrescriptionNum(strRxNum);

             String strQuantity = rxe.getQuantityTiming().getQuantity().getQuantity().getValue();
             String strRequestDateTime = rxe.getQuantityTiming().getStartDateTime().getValue();
             String strRefillStatus = rxe.getGiveCode().getIdentifier().getValue();
             String strMhvStatus = rxe.getGiveCode().getText().getValue();
             String strDivision = rxe.getProviderSAdministrationInstructions(0).getIdentifier().getValue();
             String strNumberOfRefills = rxe.getNumberOfRefills().getValue();
             try {
                prescription.setQuantity(new Float(strQuantity));
             } catch (NumberFormatException e) {
             	String err = "Received a prescription refill acknowledgment for a prescription that has a non-numeric quantity value so defaulting to null.";
             	getLogger().warn(err + " Rx # " + strRxNum);
             }

             try {
                 prescription.setMhvStatus(new Integer(strMhvStatus));
             } catch (NumberFormatException e) {
                String err = "Received a prescription refill acknowledgment for a prescription that has a non-numeric MHV status so defaulting to null.";
                getLogger().warn(err + " Rx # " + strRxNum);
             }

             try {
                 Integer division = new Integer(strDivision);
                 prescription.setDivision(division);
             } catch (NumberFormatException e) {
                 String err = "Received a prescription refill acknowledgment for a prescription that has a non-numeric division so defaulting to null.";
                 getLogger().warn(err + " Rx # " + strRxNum);
             }
             prescription.setPharmacyStatus(strRefillStatus);
             prescription.setNumReFill(intFieldToInteger(strNumberOfRefills));
             return prescription;
        }
        public void initializeListeners() {
            BaseProcessingParseErrorListener listener =
                	new BaseProcessingParseErrorListener("RXE");
            setErrorListener(listener);
            setEventListener(listener);
        }
    }
