package gov.va.med.domain.service.messaging.decode.hl7;
//gov.va.med.domain.messaging.decode.hl7.RTB_K13_RXP_Decoder
import gov.va.med.domain.model.DtoListPayload;
import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.StatusPayload;
import gov.va.med.domain.model.validation.RawPrescription;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.parse.BaseProcessingParseErrorListener;
import gov.va.med.domain.service.messaging.parse.IHL7ProcessingParseErrorListener;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.stereotype.Component;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.GenericComposite;
import ca.uhn.hl7v2.model.Type;
import ca.uhn.hl7v2.model.Varies;
import ca.uhn.hl7v2.model.v24.group.RTB_K13_RDFRDT;
import ca.uhn.hl7v2.model.v24.message.RTB_K13;
import ca.uhn.hl7v2.model.v24.segment.QAK;
import ca.uhn.hl7v2.model.v24.segment.RDT;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;

/**
* Returns a DtoList of Prescriptions by parsing an RTB_K13 message containing prescirptions
*
* @author Slava Uchitel
* @version $Id: RxProfileDecoder.java,v 1.15 2005/07/21 19:08:24 slava.uchitel Exp $
* @since MHV 2.0 <br>Mar 1, 2005
*/
@Component
public class RxProfileDecoder extends HL7MessageDecoder {
	private static Logger log = LogManager.getLogger(RxProfileDecoder.class);

	public String getMessageName() {
		return "RTB-K13";
	}

	/*
   //good response
   msg = 
       "MSH|^~\\&|EVET VISTA|658^TEST.HEVET.MED.DNS   ^DNS|EVET EVAULT|200MH^ISA-UCHITELSLAVA.VHA.MED.DNS   ^DNS|20050202152609-0400||RTB^K13|65810482598|D|2.4|||||US\r"+
       "MSA|AA|1107375956189151861|\r"+
       "QAK|0-96001|OK|Q13^REALTIME_RxList^HL70471|3|3|0\r"+
       "QPD|Q13^REALTIME_RxList^HL70471|0-96001|0|96001||||9728\r"+
       "RDF|18|Prescription Number^ST^32~IEN^NM^30~Drug Name^ST^40~Issue Date/Time^TS^26~Last Fill Date^TS^26~Release Date/Time^TS^26~Expiration or Cancel Date^TS^26~Status^ST^20~Quantity^NM^10~Days Supply^NM^3~Number of Refills^NM^3~Provider^XCN^130~Placer Order Number^ST^40~Mail/Window^ST^1~Division^NM^3~MHV Request Status^NM^3~Remarks^ST^75~SIG^TX^1024\r"+
       "RDT|2117571D||DIFLUNISAL 500MG TAB|20040426|20040426||20050427|ACTIVE|180|90|3|9669^SMITH^JOHN^ADAM^JR^DR^MD|10068316|W|1||||RENEWED FROM RX # 2117571C|TAKE ONE TABLET BY MOUTH TWICE A DAY WITH FOOD\r"+
       "RDT|2378697||IBUPROFEN 600MG TAB|20040426|20040426||20050427|ACTIVE|270|90|3|22256|10068318|W|1||-2|||TAKE ONE TABLET BY MOUTH TWICE A DAY WITH FOOD|TAKE ONE TABLET BY MOUTH EVERY 8 HOURS FOR PAIN\r"+
       "RDT|2378696||VERAPAMIL HCL 180MG SA TAB|20040426|20040426||20050427|ACTIVE|270|90|3|22256|10068317|W|1||-2|||TAKE ONE TABLET BY MOUTH TWICE A DAY WITH FOOD|TAKE ONE TABLET BY MOUTH EVERY 8 HOURS FOR PAIN|TAKE ONE TABLET BY MOUTH EVERY 8 HOURS\r";
   //error response
   msg = 
       "MSH|^~\\&|EVET VISTA|658^TEST.HEVET.MED.DNS   ^DNS|EVET EVAULT|200MH^ISA-UCHITELSLAVA.VHA.MED.DNS   ^DNS|20050204111029-0400||RTB^K13|65810482605|D|2.4|||||US\r"+
       "MSA|AE|1107533501505740330|Missing Patient ID\r"+
       "ERR|PID^1^3^101&Missing Patient ID\r"+
       "QAK|0-96001|AE|Q13^REALTIME_RxList^HL70471|0|0|0\r"+
       "QPD|Q13^REALTIME_RxList^HL70471|0-96001|0|96001||||\r"; 

	 */

	public IPayload decode(Object encodedPayload) 
			throws MessagingException            
			{
		DtoListPayload result = new DtoListPayload();
		RTB_K13 rtb_k13 = null; 
		try
		{
			rtb_k13 = (RTB_K13)parse(encodedPayload); 
			StatusPayload statusPayload = checkMSAStatus(rtb_k13);
			if(statusPayload != null) {
				return statusPayload;
			}    			

			QAK qak = rtb_k13.getQAK();
			String queryStatus = qak.getQueryResponseStatus().getValue();
			if (queryStatus.equalsIgnoreCase("NF")){
				return result; 
			}

			if (!queryStatus.equalsIgnoreCase("OK")) {
				log.error("Unable parse the message: " + encodedPayload);
				throw new MessagingException("Received processing error : " + encodedPayload); 
			}

			// All is OK, so process normally.  
			int recordCount = Integer.parseInt(qak.getThisPayload().getValue());
			RTB_K13_RDFRDT rdfrdt = rtb_k13.getRTB_K13_RDFRDT();
			for (int i =0; i<recordCount; i++)
			{                
				RawPrescription rawPrescription = buildRawPrescription(rdfrdt, i);
				// add Presciption and any errors to list as it is built and validated.
				if (log.isDebugEnabled()) {
					log.debug("Parsing RTB^K13, raw prescription: " + rawPrescription);
				}
				rawPrescription.buildAndValidate(result); 
			}
			return result;


			//          } 
			//      else {
			//              log.error("Unable parse the message: " + message);
			//              throw new MessagingException("Received processing error : " + msg);
			//          }
		} 
		catch(ClassCastException e) {
			throw new MessagingException("Error casting incoming message to RTB_K13: \r\n" + encodedPayload, e);
		}
		catch (EncodingNotSupportedException e) {
			throw new MessagingException("Error casting incoming message to RTB_K13 : \r\n" + rtb_k13, e);        
		} 
		catch (HL7Exception e) {
			throw new MessagingException("Error casting incoming message to RTB_K13 : \r\n" + rtb_k13, e); 
		}	        
			}
	/** 
	 * Builds the RawPrescription (unedited, string values) from the current RDT segment
	 */
	private RawPrescription buildRawPrescription(RTB_K13_RDFRDT rdfrdt, int i) throws HL7Exception {
		RDT rdt = rdfrdt.getRDT(i);

		RawPrescription rawPrescription = new RawPrescription();
		//Integer fields
		rawPrescription.setPlaceOrderNum(getString(rdt, 13, 0));
		rawPrescription.setNumberOfRefills(getString(rdt, 11, 0));
		rawPrescription.setQuantity(getString(rdt, 9, 0));
		rawPrescription.setDaysSupply(getString(rdt, 10, 0));
		//Date fields
		rawPrescription.setIssueDateTime(getString(rdt, 4, 0));
		rawPrescription.setLastFillDate(getString(rdt, 5, 0));
		rawPrescription.setReleaseDate(getString(rdt, 6, 0));
		rawPrescription.setExpirationOrCancelDate(getString(rdt, 7, 0));
		rawPrescription.setMhvStatusDate(getString(rdt, 18, 0));
		//String fields
		rawPrescription.setPrescriptionNum(getString(rdt, 1, 0));
		rawPrescription.setIen(getString(rdt, 2, 0));

		//provider
		Type provider = ((Varies)rdt.getField(12,0)).getData();
		if (provider instanceof GenericComposite) {	
			GenericComposite composite = (GenericComposite) provider;
			Type[] components = composite.getComponents(); 
			rawPrescription.setProviderId(((Varies)components[0]).getData().toString());
			rawPrescription.setProviderLastName(((Varies)components[1]).getData().toString());
			rawPrescription.setProviderFirstName(((Varies)components[2]).getData().toString());
		}
		else {
			rawPrescription.setProviderId(provider.toString());
		}

		rawPrescription.setDivision(getString(rdt, 15, 0));
		rawPrescription.setDivisionName(getString(rdt, 16, 0));
		rawPrescription.setDrugName(getString(rdt, 3, 0));
		rawPrescription.setSig(getString(rdt, 20, 0)); 
		rawPrescription.setRemarks(getString(rdt, 19, 0));
		rawPrescription.setMailOrWindow(getString(rdt, 14, 0));
		rawPrescription.setPharmacyStatus(getString(rdt, 8, 0));
		rawPrescription.setMhvRequestStatus(getString(rdt, 17, 0));
		return rawPrescription;
	}

	/**
	 * Determines the appropriate EventListener for the target message type,
	 * sets it and returns it.    
	 */
	public IHL7ProcessingParseErrorListener getProcessingErrorListener(){
		return (IHL7ProcessingParseErrorListener)getErrorListener();
	}

	public void initializeListeners() {
		BaseProcessingParseErrorListener listener = 
				new BaseProcessingParseErrorListener("RDT");
		setErrorListener(listener);
		setEventListener(listener);
	}  
} 
