package gov.va.med.domain.service.messaging.decode.hl7;

import gov.va.med.domain.model.DtoListPayload;
import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.Patient;
import gov.va.med.domain.service.messaging.MessagingException;
import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.GenericComposite;
import ca.uhn.hl7v2.model.Structure;
import ca.uhn.hl7v2.model.Type;
import ca.uhn.hl7v2.model.Varies;
import ca.uhn.hl7v2.model.v23.message.ACK_Q02;
import ca.uhn.hl7v2.model.v23.segment.RDF;
import ca.uhn.hl7v2.model.v23.segment.RDT;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;

/**
 * MpiQueryResultsDecoder
 * Decodes MPI patient lookup result messages using the ACK_Q02 message structure.
 *
 * @author Slava Uchitel
 * @version $Id: MpiQueryResultsDecoder.java,v 1.17 2005/09/14 14:17:17 tom Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public class MpiQueryResultsDecoder extends HL7MessageDecoder {
	/*
	String mpiQueryResult =
		"MSH|^~\\&|MPI|MPI|MPI_LOAD|200M|20050202105358-0500||ADT^A31|1107359638556959857|P|2.3\r"+
	    "MSA|AA|1107359638556959857\r"+
	    "QAK|432786934|OK\r"+
	    "RDF|8|@00108.2^ST^30~@00108.3^ST^16~@00108.1^ST^30~@00122^ST^9~@00110^ST^8~@00105^ST^19~@00756^ST^5~@00169^ST^999\r"+
	    "RDT|RANDY|KELLEE|LOWERY|432786934|19430914|1003130309V597171|658|600^200203110930-0400^A3~605^200208201058-0400^A3~658^^~200HVS^^~";
	*/

	public IPayload decode(Object encodedPayload)
	    throws MessagingException {
		
		//Replace Message Type to ensure message structure is set to ACK_Q02, as expected in parse statement below.
		//Two replaceFirst statements are necessary to cover the possibility that patientLookup responses could contain
		//ADT^A31 Message Type (as in System Test) or ACK^Q02 Message Type (as in Production).  
		encodedPayload = ((String)encodedPayload).replaceFirst("\\|ACK\\^Q02\\|", "|ACK^Q02^ACK_Q02|");
	    encodedPayload = ((String)encodedPayload).replaceFirst("ADT\\^A31", "ACK^Q02^ACK_Q02");
		
	    try {
	    	ACK_Q02 ack_q02 = (ACK_Q02)parse(encodedPayload);
			RDF rdf = (RDF)ack_q02.get("RDF");
			int columns = Integer.parseInt(rdf.getNumberOfColumnsPerRow().getValue());

			Structure[] patients = null;
			try {
				patients = ack_q02.getAll("RDT");
			}
			catch(HL7Exception e) {
				return new DtoListPayload(0);
			}

			return buildPatientList(columns, patients);
		}
		catch(ClassCastException e) {
			throw new MessagingException("Error casting incoming message: \r\n" + encodedPayload, e);
		}
		catch(EncodingNotSupportedException e) {
			throw new MessagingException("Error processing incoming message: \r\n" + encodedPayload, e);
		}
		catch(HL7Exception e) {
			throw new MessagingException("Error processing incoming message: \r\n" + encodedPayload, e);
		}
	}

	private DtoListPayload buildPatientList(int columns,
	                                        Structure[] patients)
	    throws HL7Exception, MessagingException {

	    DtoListPayload patientList = new DtoListPayload(patients.length);
		for(int i = 0; i < patients.length; i++) {
			Patient patient = new Patient();

			RDT rdt = (RDT)patients[i];
			
			for(int fieldNumber = 1; fieldNumber <= columns; fieldNumber++) {
				addFieldToPatient(patient, fieldNumber, rdt.getField(fieldNumber));
			}
			patientList.addDto(patient);
		}
		return patientList;
	}

	/**
	 * @param patient
	 * @param fieldNumber
	 * @param fld
	 * @throws MessagingException
	 */
	private void addFieldToPatient(Patient patient, 
	        					   int fieldNumber, 
	        					   Type[] fld) throws MessagingException {
		if (fld.length==0)
			return;
		Type data;
		switch(fieldNumber) {
			case 1: //last name
				data = ((Varies)fld[0]).getData();
				patient.setLastName(data.toString());
				break;
			case 3: //first name
				data = ((Varies)fld[0]).getData();
				patient.setFirstName(data.toString());
				break;
			case 4: //ssn
				data = ((Varies)fld[0]).getData();
				patient.setSocialSecurityNumber(data.toString());
				break;
			case 5: //dob
				data = ((Varies)fld[0]).getData();
				patient.setDateOfBirth(stringToDate(data.toString()));
				break;
			case 6:
				data = ((Varies)fld[0]).getData();
				patient.setIcn(data.toString());
				break;
			case 7:
				data = ((Varies)fld[0]).getData();
				patient.setCmorStationNumber(data.toString());
				break;
			case 8:
				String[] stations = new String[fld.length];
				for(int k = 0; k < fld.length; k++) {
					Type stationNumber = null;
					data = ((Varies)fld[k]).getData();
					if(data instanceof GenericComposite) {
						GenericComposite gc = (GenericComposite)data;
						Type[] rpt = gc.getComponents();
						stationNumber = ((Varies)rpt[0]).getData();
					}
					else {
					    stationNumber = data;
					}
					stations[k] = (stationNumber != null) ? stationNumber.toString() : null;
				}
				patient.setInstitutionsFromStationNumbers(stations);
				break;
			default:
				;//do nothing
		}
	}
	
    public String getExpectedHL7Version() {
        return "2.3";
    }
}
