package gov.va.med.domain.service.messaging.decode.hl7;

import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.Institution;
import gov.va.med.domain.model.Patient;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7v24Helper;
import java.util.HashSet;
import java.util.Set;
import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.v24.datatype.CX;
import ca.uhn.hl7v2.model.v24.datatype.XPN;
import ca.uhn.hl7v2.model.v24.message.ADT_A01;
import ca.uhn.hl7v2.model.v24.segment.EVN;
import ca.uhn.hl7v2.model.v24.segment.PD1;
import ca.uhn.hl7v2.model.v24.segment.PID;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;

/**
 * Decodes HL7 Message ADT-A08 - update Patient information
 * MPI sends this message each time there is a change to patient
 * demographic information such as name, address, SSN, date of death, etc.
 * The message contains the most current demographic information including any changed fields.
 *
 * @author Joel Goldberg
 * @version $Id: MpiPatientUpdateDecoder.java,v 1.5 2005/08/17 21:51:52 tom Exp $
 * @since MHV 2.0 <br>May 12, 2005
 */
public class MpiPatientUpdateDecoder extends HL7MessageDecoder {

	public static String sampleMessage =
	    "MSH^~|\\&^RG ADT^500~DNS   .DNS      .MED.DNS   ~DNS^RG ADT^200M~MPI.DNS      .MED.DNS   ~DNS^20020921133629-0500^^ADT~A08^500167099^T^2.4^^^AL^AL^US\r" +
	    "EVN^A1^20020913112211-0500^^A1^12584~REED~DANNY~~~~~~USVHA&&0363~L~~~NI~VA FACILITY ID&500&L^20020913112211-0500^500\r" +
	    "PID^1^^1001170555V~~~USVHA&&0363~NI~VA FACILITY ID&500&L|040345123~~~USSSA&&0363~SS~VA FACILITY ID&500&L|100000050~~~USVHA&&0363~PI~VA FACILITY ID&500&L|5678923~~~USVBA&&0363~PN~VA FACILITY ID&500&L|500000524V103809~~~USVHA&&0363~NI~VA FACILITY ID&500&L~~20020913110104-0500|1001170419V238836~~~USVHA&&0363~NI~VA FACILITY ID&500&L~~20020920181506-0500^^JOSEPH~TEST~FIRST~~~~L^WONDER BREAD~~~~~~M^19450403^M^^^\"\"~~\"\"~ALABAMA~\"\"~~P|~~MOBILE~ALABAMA~~~N^\"\"^\"\"^\"\"^^S^9^^^^^^^^^^^^\"\"^^\r" +
	    "PD1^^^DETROIT~~553^\"\"\r" +
	    "PV1^1^I^11E REHAB~1102~10^^^\"\"~\"\"~\"\"^623~BAKER~ROY^^^15^^^^^^^^NON-VETERAN (OTHER)^^^12^^^^^^^^^^^^^^^^^^500A^^^^^20020913112211-0500^\"\"\r" +
	    "OBX^1^CE^~SECURITY LEVEL^\"\"^0^\"\"^\"\"^\"\"^\"\"^\"\"^F^\"\"^\"\"^20020918095831-0500^\"\"^\"\"^\"\"\r" +
	    "ZPD^1^\"\"^YES^AL^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^0^\"\"^\"\"^\"\"^9^\"\"^\"\"^0^\"\"^\"\"^\"\"\r" +
	    "ZSP^1^0^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"\r" +
	    "ZEL^1^\"\"^\"\"^\"\"^\"\"^5678923^\"\"^0^NON-VETERAN OTHER)^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"\r" +
	    "ZCT^1^1^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"\r" +
	    "ZEM^1^1^\"\"^\"\"^\"\"^\"\"^\"\"^\"\"\r" +
	    "ZFF^2^.08;\r";

	public IPayload decode(Object encodedPayload) throws MessagingException {

		try {
			ADT_A01 adt_a01 = (ADT_A01)parse(encodedPayload);
			return buildDto(adt_a01);
		}
		catch(ClassCastException e) {
			throw new MessagingException("Error casting incoming message to ADT_A01: \r\n" + encodedPayload, e);
		}
		catch(EncodingNotSupportedException e) {
			throw new MessagingException("Error processing incoming message: \r\n" + encodedPayload, e);
		}
		catch(HL7Exception e) {
			throw new MessagingException("Error processing incoming message: \r\n" + encodedPayload, e);
		}
	}

	protected IPayload buildDto(ADT_A01 adt_a01) throws MessagingException, HL7Exception {
		PID pidSegment =  adt_a01.getPID();
		PD1 pd1Segment =  adt_a01.getPD1();
		EVN evnSegment =  adt_a01.getEVN();
		
		Patient patient = new Patient();
		
		populateUser(pidSegment, patient);
		populateLocation(evnSegment, patient);
		patient.setCmorInstitutionId(getLong(pd1Segment.getPatientPrimaryFacility(0).getIDNumber()));		
		return patient;
	}

	/**
	 * Populate Patient User from PID segment and Extended Patient Name fields.
	 * @param pidSegment
	 * @param patient
	 * @throws MessagingException
	 * @throws HL7Exception
	 */
	private void populateUser(PID pidSegment, Patient patient) throws MessagingException {

        //Set first name, last name, middle initial
		XPN[] xpn = pidSegment.getPatientName();
		patient.setLastName(xpn[0].getFamilyName().getSurname().getValue());
		patient.setFirstName(xpn[0].getGivenName().getValue());
		patient.setMiddleInitial(xpn[0].getSecondAndFurtherGivenNamesOrInitialsThereof().getValue());

		//Set SSN
        CX[] patientIdList = pidSegment.getPatientIdentifierList();
        for (int i=0; i<patientIdList.length; i++)
        {
            CX patientId =  patientIdList[i];
            String idType = patientId.getIdentifierTypeCode().getValue();
            if (idType.equals(HL7v24Helper.pidType[HL7v24Helper.ID_SSN])) {	
        		patient.setSocialSecurityNumber(patientId.getID().getValue());
        		break;
            }
        } 
        
		//Set Date of Birth
		patient.setDateOfBirth(stringToDate(pidSegment.getDateTimeOfBirth().getValue()));
	}

	/**
	 * Populate Patient Treatment Location from EVN segment. 
	 * @param evnSegment
	 * @param patient
	 */
	private void populateLocation(EVN evnSegment, Patient patient) {
		Institution institution = new Institution();
		Set institutions = new HashSet();
		
		institution.setId(evnSegment.getEventFacility().getNamespaceID().getValue());
		institution.setName(evnSegment.getEventFacility().getUniversalID().getValue());
		institutions.add(institution);
		patient.setInstitutions(institutions);
	}
}
