package gov.va.med.domain.service.messaging;

import java.io.Serializable;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Holds a description of the message in the form of 
 * the function  and the MessageExchangePatternType which indicates  
 * whether the message is intended to be Synchronous or Asynchronous.
 * <P>
 * Currently the MessageExchangePatternType is not used but still required.   
 * The only stated intent is to allow the application to specifiy this trigger 
 * a JMS correlation id so the application can block and wait for the processing 
 * of the JMS messages. 
 * <P>
 * This design creates undesirable coupling between the application and the 
 * Messaging Framework's design to use internal JMS queues.  It also confuses 
 * the mechanics of what drives the framework:  the configuration database or
 * hard-coded application directives like the MessageExchangePatternType. 
 * <P>
 * For example, it is possible to make an asynchronous call by using a function
 * and destination that is configured to use an asynchronous transceiver while
 * specifying a SYNCHRONOUS_MESSAGE_EXCHANGE_PATTERN.  
 *      
 * @author Odysseas Pentakalos
 * @version $Id: MessageType.java,v 1.4 2005/08/27 19:19:20 alex.kalinovsky Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public class MessageType implements Serializable
{
 
    private static final long serialVersionUID = 3905809694621904948L;
    private String remoteFunctionName;
    private MessageExchangePatternType mepType;

    public MessageType(String remoteFunctionName, MessageExchangePatternType mepType) {
        this.remoteFunctionName = remoteFunctionName;
        this.mepType = mepType;
    }
    
    public boolean isSynchronous() {
        return mepType.equals(MessageExchangePatternType.SYNCHRONOUS_MESSAGE_EXCHANGE_PATTERN);

    }
    
    public boolean isAsynchronous() {
        return mepType.equals(MessageExchangePatternType.ASYNCHRONOUS_MESSAGE_EXCHANGE_PATTERN);
    }
    
    public void setMessageExchangePattern(MessageExchangePatternType type) {
    	this.mepType = type;
    }
    
    public MessageExchangePatternType getMessageExchangePattern() {
    	return this.mepType;
    }
    
    public void setRemoteFunctionName(String remoteFunctionName) {
        this.remoteFunctionName = remoteFunctionName;
    }

    public String getRemoteFunctionName() {
        return remoteFunctionName;
    }

   public String toString() {
      return new ToStringBuilder(this)
      .append("remoteFunctionName", getRemoteFunctionName())
      .append("exchangePattern", getMessageExchangePattern())
      .toString();
   }
}
