package gov.va.med.domain.service.messaging;

import gov.va.med.domain.model.IdentityPayload;
import gov.va.med.domain.model.MessagingRequest;
//import gov.va.med.domain.persistence.orm.dao.MessagingRequestDAO;
import gov.va.med.domain.service.messaging.environment.Destination;
import gov.va.med.domain.service.messaging.environment.NamingDirectory;
import gov.va.med.domain.service.messaging.transceiver.TransceiverFacade;
import gov.va.med.domain.service.messaging.transceiver.TransceiverFactory;
//import gov.va.med.framework.util.ObjectFactory;



import java.util.Date;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * Facade class to build a Transceiver, encode, send and decode the response as appropriate. The
 * Destination is obtained to determine the particulars of the Transciever construction. <p/> NOTE:
 * If either Encoding or Decoding must be supressed, use IndentityEncoder or IndentityEncoder
 * respectively. <p/> NOTE: Decoding may not be necessary for some type of protocols and
 * transceivers.
 * <P>
 *
 * @author Slava Uchitel
 * @version $Id: MessageSender.java,v 1.44 2005/09/20 13:26:55 slava.uchitel Exp $
 * @since MHV 2.0 <br>
 *        03/18/2005
 */
@Component
public class MessageSender implements IMessageSender {

   private static Logger logger = LogManager.getLogger(MessageSender.class);

   private MessagingRequest messagingRequest = null;
   //TODO: come back and visit this section - didn't find table MESSAGING_REQUESTS
   //private MessagingRequestDAO messagingRequestDao = null;
   private DestinedRequest request = null;

   @Autowired
   private TransceiverFactory transceiverFactory;

   @Autowired
   private NamingDirectory namingDirectory;

   protected void setMessageId(String messageControlId) {
	if (messagingRequest == null)
		return;

	    messagingRequest.setMessageId(messageControlId);
   }

   protected void setRequestStatus(String status) {
		if (messagingRequest == null)
			return;

		if (status != null && !status.equalsIgnoreCase(messagingRequest.getMessagingRequestStatus()) )
		{
			messagingRequest.setMessagingRequestStatus(status);
			if (logger.isInfoEnabled())
				logger.info("Message Request Status for Id [" + messagingRequest.getId() + "] changing from [" + messagingRequest.getMessagingRequestStatus() + "] to [" + status + "]");
		}
    }

   /**
    * Sends the messages by using a TransceiverFacade with encode/decode functionality to decorate
    * the low-level Tansceiver associated with the endPoint. Normally, the decoder for this function
    * is an IdentityDecoder because the messages are assumed to be asynchrounous, but this is not
    * required. If it is a Synchronous message, the response will be returned after being decoded
    * per the function's decoder.
    * <P>
    * Preconditions: If message is not application ack and has a MessagingRequestId, the
    * MessagingRequest must exist and have its key on the DestintedRequest.
    * <P>
    * Updates the status of the MessagingRequest.
    * <P>
    *
    * @param request containing the payload, functionName and desinationKey.
    * @return the Response resulting from the send and optional decode or an invalid Response
    *         (!isOk() ) containing the exception that stopped processing.
    * @throws MessagingException
    */
   public Response sendMessage(DestinedRequest request) throws MessagingException {
      logger.debug("Entered sendMessage()");
      Response response = null;
      this.request = request;
      //messagingRequest = null;
      if (request.isTrackedRequest()) {
          readMessagingRequest();
      }
      // FUTURE - design changes?
      //
      // Should we have the reponse handled right here and update the
      // database using handlers and let the client query the tables after
      // some wait time.?
      // Or should we use another approach that is is to put the response
      // on the same inbound queue as asynch and let the InHl7MessageProcessor
      // handle it?

      try {
		 logger.debug("Transceive request Function Name::"+request.getFunctionName());
		 logger.debug("Transceive request Id::"+request.getRequestId());
         response = transceive(request);
      }
      catch (MessagingException e) {
         if (request.getMessageType().isSynchronous()) {
            response = new ErrorResponse(request.getMessageType(), request.getPayload(), e);
            response.setStatusDescription("Transmission of Message failed");
            logger.warn("Transmission of synchronous message failed", e);
            return response;
         }
         throw e;
      }
      logger.debug("Transceived request");
      if (request.isTrackedRequest()) {
         if (response.getPayload() instanceof IdentityPayload) {
            // This expects that the Transceiver is returing the messageControlId in the response
            // which is specific to Delivery Services which overrides this value in its envelope.
            String messageControlId = ((IdentityPayload) response.getPayload()).getPayloadAsString();
            setMessageId(messageControlId);
         }
         setRequestStatus(MessagingRequest.STATUS_TRANSMITTED);
      }
      logger.debug("About to update request status - normal");
      updateRequestStatus(null);
      logger.debug("Updated request status - normal");
      return response;
   }

   /**
    * Same as SendMessage but wraps exception in ErrorResponse.
    *
    * @param request
    */
   public Response sendSynchronousMessage(DestinedRequest request) {
      try {
         return sendMessage(request);
      } catch (Exception e) {
         // Retransmission policy was pushed up into MDB.
        Response response = new ErrorResponse(request.getMessageType(), request.getPayload(), e);
        response.setStatusDescription("Transmission of message failed");
        logger.warn("Synchronous transmission of message failed for request: " + request, e);
        return response;
      }
   }

   /**
    * Performs only the transceive with no coupling to the MessagingRequest tables. <BR>
    * The messages are sent using a TransceiverFacade with encode/decode functionality to decorate
    * the low-level Tansceiver associated with the endPoint. Normally, the decoder for this function
    * is an IdentityDecoder because the messages are assumed to be asynchrounous, but this is not
    * required. If it is a Synchronous message, the response will be returned after being decoded
    * per the function's decoder.
    * <P>
    *
    * @param request contianing the payload, functionName and desinationKey.
    * @return the Response resulting from the send and optional decode
    * @throws MessagingException
    * @throws MessagingException if any problem occurs in the process
    */

   public Response transceive(DestinedRequest request) throws MessagingException {
	   // TODO Defensive code to protect against Destination not cofigured or
	   // Function not mapped to Destination.
	   logger.debug("Getting the destination");
	   Destination destination = namingDirectory.getDestination(request.getDestinationKey());
	   if (destination == null) {
		   logger.error(String.format("Attempt to send to unconfigured Destination [%s] for function [%s]",
				   request.getDestinationKey().getStationNumber(), request.getFunctionName()));
		   /*MessagingLogHelper.logMessagingError(logger, "Attempt to send to unconfigured Destination ["
				   + request.getDestinationKey().getStationNumber() + "] for function ["
				   + request.getFunctionName() + "]", null, null);*/
		   return null;
	   }

	   TransceiverFacade transceiverFacade = transceiverFactory.constructTransceiverFacade(request, destination);

	   logger.debug("About to call transceiverFacade.transceive()");
	   Response result = transceiverFacade.transceive();
	   logger.debug("Finished calling transceiverFacade.transceive()");

	   return result;
   }

   /**
    * Must be called from outside of MessageSender's transaction
    * Changes to the request within MessageSender's sendMessage transaction are automatically
    * detected and persisted by Hibernate
    */
   protected void updateRequestStatus(String status) {
       if (messagingRequest == null)
           return;
   	   setRequestStatus(status);
       messagingRequest.setModifiedDate(new Date());
       //TODO: come back and visit this section
       //getMessagingRequestDao().update(messagingRequest);
   }

   /**
    * @return Returns the messagingRequest.
    */
   public MessagingRequest readMessagingRequest() throws MessagingException {
   	 if (messagingRequest == null) {
   	    logger.debug("Retrieving messaging request with id[" + request.getRequestId() + "]");
   	    //TODO: come back and visit this section
   	 	//messagingRequest = getMessagingRequestDao().getById(request.getRequestId());
   	 	/*try
		{
   	 		getMessagingRequestDao().initialize(messagingRequest);
		}
   	 	catch(Throwable t)
		{
   	 		messagingRequest = null;
   	 		String message = "sendMessagefailed: No MessagingRequest found for DestinedRequest: [" + request + "]";
   	 		logger.error(message);
   	 		throw new MessagingException(message);
		}*/
     }
   	 logger.debug("Retrieved messaging request with id[" + messagingRequest.getId() + "]");
   	 return messagingRequest;

   }

   /**
    * @return Returns the dao.
    */
   /*protected MessagingRequestDAO getMessagingRequestDao() {
	   if (messagingRequestDao == null) {
		   messagingRequestDao = (MessagingRequestDAO) ObjectFactory.getBean(MessagingConstants.MESSAGING_REQUEST_DAO_BEAN_NAME);
	   }
	   return messagingRequestDao;
   }*/

   /**
    * @return Returns the request.
    */
   public DestinedRequest getRequest() {
      return request;
   }

   /* (non-Javadoc)
    * @see gov.va.med.domain.service.messaging.IMessageSender#setRequestStatusFailed(java.lang.String)
    */
   public void setRequestStatusFailed() {
   	   updateRequestStatus(MessagingRequest.STATUS_FAILED_TRANSMISSION);
   }

}
