package gov.va.med.domain.model.validation;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import gov.va.med.domain.model.DtoListPayload;
import gov.va.med.framework.bean.AbstractVO;

/**
 * Superclass for Raw Value Objects that can validate and build a target DTO.
 * Subclasses are written to populate a DtoListPayload with either 
 * a valid DTO or an errant DTO with associated errors in the buildAndValidate method.   
 * <P>
 * These objects generally contain untyped values against which business rules
 * are applied including but not limited to:
 * <UL>
 * <LI>type/conversion checks
 * <LI>required field checks
 * <LI>referantial field checks
 * <LI>value in range or enumerated list checks
 * <LI>etc
 * </UL>
 * 
 * @author Joel Goldberg
 * @version $Id: BaseRawVO.java,v 1.5 2005/09/16 21:43:08 ye.jin Exp $
 * @since MHV 2.0 <br>
 *        Jul 2, 2005
 */
public abstract class BaseRawVO extends AbstractVO {

    List errors;

    /**
     * Subclasses do not need to override this mehtod. 
     * It is a template method for the main processing for all subclasses.
     * <P>
     * It provides consistent handling of the DtoListPayload and calls out to <BR>
     * finer-grained methods to create, populate and validate the DTO. 
     * The result of this method is:
     * <OL>
     * <LI>A new DTO is bulit
     * <LI>If the new DTO has validation errors it is added to the
     * DtoListPayload's errors with the DTO as the key and the resulting error
     * list as the value. false is then returned
     * <LI>If the DTO is valid, it is added to the to the DtoListPayload's dtos
     * list true is then returned
     * </OL>
     * 
     * @param dtoList
     *            the list to be populated with either a valid or errant dto
     * @return true is valid,  false if an error was detected
     */
    public boolean buildAndValidate(DtoListPayload dtoList) {
        
        Object dto = createDto();
        populateAndValidate(dto);
        
        if (!hasErrors()) {
            dtoList.addDto(dto);
            return true;
        }
        
        dtoList.addErrorList(dto, getErrors());
        setErrors(new ArrayList());
        return false;
    }

    /**
     * Creates the instance of the target DTO to be populated
     * 
     * @return a new instance of the target of this RawVO
     */
    public abstract Object createDto();

    /**
     * Checks variables in this RawVO and moves, converts them as dictated by
     * the business rules to the target Dto passed in.
     * <P>
     * If any of the fields violate business rules, error objects of type
     * ValidationError are added to the the errors variable. 
     * <P>
     * Helper conversions and validation methods are provided for common
     * conversions and validations.  
     * 
     * @param dto the target object to populate
     */
    public abstract void populateAndValidate(Object dto);

    /**
     * Returns a Date from the value passed if valid.  
     * If invalid, adds Invalid date error and returns null.
     * 
     * @param value to be validated and converted
     * @param fieldName name to use if an error occurs
     * @return converted Date
     */
    public Date validateDate(String value, String fieldName) {
        try {
            return stringToDate(value); 
        } 
        catch (ParseException e) {
            addError(ValidationError.INVALID_DATE_ERROR, fieldName, value);
            return null;
        }
    }
    /**
     * Returns an Integer from the value passed if valid.  
     * If invalid, adds invalid number error and returns null.
     * 
     * @param value to be validated and converted
     * @param fieldName name to use if an error occurs
     * @return converted Integer
     */
    public Integer validateInteger(String value, String fieldName){
        try {
            return stringToInteger(value); 
        } 
        catch (Exception e) {
            addError(ValidationError.INVALID_NUMBER_ERROR, fieldName, value);
            return null;
        }
    }
    
    /**
     * Returns an Float from the value passed if valid.  
     * If invalid, adds invalid number error and returns null.
     * 
     * @param value to be validated and converted
     * @param fieldName name to use if an error occurs
     * @return converted Integer
     */
    public Float validateFloat(String value, String fieldName){
        try {
        	if (isNull(value)) {
                return null;
            }
            return new Float(value);
        	
            
        } 
        catch (Exception e) {
            addError(ValidationError.INVALID_NUMBER_ERROR, fieldName, value);
            return null;
        }
    }
    /**
     * Same as validateInteger but first checks to see if not null or "empty".
     * If null, adds missing field error and returns null.
     * 
     * @param value to be validated and converted
     * @param fieldName name to use if an error occurs
     * @return converted Integer
     */
    public Integer validateNonNullInteger(String value, String fieldName) {
        if (validateNotNull(value, fieldName) == null) {
            return null;
        }	
        return validateInteger(value, fieldName);
    }
    
    
    /**
     * Same as validateInteger but first checks to see if not null or "empty".
     * If null, adds missing field error and returns null.
     * 
     * @param value to be validated and converted
     * @param fieldName name to use if an error occurs
     * @return converted Integer
     */
    public Float validateNonNullFloat(String value, String fieldName) {
        if (validateNotNull(value, fieldName) == null) {
            return null;
        }	
        return validateFloat(value, fieldName);
    }
    /**
     * Same as validateDate but first checks to see if not null or "empty".
     * If null, adds missing field error and returns null.
     * 
     * @param value to be validated and converted
     * @param fieldName name to use if an error occurs
     * @return converted Date
     */
    public Date validateNonNullDate(String value, String fieldName) {
        if (validateNotNull(value, fieldName) == null) {
            return null;
        }	
        return validateDate(value, fieldName);
    }

    /**
     * Checks if null or "empty" value and adds a missing field error 
     * or the value passed in if not.  
     * 
     * @param value to be validated and converted
     * @param fieldName name to use if an error occurs
     * @return converted Integer
     */
    public String validateNotNull(String value, String fieldName){
        if (isNull(value)) {
            addError(ValidationError.MISSING_FIELD_ERROR, fieldName);
		    return null;
        }
        return value;
    }
    
    /**
     * Checks for null or a logically empty string.
     * Called by all methods that validate for null. 
     * If your subclass has a diffferent rule for this check, override
     * this method.   
     */
    public boolean isNull(String value) {
        return  value==null || value.trim().length()==0 || value.trim().equals("\"\"");
    }
    
    /**
     * Converts the String to an Integer or null if the value is null or "empty". 
     * @param rawNumber the value to convert
     * @throws NumberFormatException if the raw number is not a valid integer
     */                                 
    public Integer stringToInteger(String rawNumber) {
        if (exists(rawNumber)) {
            return null;
        }
        return parseToInteger(rawNumber);
    }
    
    /**
     * Converts the String to a Date or null if the value is null or "empty". 
     * @param rawDate the value to convert
     * @throws ParseException if the raw date cannot be converted to a valid Date
     */     
    public Date stringToDate(String rawDate) throws ParseException {
        if (exists(rawDate)) {
            return null;
        }
        return parseToDate(rawDate);
    }
    
    public Integer parseToInteger(String value) {
        return new Integer(value);
    }
    
    public Date parseToDate(String value) throws ParseException {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
        return sdf.parse(value);
    }
    
    /**
     * Used to decide if field value is usable for field conversion.
     * By Default, uses isNull() but can be overridden. 
     */
    public boolean exists(String value) {
        return isNull(value);
    }

    /**
     * Convenience method to add an error to the errors list
     * @param key the error key
     * @param fieldName the errant field name
     */
    public void addError(String key, String fieldName) {
	   addError(key, fieldName, null);
    }
    
    /**
     * Convenience method to add an error to the errors list
     * @param key the error key
     * @param fieldName the errant field name
     * @param value the errant value added a entry in the error's Parameters
     */
    public void addError(String key, String fieldName, String value) {
	    ValidationError error = new ValidationError();
	    error.setFieldName(fieldName);
	    error.setErrorKey(key);
	    if (value != null) {
		    error.setParameters(new Object[] {value});
	    }
	    getErrors().add(error);
    }
    
    public boolean hasErrors() {
        return !getErrors().isEmpty();
    }
    
    public List getErrors() {
        if (errors == null) {
            errors = new ArrayList();
        }
        return errors;
    }

    public void setErrors(List errors) {
        this.errors = errors;
    }
}
