package gov.va.med.mhv.mrp.web.controller;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.ManagedProperty;
import javax.faces.bean.ViewScoped;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.va.med.mhv.mrp.common.dto.MyGoalDTO;
import gov.va.med.mhv.mrp.web.bean.UserProfile;
import gov.va.med.mhv.mrp.web.converter.MyGoalConverter;
import gov.va.med.mhv.mrp.web.converter.MyGoalDTOConverter;
import gov.va.med.mhv.mrp.web.model.MyGoal;
import gov.va.med.mhv.mrp.web.model.MyGoalTask;
import gov.va.med.mhv.mrp.web.model.MyGoalsExtension;
import gov.va.med.mhv.mrp.webservice.MyGoalsWebService;

@ManagedBean
@ViewScoped
public class TrackGoalController implements Serializable {
	private static final long serialVersionUID = 1L;

	private static Logger log = LogManager.getLogger(TrackGoalController.class);

	@ManagedProperty("#{myGoalsWebServiceProxy}")
	private MyGoalsWebService myGoalsWebServiceProxy;

	@ManagedProperty("#{myGoalConverter}")
	private MyGoalConverter myGoalConverter;

	@ManagedProperty("#{myGoalDTOConverter}")
	private MyGoalDTOConverter myGoalDTOConverter;

	@ManagedProperty("#{userProfile}")
	private UserProfile userProfile;

	private Long goalId;

	// show save confirmation (true) or the main page (false)
	private Boolean showSaveConfirmation = false;

	// calendar date of start of week (Sunday)
	private Calendar startDate;

	// calendar date of end of week (Saturday)
	private Calendar endDate;

	// calendar date of max Date for this goal (current day)
	private Calendar maxDate;

	// used by calendar control
	private Date searchDate;

	private MyGoal goal;

	private List<MyGoalsExtension> obstacles;

	// replace this with viewAction and @PostConstruct when using JSF 2.2
	public void init(ComponentSystemEvent event) {
		log.debug("init");

		FacesContext facesContext = FacesContext.getCurrentInstance();

		if (!facesContext.isPostback()) {
			log.debug("not postback");
			// one-time initialization
			postConstruct();
		} else {
			log.debug("postback");
			log.debug(this);
		}
	}

	// @PostConstruct
	private void postConstruct() {
		log.debug("postConstruct");

		log.debug("goalId = ", goalId);

		log.debug("userProfileId=" + userProfile.getUserProfileId());

		if (goalId != null && userProfile.getUserProfileId() != null) {

			MyGoalDTO dto = myGoalsWebServiceProxy.getMyGoalById(userProfile.getUserProfileId(), goalId);

			goal = myGoalConverter.convert(dto);

			setDefaultDates();

			obstacles = new ArrayList<MyGoalsExtension>(goal.getObstaclesWithUnfinishedTasks());

			setTasksCurrentWeek(startDate);
		}
	}

	public void previousWeek() {
		log.debug("previousWeek");

		startDate.add(Calendar.DAY_OF_MONTH, -7);
		endDate.add(Calendar.DAY_OF_MONTH, -7);

		setTasksCurrentWeek(startDate);
	}

	public void nextWeek() {
		log.debug("nextWeek");

		Calendar c = (Calendar) endDate.clone();
		c.add(Calendar.DAY_OF_MONTH, 7);

		// ensure not going past maxDate
		// the UI button should be disabled anyway, but this is an extra check
		if (c.compareTo(maxDate) <= 0) {
			startDate.add(Calendar.DAY_OF_MONTH, 7);
			endDate.add(Calendar.DAY_OF_MONTH, 7);
		}

		setTasksCurrentWeek(startDate);
	}

	public void search() {
		log.debug("search");

		Calendar c = Calendar.getInstance();
		c.setTime(searchDate);

		startDate = (Calendar) c.clone();
		startDate.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);

		endDate = (Calendar) c.clone();
		endDate.set(Calendar.DAY_OF_WEEK, Calendar.SATURDAY);

		setTasksCurrentWeek(startDate);

	}

	public boolean getShowPreviousWeek() {
		log.debug("getShowPreviousWeek");

		log.debug(this);

		Calendar c = Calendar.getInstance();
		c.setTime(goal.getStartDate());
		return startDate.compareTo(c) >= 0;
	}

	public boolean getShowNextWeek() {
		log.debug("getShowNextWeek");

		log.debug(this);

		return endDate.compareTo(maxDate) < 0;
	}

	private void setDefaultDates() {
		// set searchDate to today
		// startDate and endDate are beginning/end of current week
		// set max date to end of current week
		Calendar today = Calendar.getInstance();

		searchDate = today.getTime();

		startDate = (Calendar) today.clone();
		startDate.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);

		endDate = (Calendar) today.clone();
		endDate.set(Calendar.DAY_OF_WEEK, Calendar.SATURDAY);

		maxDate = (Calendar) endDate.clone();
	}

	private void setTasksCurrentWeek(Calendar beginDate) {
		for (MyGoalsExtension e : obstacles) {
			for (MyGoalTask t : e.getTasks()) {
				t.fillTrackTaskActivities(beginDate);
			}
		}
	}

	public Boolean getAllTasksCompleted() {
		Boolean allMarkCompleted = true;
		
		log.debug("getAllTasksCompleted");
		
		if (obstacles == null) {
			log.debug("empty obstacles");
			return false;
		}
		
		for (MyGoalsExtension e : obstacles) {
			if (e != null) {
				for (MyGoalTask t : e.getTasks()) {
					if (t == null) {
						continue;
					} 

					log.debug("taskName=" + t.getTaskName() + ", marked=" + t.getMarkCompleted() + ", completed=" + t.getCompleted());

					if (t.getCompleted()) {
						continue;
					}
					
					if (t.getMarkCompleted() == null || !t.getMarkCompleted()) {
						allMarkCompleted = false;
					}
				}
			}
		}

		return allMarkCompleted;
	}

	/**
	 * confirmation to mark task(s) as completed
	 */
	public void confirmSave(Boolean confirm) {
		if (confirm) {
			// user confirmed
			for (MyGoalsExtension e : obstacles) {
				for (MyGoalTask t : e.getTasks()) {
					if (t.getCompleted()) {
						continue;
					}
					if (t.getMarkCompleted()) {
						String info = "<b>Congratulations!</b> You have successfully finished the Task <b>'" + t.getTaskName() + "'</b>";
						FacesContext.getCurrentInstance().addMessage("", new FacesMessage(FacesMessage.SEVERITY_INFO, info, info));
						
						t.setCompleted(true);
						log.debug("setting taskName=" + t.getTaskName() + " completed on " + (t.getCompletedDate() != null ? t.getCompletedDate().getTime() : "null"));
					}
					
				}
			}

			saveGoal();
			
			if (log.isDebugEnabled()) {
				for (MyGoalsExtension e2 : obstacles) {
					for (MyGoalTask t2 : e2.getTasks()) {
						log.debug("setting taskName=" + t2.getTaskName() + " completed on " + (t2.getCompletedDate() != null ? t2.getCompletedDate().getTime() : "null"));
					}
				}

			}

		} else {
			// user declined, so unmark all tasks 
			for (MyGoalsExtension e : obstacles) {
				for (MyGoalTask t : e.getTasks()) {
					if (t.getCompleted()) {
						continue;
					}
					t.setMarkCompleted(false);
				}
			}			
		}

		// always show main page after save confirmation dialog
		showSaveConfirmation = false; 
	}
	
	public String markCompleted() {

		log.debug("markCompleted");
		
		try {
			MyGoalDTO myGoalDTO = myGoalDTOConverter.convert(goal);

			Calendar completedDate = Calendar.getInstance();
			myGoalDTO.setCompletedDate(completedDate);

			log.debug("saving");

			myGoalsWebServiceProxy.saveMyGoal(userProfile.getUserProfileId(), myGoalDTO);

		} catch (Exception e) {
			log.debug(e);
		}

		return "/views/completedgoals/summary";
	}

	public void save() {
		boolean anyMarkCompleted = false;
		
		log.debug("save");
		
		for (MyGoalsExtension e : obstacles) {
			for (MyGoalTask t : e.getTasks()) {
				log.debug("taskName=" + t.getTaskName() + ", marked=" + t.getMarkCompleted() + ", completed=" + t.getCompleted());
				if (!t.getCompleted() && t.getMarkCompleted()) {
					anyMarkCompleted = true;
				}
			}
		}

		if (anyMarkCompleted) {
			log.debug("show confirmation");
			log.debug(this);
			showSaveConfirmation = true;
			return;
		}

		// saveGoal();
	}

	private boolean saveGoal() {
		Boolean success = false;

		log.debug("saveGoal");

		try {
			log.debug("Converting to DTO");
			MyGoalDTO myGoalDTO = myGoalDTOConverter.convert(goal);

			log.debug("before save");

			MyGoalDTO resultDTO = myGoalsWebServiceProxy.saveMyGoal(myGoalDTO.getUserProfileId(), myGoalDTO);

			log.debug("after save");

			if (resultDTO != null) {

				goal = myGoalConverter.convert(resultDTO);
				
				obstacles = new ArrayList<MyGoalsExtension>(goal.getObstaclesWithUnfinishedTasks());

				log.debug(this);
				
				setTasksCurrentWeek(startDate);

				success = true;
			}

		} catch (Exception e) {
			log.error(e);
		}

		if (!success) {
			log.debug("Save failed");
			FacesContext.getCurrentInstance().addMessage("", new FacesMessage(FacesMessage.SEVERITY_ERROR, "save failed", "save failed"));
		}
		
		return success;
	}

	public List<Calendar> getTaskDaysOfWeek() {

		List<Calendar> list = new ArrayList<Calendar>();

		for (int i = 0; i <= 6; i++) {
			Calendar offset = (Calendar) getStartDate().clone();
			offset.add(Calendar.DAY_OF_MONTH, i);
			list.add(offset);
		}

		return list;
	}

	public Calendar getStartDate() {
		return startDate;
	}

	public void setStartDate(Calendar startDate) {
		this.startDate = startDate;
	}

	public Calendar getEndDate() {
		return endDate;
	}

	public void setEndDate(Calendar endDate) {
		this.endDate = endDate;
	}

	public Date getSearchDate() {
		return searchDate;
	}

	public void setSearchDate(Date searchDate) {
		this.searchDate = searchDate;
	}

	public Calendar getMaxDate() {
		return maxDate;
	}

	public void setMaxDate(Calendar maxDate) {
		this.maxDate = maxDate;
	}

	public MyGoal getGoal() {
		return goal;
	}

	public Long getGoalId() {
		return goalId;
	}

	public void setGoalId(Long goalId) {
		this.goalId = goalId;
	}

	public void setMyGoalsWebServiceProxy(MyGoalsWebService myGoalsWebServiceProxy) {
		this.myGoalsWebServiceProxy = myGoalsWebServiceProxy;
	}

	public void setMyGoalConverter(MyGoalConverter myGoalConverter) {
		this.myGoalConverter = myGoalConverter;
	}

	public void setMyGoalDTOConverter(MyGoalDTOConverter myGoalDTOConverter) {
		this.myGoalDTOConverter = myGoalDTOConverter;
	}

	public void setUserProfile(UserProfile userProfile) {
		this.userProfile = userProfile;
	}

	private String formatDate(Calendar c) {
		return (c != null ? c.getTime().toString() : "null");
	}

	public Boolean getShowSaveConfirmation() {
		return showSaveConfirmation;
	}

	public void setShowSaveConfirmation(Boolean showSaveConfirmation) {
		this.showSaveConfirmation = showSaveConfirmation;
	}

	public List<MyGoalsExtension> getObstacles() {
		return obstacles;
	}

	public void setObstacles(List<MyGoalsExtension> obstacles) {
		this.obstacles = obstacles;
	}

	@Override
	public String toString() {
		return "TrackGoalController [goalId=" + goalId + ", startDate=" + formatDate(startDate) + ", endDate=" + formatDate(endDate) + ", maxDate="
				+ formatDate(maxDate) + ", searchDate=" + searchDate + ", showSaveConfirmation=" + showSaveConfirmation + "]";
	}

}
