package gov.va.med.mhv.mrp.service.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.mrp.common.dto.MyGoalDTO;
import gov.va.med.mhv.mrp.common.dto.MyGoalTaskDTO;
import gov.va.med.mhv.mrp.common.dto.MyGoalTaskSearchDTO;
import gov.va.med.mhv.mrp.common.dto.MyGoalsExtensionDTO;
import gov.va.med.mhv.mrp.common.dto.MyTaskDTO;
import gov.va.med.mhv.mrp.common.dto.MyTaskSearchDTO;
import gov.va.med.mhv.mrp.data.model.MyGoals;
import gov.va.med.mhv.mrp.data.model.MyGoalsExtensions;
import gov.va.med.mhv.mrp.data.model.MyGoalsTasks;
import gov.va.med.mhv.mrp.data.repository.MyGoalExtensionRepository;
import gov.va.med.mhv.mrp.data.repository.MyGoalRepository;
import gov.va.med.mhv.mrp.data.repository.MyGoalTaskRepository;
import gov.va.med.mhv.mrp.service.MyGoalsService;
import gov.va.med.mhv.mrp.service.converter.MyGoalConverter;
import gov.va.med.mhv.mrp.service.converter.MyGoalDTOConverter;
import gov.va.med.mhv.mrp.service.converter.MyGoalExtensionsConverter;
import gov.va.med.mhv.mrp.service.converter.MyGoalExtensionsDTOConverter;
import gov.va.med.mhv.mrp.service.converter.MyGoalListConverter;
import gov.va.med.mhv.mrp.service.converter.MyGoalTaskConverter;
import gov.va.med.mhv.mrp.service.converter.MyGoalTaskDTOConverter;
import gov.va.med.mhv.mrp.service.converter.MyTaskConverter;

@Component
public class MyGoalsServiceImpl implements MyGoalsService {
	private static Logger log = LogManager.getLogger(MyGoalsServiceImpl.class);

	@Autowired
	private MyGoalRepository myGoalRepository;

	@Autowired
	private MyGoalExtensionRepository goalExtensionRepository;

	@Autowired
	private MyGoalTaskRepository goalTaskRepository;

	@Autowired
	private MyGoalListConverter myGoalListConverter;

	@Autowired
	private MyGoalConverter myGoalConverter;

	@Autowired
	private MyGoalDTOConverter myGoalDTOConverter;

	@Autowired
	private MyGoalTaskConverter myGoalTaskConverter;

	@Autowired
	private MyGoalTaskDTOConverter myGoalTaskDTOConverter;

	@Autowired
	private MyTaskConverter myTaskConverter;

	@Autowired
	private MyGoalExtensionsConverter myGoalExtensionsConverter;

	@Autowired
	private MyGoalExtensionsDTOConverter myGoalExtensionsDTOConverter;

	@Override
	public List<MyGoalDTO> getMyGoalsForUser(Long userProfileId) {
		List<MyGoals> mygoals = myGoalRepository.findByUserProfileId(userProfileId);
		if (mygoals != null) {
			List<MyGoalDTO> dtos = myGoalListConverter.convert(mygoals);
			return dtos;
		} else {
			log.debug("My Goal Records not found for userProfileId:" + userProfileId);
		}
		return null;
	}

	@Override
	public List<MyGoalDTO> getMyCompletedGoalsForUser(Long userProfileId) {
		List<MyGoals> mygoals = myGoalRepository.findCompletedByUserProfileId(userProfileId);
		if (mygoals != null) {
			List<MyGoalDTO> dtos = myGoalListConverter.convert(mygoals);
			return dtos;
		} else {
			log.debug("My Goal Records not found for userProfileId:" + userProfileId);
		}
		return null;
	}

	@Override
	public MyGoalDTO getMyGoalById(Long userProfileId, Long goalId) {
		log.debug("getMyGoalById, userProfileId=" + userProfileId + ", goalId=" + goalId);

		MyGoals mygoal = myGoalRepository.findMyGoalById(userProfileId, goalId);
		if (mygoal != null) {
			MyGoalDTO myGoalDTO = myGoalConverter.convert(mygoal);
			if (log.isDebugEnabled()) {
				log.debug(myGoalDTO);
			}
			return myGoalDTO;
		} else {
			log.debug("My Goal record not found for id:" + goalId);
		}
		return null;
	}

	@Override
	public MyGoalDTO getMyGoalByName(Long userProfileId, String goalName) {
		log.debug("getMyGoalByName, userProfileId=" + userProfileId + ", goalName=" + goalName);

		MyGoals mygoal = myGoalRepository.findMyGoalByName(userProfileId, goalName);
		if (mygoal != null) {
			MyGoalDTO myGoalDTO = myGoalConverter.convert(mygoal);
			if (log.isDebugEnabled()) {
				log.debug(myGoalDTO);
			}
			return myGoalDTO;
		} else {
			log.debug("My Goal record not found for name:" + goalName);
		}
		return null;
	}

	@Override
	public MyGoalDTO saveMyGoal(MyGoalDTO myGoalDTO) {
		log.debug("saveMyGoal");

		if (myGoalDTO == null) {
			log.warn("Missing paramaeters");
			return null;
		}

		try {

			if (myGoalDTO.getGoalId() == null) {
				// creating new goal
				log.debug("Creating new goal, goalName=" + myGoalDTO.getGoalName());
				myGoalDTO.setOplock(0);
			} else {
				// updating existing goal
				log.debug("Updating goal, goalId=" + myGoalDTO.getGoalId());

				MyGoals search = myGoalRepository.findOne(myGoalDTO.getGoalId());

				if (search == null) {
					log.warn("Could not find existing Goal with goalId=" + myGoalDTO.getGoalId());
					return null;
				}
			}

			log.debug("Converting to JPA entity");
			MyGoals myGoals = myGoalDTOConverter.convert(myGoalDTO);

			log.debug("Saving");
			MyGoals result = myGoalRepository.saveAndFlush(myGoals);
			log.debug("after save");

			if (result != null) {
				log.debug("converting to DTO");
				MyGoalDTO resultDTO = myGoalConverter.convert(result);
				if (log.isDebugEnabled()) {
					log.debug(resultDTO);
				}
				return resultDTO;
			} else {
				log.error("save failed");
			}

		} catch (Exception e) {
			log.error(e);
		}

		return null;
	}

	@Override
	public Boolean deleteMyGoal(Long userProfileId, Long goalId) {
		Boolean success = false;
		
		log.debug("deleteMyGoal");

		if (userProfileId == null || goalId == null) {
			log.warn("missing parameters");
			return false;
		}
		
		try {
			log.debug("deleting MyGoal userProfileId=" + userProfileId + ", goalId=" + goalId);

			MyGoals result = myGoalRepository.findMyGoalById(userProfileId, goalId);
			
			if (result == null) {
				log.debug("Could not find goalId=" + goalId);
			} else {

				log.debug("before deleting goalId=" + goalId);
				myGoalRepository.delete(result);
				log.debug("after deleting goalId=" + goalId);

				success = true;
			}

		} catch (Exception e) {
			log.error(e);
		}

		return success;
	}

	@Override
	public List<MyTaskDTO> getMyGoalTasksForUser(MyGoalTaskSearchDTO myGoalTaskSearchDTO) {
		log.debug(myGoalTaskSearchDTO);
		try {
			List<MyGoalsTasks> myGoalsTasksList = goalTaskRepository.getMyGoalTasksForUser(myGoalTaskSearchDTO.getUserProfileId());
			
			if (myGoalsTasksList != null) {
				String goalName = "";
				List<MyTaskDTO> dtos = new ArrayList<MyTaskDTO>();
				MyTaskDTO dto = null;
				for (MyGoalsTasks task : myGoalsTasksList) {
					goalName = task.getMyGoalsExtension().getMyGoal().getGoalName();
					dto = new MyTaskDTO();
					dto = myTaskConverter.convert(task);
					dto.setGoalName(goalName);
					dtos.add(dto);
				}

				return dtos;
			} else {
				log.debug("My Goals Task Records not found for userProfileId:" + myGoalTaskSearchDTO.getUserProfileId());
			}
		} catch (Exception e) {
			log.error(e);
		}

		return null;
	}

	@Override
	public MyGoalsExtensionDTO getMyGoalsExtension(Long goalExtensionId) {

		log.debug("Searching for MyGoalExtension id = " + goalExtensionId);

		try {
			MyGoalsExtensions result = goalExtensionRepository.findOne(goalExtensionId);

			if (result != null) {
				MyGoalsExtensionDTO dto = myGoalExtensionsConverter.convert(result);
				return dto;
			} else {
				log.warn("MyGoalExtension id = " + goalExtensionId + " not found.");
			}
		} catch (Exception e) {
			log.error(e);
		}

		return null;
	}

	@Override
	public MyGoalsExtensionDTO addMyGoalsExtension(Long goalId, MyGoalsExtensionDTO myGoalsExtensionDTO) {

		log.debug("addMyGoalsExtension, goalId=" + goalId, ", goalExtensionId" + myGoalsExtensionDTO.getGoalExtensionId());

		if (goalId == null || myGoalsExtensionDTO == null) {
			log.warn("missing parameters");
			return null;
		}

		try {
			if (myGoalsExtensionDTO.getGoalExtensionId() != null) {
				// update extension
				log.debug("Updating extension");
				MyGoalsExtensions result = goalExtensionRepository.findOne(myGoalsExtensionDTO.getGoalExtensionId());

				if (result == null) {
					log.warn("goalExtensionId=" + myGoalsExtensionDTO.getGoalExtensionId() + " not found.");
					return null;
				}

				if (log.isDebugEnabled()) {
					log.debug("current extension in database");
					for (MyGoalsTasks t : result.getMyGoalsTasks()) {
						log.debug("taskId=" + t.getTaskId() + ", taskName=" + t.getTaskName() + ", taskSeqNo=" + t.getTaskSequenceNumber());
					}

					log.debug("From inner MyGoal");
					for (MyGoalsExtensions ext : result.getMyGoal().getMyGoalsExtensions()) {
						if (ext.getGoalExtensionId().equals(myGoalsExtensionDTO.getGoalExtensionId())) {
							for (MyGoalsTasks t : ext.getMyGoalsTasks()) {
								log.debug("taskId=" + t.getTaskId() + ", taskName=" + t.getTaskName() + ", taskSeqNo=" + t.getTaskSequenceNumber());
							}
						}
					}

				}

				log.debug("converting to JPA entity");

				MyGoalsExtensions mge = myGoalExtensionsDTOConverter.convert(myGoalsExtensionDTO);
				mge.setMyGoal(result.getMyGoal());
				
				if (log.isDebugEnabled()) {
					log.debug(mge);
					for (MyGoalsTasks t : mge.getMyGoalsTasks()) {
						log.debug(t);
					}
				}

				log.debug("Saving extension");
				MyGoalsExtensions save = goalExtensionRepository.saveAndFlush(mge);
				log.debug("Done saving extension");

				if (log.isDebugEnabled()) {
					log.debug("after save from return");
					for (MyGoalsTasks t : save.getMyGoalsTasks()) {
						log.debug("taskId=" + t.getTaskId() + ", taskName=" + t.getTaskName() + ", taskSeqNo=" + t.getTaskSequenceNumber());
					}

					result = goalExtensionRepository.findOne(myGoalsExtensionDTO.getGoalExtensionId());

					log.debug("from query");
					for (MyGoalsTasks t : result.getMyGoalsTasks()) {
						log.debug("taskId=" + t.getTaskId() + ", taskName=" + t.getTaskName() + ", taskSeqNo=" + t.getTaskSequenceNumber());
					}
				}

				log.debug("Converting to DTO");
				MyGoalsExtensionDTO dto = myGoalExtensionsConverter.convert(save);

				if (log.isDebugEnabled()) {
					log.debug("after conversion");
					for (MyGoalTaskDTO t : dto.getMyGoalsTasks()) {
						log.debug("taskId=" + t.getTaskId() + ", taskName=" + t.getTaskName() + ", taskSeqNo=" + t.getTaskSequenceNumber());
					}
				}

				return dto;

			} else {
				// new extension
				log.debug("Creating new extension");
				MyGoals myGoals = myGoalRepository.findOne(goalId);

				MyGoalsExtensions mge = new MyGoalsExtensions();

				mge.setDescription(myGoalsExtensionDTO.getDescription());
				mge.setGoalExtensionType(myGoalsExtensionDTO.getGoalExtensionType());
				mge.setOplock(0);

				Integer seqNo = 0;
				if (myGoals.getMyGoalsExtensions() != null) {
					for (MyGoalsExtensions ext : myGoals.getMyGoalsExtensions()) {
						seqNo = Math.max(seqNo, ext.getGoalExtensionSequenceNumber());
					}

				}
				mge.setGoalExtensionSequenceNumber(++seqNo);

				log.debug("SequenceNo=" + seqNo);

				myGoals.addMyGoalsExtension(mge);

				log.debug("Saving extension");
				MyGoals result = myGoalRepository.saveAndFlush(myGoals);
				log.debug("return from save");

				if (result.getMyGoalsExtensions() != null) {
					for (MyGoalsExtensions ext : result.getMyGoalsExtensions()) {
						if (ext.getGoalExtensionSequenceNumber().equals(seqNo)) {

							MyGoalsExtensionDTO dto = myGoalExtensionsConverter.convert(ext);

							log.debug(dto);

							return dto;
						}
					}

				}

				log.warn("Could not find newly added extension");
			}

		} catch (Exception e) {
			log.error(e);
		}

		return null;
	}

	@Override
	public Boolean deleteMyGoalsExtension(Long goalExtensionId) {
		Boolean success = false;

		log.debug("Deleting goalExtensionId=" + goalExtensionId);

		if (goalExtensionId == null) {
			log.warn("missing parameters");
			return false;
		}

		try {

			MyGoalsExtensions mge = goalExtensionRepository.findOne(goalExtensionId);

			if (mge != null) {
				log.debug("found MyGoalsExtension " + mge.getGoalExtensionId());
				log.debug("Executing JPA delete goalExtensionId=" + goalExtensionId);
				goalExtensionRepository.delete(mge);
				log.debug("Completed JPA delete goalExtensionId=" + goalExtensionId);
				success = true;
				
				if (log.isDebugEnabled()) {
					MyGoalsExtensions result = goalExtensionRepository.findOne(goalExtensionId);
					if (result != null) {
						log.debug("after delete, still found goalExtensionId= " + result.getGoalExtensionId());
					} else {
						log.debug("confirmed delete, goalExtensionId= " + goalExtensionId);
					}
				}
			} else {
				log.warn("MyGoalsExtension with goalExtensionId=" + goalExtensionId + " not found.");
			}

		} catch (Exception e) {
			log.error(e);
		}

		return success;
	}

	@Override
	public MyGoalTaskDTO addMyGoalTask(Long goalExtensionId, MyGoalTaskDTO myGoalTaskDTO) {

		log.debug("Saving MyGoalTask, goalExtensionId=" + goalExtensionId);

		if (goalExtensionId == null || myGoalTaskDTO == null) {
			log.warn("missing parameters");
			return null;
		}

		try {
			if (myGoalTaskDTO.getTaskId() != null) {
				// update task
				log.debug("Updating Task");

				MyGoalsTasks result = goalTaskRepository.findOne(myGoalTaskDTO.getTaskId());

				if (result == null) {
					log.warn("TaskId=" + myGoalTaskDTO.getTaskId() + " not found.");
					return null;
				}

				MyGoalsTasks tasks = myGoalTaskDTOConverter.convert(myGoalTaskDTO);
				tasks.setMyGoalsExtension(result.getMyGoalsExtension());
				
				if (!result.getMyGoalsExtension().getGoalExtensionId().equals(myGoalTaskDTO.getGoalExtensionId())) {
					MyGoalsExtensions ext = this.goalExtensionRepository.findOne(myGoalTaskDTO.getGoalExtensionId());
					result.setMyGoalsExtension(ext);
				}

				log.debug("before save");
				log.debug(myGoalTaskDTO);

				MyGoalsTasks save = goalTaskRepository.saveAndFlush(result);

				MyGoalTaskDTO dto = myGoalTaskConverter.convert(save);

				log.debug("after save");
				log.debug(myGoalTaskDTO);

				return dto;

			} else {
				// new task
				log.debug("Creating new Task");

				MyGoalsExtensions myGoalsExtensions = goalExtensionRepository.findOne(goalExtensionId);

				if (myGoalsExtensions == null) {
					log.warn("Goal Extension with id=" + goalExtensionId + " not found.");
					return null;
				}

				MyGoalsTasks myGoalTasks = myGoalTaskDTOConverter.convert(myGoalTaskDTO);

				myGoalTasks.setMyGoalsExtension(myGoalsExtensions);
				myGoalTasks.setOplock(0);

				// set the sequence number to next value
				Integer taskSequenceNumber = 0;

				for (MyGoalsTasks mgt : myGoalsExtensions.getMyGoalsTasks()) {
					taskSequenceNumber = Math.max(mgt.getTaskSequenceNumber(), taskSequenceNumber);
				}
				myGoalTasks.setTaskSequenceNumber(++taskSequenceNumber);

				MyGoalsTasks save = goalTaskRepository.saveAndFlush(myGoalTasks);

				MyGoalTaskDTO dto = myGoalTaskConverter.convert(save);

				return dto;
			}

		} catch (Exception e) {
			log.error(e);
		}

		return null;
	}

	@Override
	public Boolean deleteMyGoalTask(Long taskId) {
		Boolean success = false;
		
		log.debug("deleteMyGoalTask taskId=" + taskId);

		if (taskId == null) {
			log.warn("missing parameters");
			return false;
		}

		try {
			log.debug("before MyGoalsTasks taskId=" + taskId);

			goalTaskRepository.delete(taskId);

			log.debug("Done deleting MyGoalsTasks taskId=" + taskId);

			success = true;

		} catch (Exception e) {
			log.error(e);
		}

		return success;
	}

	@Override
	public List<MyTaskDTO> getMyGoalTasksByKeyword(MyGoalTaskSearchDTO myGoalTaskSearchDTO) {
		log.debug(myGoalTaskSearchDTO);
		String keyword = "%" + myGoalTaskSearchDTO.getKeyword().toUpperCase() + "%";
		try {
			List<MyGoalsTasks> myGoalsTasksList = goalTaskRepository.getMyGoalTasksBykeyword(myGoalTaskSearchDTO.getUserProfileId(), keyword);
			if (myGoalsTasksList != null) {
				String goalName = "";
				List<MyTaskDTO> dtos = new ArrayList<MyTaskDTO>();
				MyTaskDTO dto = null;
				for (MyGoalsTasks task : myGoalsTasksList) {
					goalName = task.getMyGoalsExtension().getMyGoal().getGoalName();
					dto = new MyTaskDTO();
					dto = myTaskConverter.convert(task);
					dto.setGoalName(goalName);
					dtos.add(dto);
				}

				return dtos;
			} else {
				log.debug("My Goals Task Records not found for userProfileId:" + myGoalTaskSearchDTO.getUserProfileId());
			}
		} catch (Exception e) {
			log.error(e);
		}
		return null;
	}

	@Override
	public MyTaskSearchDTO findMyGoalForTask(Long userProfileId, MyTaskSearchDTO myTaskSearchDTO) {
		MyTaskSearchDTO dto = null;
		
		log.debug("findMyGoalForTask");
		
		if (myTaskSearchDTO == null) {
			log.debug("invalid parameters");
			return null;
		}
		
		try {
			
			log.debug("userProfileId=" + userProfileId);
			log.debug(myTaskSearchDTO);
			
			List<MyGoalsTasks> list = goalTaskRepository.getMyGoalTasksForName(userProfileId, myTaskSearchDTO.getTaskName(), myTaskSearchDTO.getGoalName());
			
			if (list == null || list.size() != 1) {
				if (log.isDebugEnabled()) {
					if (list == null) {
						log.debug("no results");
					} else {
						log.debug("list size=" + list.size());
					}
				}
				return null;
			}
			
			MyGoalsTasks task = list.get(0);
			MyGoalsExtensions ext = task.getMyGoalsExtension();

			MyTaskSearchDTO result = new MyTaskSearchDTO();
			result.setUserProfileId(myTaskSearchDTO.getUserProfileId());
			result.setGoalName(myTaskSearchDTO.getGoalName());
			result.setTaskName(myTaskSearchDTO.getTaskName());
			result.setGoalId(ext.getMyGoal().getGoalId());
			result.setTaskId(task.getTaskId());
			
			log.debug(result);
			
			dto = result;
			
		} catch (Exception e) {
			log.debug(e);
		}
		
		return dto;
	}

}
