package gov.va.med.mhv.journal.service.validator;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.journal.dto.MealItemDTO;
import gov.va.med.mhv.journal.enums.MealTypeEnumeration;
import gov.va.med.mhv.journal.web.util.JournalMessagesUtil;

import java.util.Date;
import java.util.LinkedHashMap;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class MealItemValidator {

	public static final int ITEM_MAX_LENGTH = 50;

	@Autowired
	private JournalMessagesUtil messagesUtil;
	
	/**
	 * Validates createdDate
	 */
	protected void validateCreatedDate(MealItemDTO mealItemDto, LinkedHashMap<String, String> validationErrors) {
		Date value = mealItemDto.getCreatedDate();
		//check requiredness:		
		if( value == null ) {
			Validations.validateRequired("CreatedDate", null, validationErrors);
						
			//no need to continue:
			return;
		}		
	}	
	
	
	/**
	 * Validates Food/Beverage Item
	 */
	protected void validateItem(MealItemDTO mealItemDto, LinkedHashMap<String, String> validationErrors) {
		String value = StringUtils.trimToNull(mealItemDto.getItem());
		//check requiredness:		
		if( value == null ) {
			Validations.validateRequired("Food/Beverage Item", value, validationErrors);
						
			//no need to continue:
			return;
		}		
		//check max length
		Validations.validateMaxLength("Food/Beverage Item", value, ITEM_MAX_LENGTH, validationErrors);
	}	
	
	
	/**
	 * Validates modifiedDate
	 */
	protected void validateModifiedDate(MealItemDTO mealItemDto, LinkedHashMap<String, String> validationErrors) {
		Date value = mealItemDto.getModifiedDate();
		//check for a value to validate:
		if ( value == null ) {
			//nothing to validate
			return;
		}						
	}	
	
	
	/**
	 * Validates Meal Type
	 */
	protected void validateMealType(MealItemDTO mealItemDto, LinkedHashMap<String, String> validationErrors) {
		String value = StringUtils.trimToNull(mealItemDto.getMealType());
		//check requiredness:
		if( value == null ) {
			Validations.validateRequired("Meal Type", value, validationErrors);
						
			//no need to continue:
			return;
		}		
		
		//validate a valid enumeration value:
		boolean valid = false;
		if (Validations.hasValue(value)) {
			for (MealTypeEnumeration s : MealTypeEnumeration.values()) {
				if (s.getMealType().equals(value)) {
					valid = true;
					break;
				}
			}
			
			if (!valid) {
				validationErrors.put(messagesUtil.getMealTypeRequired(), messagesUtil.getMealTypeRequired());
			}
		}
	
	}
	
	
	/**
	 * Executes all field-level validations
	 */
	public ResponseUtil<MealItemDTO> fieldValidation(MealItemDTO mealItemDto) {
		ResponseUtil response = new ResponseUtil();
		LinkedHashMap<String, String> validationErrors = new LinkedHashMap<String, String>();
		
		validateMealType(mealItemDto, validationErrors);
		validateItem(mealItemDto, validationErrors);
		validateCreatedDate(mealItemDto, validationErrors);
		validateModifiedDate(mealItemDto, validationErrors);
		
		response.setValidationErrors(validationErrors);
		
		return response;
	}
	
	public boolean isMealItemNull(MealItemDTO mealItemDto) {
		boolean mealItemNull = false;
		
		if (null != StringUtils.trimToNull(mealItemDto.getMealType()) 
			&& null != mealItemDto.getQuantity()
			&& null != StringUtils.trimToNull(mealItemDto.getServingSize())
			&& null != StringUtils.trimToNull(mealItemDto.getItem())
			&& null != StringUtils.trimToNull(mealItemDto.getPrepMethod())) {
			mealItemNull = true;
		}
		
		return mealItemNull;
	}
}