package gov.va.med.mhv.journal.service.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.journal.converter.ActivityDetailConverter;
import gov.va.med.mhv.journal.converter.ActivityJournalConverter;
import gov.va.med.mhv.journal.data.model.ActivityDetail;
import gov.va.med.mhv.journal.data.model.ActivityJournal;
import gov.va.med.mhv.journal.dto.ActivityDetailDTO;
import gov.va.med.mhv.journal.dto.ActivityJournalDTO;
import gov.va.med.mhv.journal.repository.ActivityDetailRepository;
import gov.va.med.mhv.journal.repository.ActivityJournalRepository;
import gov.va.med.mhv.journal.service.ActivityJournalService;
import gov.va.med.mhv.journal.service.validator.ActivityJournalValidator;

@Component
public class ActivityJournalServiceImpl implements ActivityJournalService {
	private static Logger log = LogManager.getLogger(ActivityJournalServiceImpl.class);

	@Autowired
	private ActivityJournalRepository activityJournalRepository;

	@Autowired
	private ActivityDetailRepository activityDetailRepository;

	@Autowired
	private ActivityJournalConverter activityJournalConverter;

	@Autowired
	private ActivityDetailConverter activityDetailConverter;

	@Autowired
	private ActivityJournalValidator activityJournalValidator;

	@Override
	@Transactional
	public List<ActivityJournalDTO> findActivityJournalsForUser(Long userProfileId) throws MHVException {
		List<ActivityJournal> list = null;
		List<ActivityJournalDTO> dtoList = null;

		try {
			list = activityJournalRepository.findActivityJournalsForUser(userProfileId);

			if (list != null) {
				dtoList = new ArrayList<ActivityJournalDTO>();
				for (ActivityJournal activityJournal : list) {
					dtoList.add(activityJournalConverter.convertDomainActivityJournal(activityJournal));
				}
			}
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in fetching activityJournals " + e.getMessage(), e);
		}

		return dtoList;
	}

	@Override
	@Transactional
	public ActivityJournalDTO findActivityJournalById(Long activityJournalId) throws MHVException {
		ActivityJournal activityJournal = null;
		ActivityJournalDTO activityJournalDTO = null;

		try {
			activityJournal = activityJournalRepository.findOne(activityJournalId);

			if (activityJournal != null) {
				activityJournalDTO = activityJournalConverter.convertDomainActivityJournal(activityJournal);
			}
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in fetching activityJournal " + e.getMessage(), e);
		}

		return activityJournalDTO;
	}

	@Override
	@Transactional
	public ActivityDetailDTO addActivityDetails(ActivityDetailDTO activityDetailDTO) throws MHVException {
		ActivityDetail activityDetail = null;
		ActivityJournal activityJournal = null;
		ActivityDetailDTO dto = null;

		try {
			activityDetail = activityDetailConverter.convertDTODetailItem(activityDetailDTO);
			activityJournal = activityJournalRepository.findOne(activityDetail.getActivityJournalId());
			activityDetail.setActivityJournal(activityJournal);
			activityDetail = activityDetailRepository.save(activityDetail);

			dto = activityDetailConverter.convertDomainDetailItem(activityDetail);

		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in storing activity detail in DB " + e.getMessage(), e);
		}

		return dto;
	}

	@Override
	@Transactional
	public List<ActivityJournalDTO> getActivityJournalsForDashBoard(Long userProfileId) throws MHVException {
		List<ActivityJournal> list = null;
		List<ActivityJournalDTO> dtoList = null;

		try {
			Pageable topFive = new PageRequest(0, 5);
			list = activityJournalRepository.getDashBoardActivityJournalsForUser(userProfileId, topFive);

			if (list != null) {
				dtoList = new ArrayList<ActivityJournalDTO>();
				for (ActivityJournal activityJournal : list) {
					dtoList.add(activityJournalConverter.convertDomainActivityJournal(activityJournal));
				}
			}

		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in fetching top5 activityJournals " + e.getMessage(), e);
		}

		return dtoList;
	}

	@Override
	@Transactional
	public ActivityJournalDTO addActivityJournal(ActivityJournalDTO activityJournalDTO) throws MHVException {
		ResponseUtil<ActivityJournalDTO> response = new ResponseUtil<ActivityJournalDTO>();
		ActivityJournal activityJournal = null;
		ActivityJournalDTO dto = null;

		response = activityJournalValidator.activityJournalValidations(activityJournalDTO);

		// input validation errors
		if (response.getValidationErrors().size() > 0) {
			throw new MHVException(response);
		}

		try {
			activityJournal = activityJournalConverter.convertDTOActivityJournal(activityJournalDTO);

			if (activityJournal.getActivityDetails() != null) {
				for (ActivityDetail activityDetail : activityJournal.getActivityDetails()) {
					activityDetail.setActivityJournal(activityJournal);
				}
			}

			activityJournal = activityJournalRepository.save(activityJournal);

			// remove activityJournal object from activity detail to avoid
			// circular references
			if (null != activityJournal.getActivityDetails()) {
				Long activityJounralId = (null != activityJournal.getActivityJournalId()) ? activityJournal.getActivityJournalId() : null;
				for (ActivityDetail activityDetail : activityJournal.getActivityDetails()) {
					activityDetail.setActivityJournalId(activityJounralId);
				}
			}

			dto = activityJournalConverter.convertDomainActivityJournal(activityJournal);

		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in storing meal Item in DB " + e.getMessage(), e);
		}

		return dto;
	}

	@Override
	@Transactional
	public void deleteActivityJournal(Long activityJournalId) throws MHVException {

		try {
			activityJournalRepository.delete(activityJournalId);
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in deleting activityJournal " + e.getMessage(), e);
		}
	}

	@Override
	@Transactional
	public void deleteActivityDetail(Long activityDetailId) throws MHVException {
		ActivityDetail activityDetail = null;
		ActivityJournal activityJournal = null;
		Long activityJournalId = null;

		try {

			activityDetail = activityDetailRepository.findOne(activityDetailId);
			activityJournalId = activityDetail.getActivityJournalId();
			activityJournal = activityJournalRepository.findOne(activityJournalId);
			activityJournal.getActivityDetails().remove(activityDetail);
			activityJournalRepository.save(activityJournal);

		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in deleting activity detail " + e.getMessage(), e);
		}

	}

	@Override
	@Transactional
	public ActivityDetailDTO getActivityDetailById(Long activityDetailById) throws MHVException {
		ActivityDetail activityDetail = null;
		ActivityDetailDTO activityDetailDTO = null;

		try {
			activityDetail = activityDetailRepository.findOne(activityDetailById);
			
			if (activityDetail != null) {
				activityDetailDTO = activityDetailConverter.convertDomainDetailItem(activityDetail);
			}
			
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in fetching activityDetail " + e.getMessage(), e);
		}

		return activityDetailDTO;
	}

}
